// Copyright (c) 2021 Emmanuel Gil Peyrot <linkmauve@linkmauve.fr>
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

#![deny(missing_docs)]
#![allow(non_camel_case_types)]

//! Generates a shared object suitable for use with gdk-pixbuf, adding support for the HVIF format.

use bitflags::bitflags;
use cairo::{Format, ImageSurface};
use core::ffi::c_void;
use core::ptr::{null, null_mut};
use hvif::Hvif;
use std::os::raw::c_char;

type GdkPixbuf = c_void;
type GdkPixbufAnimation = c_void;
type GModule = c_void;
type GError = c_void;
type gpointer = *mut c_void;
type gchar = c_char;
type guchar = u8;

/// The gobject version of bool.
#[repr(C)]
pub struct gboolean(i32);

impl gboolean {
    const FALSE: gboolean = gboolean(0);
    const TRUE: gboolean = gboolean(1);
}

#[repr(C)]
struct GdkPixbufModulePattern {
    prefix: *const c_char,
    mask: *const c_char,
    relevance: i32,
}

bitflags! {
    /// Flags which allow a module to specify further details about the supported operations.
    pub struct GdkPixbufFormatFlags: u32 {
        /// The module can write out images in the format.
        const WRITABLE = 0b001;

        /// The image format is scalable.
        const SCALABLE = 0b010;

        /// The module is threadsafe. gdk-pixbuf ignores modules that are not marked as threadsafe.
        const THREADSAFE = 0b100;
    }
}

/// A `GdkPixbufFormat` contains information about the image format accepted by a module.
#[repr(C)]
pub struct GdkPixbufFormat {
    name: *const gchar,
    signature: *const GdkPixbufModulePattern,
    domain: *mut gchar,
    description: *const gchar,
    mime_types: *const *const gchar,
    extensions: *const *const gchar,
    flags: GdkPixbufFormatFlags,
    disabled: gboolean,
    license: *const gchar,
}

type GdkPixbufModuleSizeFunc =
    Option<extern "C" fn(width: *mut i32, height: *mut i32, user_data: gpointer)>;
type GdkPixbufModulePreparedFunc = Option<
    extern "C" fn(pixbuf: *mut GdkPixbuf, anim: *mut GdkPixbufAnimation, user_data: gpointer),
>;
type GdkPixbufModuleUpdatedFunc = Option<
    extern "C" fn(
        pixbuf: *mut GdkPixbuf,
        x: i32,
        y: i32,
        width: i32,
        height: i32,
        user_data: gpointer,
    ),
>;

/// A module.
#[repr(C)]
pub struct GdkPixbufModule {
    module_name: *const c_char,
    module_path: *const c_char,
    module: *const GModule,
    info: *const GdkPixbufFormat,
    load: *const c_void,
    load_xpm_data: *const c_void,
    begin_load: Option<
        extern "C" fn(
            GdkPixbufModuleSizeFunc,
            GdkPixbufModulePreparedFunc,
            GdkPixbufModuleUpdatedFunc,
            gpointer,
            *mut *mut GError,
        ) -> gpointer,
    >,
    stop_load: Option<extern "C" fn(gpointer, *mut *mut GError) -> gboolean>,
    load_increment:
        Option<extern "C" fn(gpointer, *const guchar, u32, *mut *mut GError) -> gboolean>,
    load_animation: *const c_void,
    save: *const c_void,
    save_to_callback: *const c_void,
    is_save_option_supported: *const c_void,
    reserved1: *const c_void,
    reserved2: *const c_void,
    reserved3: *const c_void,
    reserved4: *const c_void,
}

struct Decoder {
    data: Vec<u8>,
    size_func: GdkPixbufModuleSizeFunc,
    prepared_func: GdkPixbufModulePreparedFunc,
    user_data: gpointer,
}

impl Decoder {
    fn new(
        size_func: GdkPixbufModuleSizeFunc,
        prepared_func: GdkPixbufModulePreparedFunc,
        user_data: gpointer,
    ) -> Decoder {
        Decoder {
            data: Vec::with_capacity(1024),
            size_func,
            prepared_func,
            user_data,
        }
    }
}

extern "C" fn begin_load(
    size_func: GdkPixbufModuleSizeFunc,
    prepared_func: GdkPixbufModulePreparedFunc,
    _updated_func: GdkPixbufModuleUpdatedFunc,
    user_data: gpointer,
    _error: *mut *mut GError,
) -> gpointer {
    let decoder = Decoder::new(size_func, prepared_func, user_data);
    let boxed = Box::new(decoder);
    Box::into_raw(boxed) as *mut c_void
}

#[repr(u32)]
enum GdkColorspace {
    Rgb = 0,
}

#[link(name = "gdk_pixbuf-2.0")]
extern "C" {
    fn gdk_pixbuf_new(
        colorspace: GdkColorspace,
        has_alpha: gboolean,
        bits_per_sample: i32,
        width: i32,
        height: i32,
    ) -> *mut GdkPixbuf;
    fn gdk_pixbuf_get_pixels_with_length(pixbuf: *const GdkPixbuf, length: *mut u32)
        -> *mut guchar;
    fn gdk_pixbuf_get_rowstride(pixbuf: *const GdkPixbuf) -> i32;
}

struct Pixbuf {
    inner: *mut GdkPixbuf,
}

impl Pixbuf {
    fn new(has_alpha: bool, bits_per_sample: i32, width: i32, height: i32) -> Pixbuf {
        let has_alpha = if has_alpha {
            gboolean::TRUE
        } else {
            gboolean::FALSE
        };
        let inner = unsafe {
            gdk_pixbuf_new(
                GdkColorspace::Rgb,
                has_alpha,
                bits_per_sample,
                width,
                height,
            )
        };
        Pixbuf { inner }
    }

    fn get_pixels_mut(&self) -> &mut [u8] {
        let mut length = core::mem::MaybeUninit::uninit();
        unsafe {
            let pixels = gdk_pixbuf_get_pixels_with_length(self.inner, length.as_mut_ptr());
            let length = length.assume_init();
            std::slice::from_raw_parts_mut(pixels, length as usize)
        }
    }

    fn get_rowstride(&self) -> i32 {
        unsafe { gdk_pixbuf_get_rowstride(self.inner) }
    }
}

// TODO: optimise that a bit.
fn bgra_to_rgba(data: &mut [u8]) {
    for chunk in data.chunks_exact_mut(4) {
        let b = chunk[0];
        let r = chunk[2];
        chunk[0] = r;
        chunk[2] = b;
    }
}

extern "C" fn stop_load(context: gpointer, _error: *mut *mut GError) -> gboolean {
    let decoder = unsafe { Box::from_raw(context as *mut Decoder) };
    if let Ok((i, hvif)) = Hvif::parse(&decoder.data) {
        if !i.is_empty() {
            return gboolean::FALSE;
        }

        // Query the size wanted by the caller.
        let mut width = 64;
        let mut height = 64;
        if let Some(size_func) = decoder.size_func {
            size_func(&mut width, &mut height, decoder.user_data);
        }
        if width == 0 || height == 0 {
            return gboolean::FALSE;
        }

        let pixbuf = Pixbuf::new(true, 8, width, height);
        let stride = pixbuf.get_rowstride();

        unsafe {
            let pixels = pixbuf.get_pixels_mut();
            pixels.fill(0);
            let mut surface =
                ImageSurface::create_for_data_unsafe(pixels.as_mut_ptr(), Format::ARgb32, width, height, stride)
                    .unwrap();
            hvif::render(hvif, &mut surface).unwrap();
        }

        // Cairo and GDK-Pixbuf don’t support the same colour format, so we have to implement the
        // conversion manually…
        bgra_to_rgba(pixbuf.get_pixels_mut());

        // Everything went fine, now we can signal the pixbuf is ready!
        if let Some(prepared_func) = decoder.prepared_func {
            prepared_func(pixbuf.inner, null_mut(), decoder.user_data);
        }
        gboolean::TRUE
    } else {
        gboolean::FALSE
    }
}

extern "C" fn load_increment(
    context: gpointer,
    buf: *const guchar,
    size: u32,
    _error: *mut *mut GError,
) -> gboolean {
    let mut decoder = unsafe { Box::from_raw(context as *mut Decoder) };
    let slice = unsafe { std::slice::from_raw_parts(buf, size as usize) };
    decoder.data.extend(slice);
    Box::into_raw(decoder);
    gboolean::TRUE
}

/// Sets the entrypoints of the module’s vtable.
#[no_mangle]
pub unsafe extern "C" fn fill_vtable(module: *mut GdkPixbufModule) {
    let mut module = Box::from_raw(module);

    module.begin_load = Some(begin_load);
    module.stop_load = Some(stop_load);
    module.load_increment = Some(load_increment);

    Box::into_raw(module);
}

const fn c_str(content: &[u8]) -> *const gchar {
    content.as_ptr() as *const gchar
}

/// Fills the metadata of this module.
#[no_mangle]
pub unsafe extern "C" fn fill_info(info: *mut GdkPixbufFormat) {
    const SIGNATURE: [GdkPixbufModulePattern; 2] = [
        GdkPixbufModulePattern {
            prefix: c_str(b"ncif\0"),
            mask: c_str(b"    \0"),
            relevance: 100,
        },
        GdkPixbufModulePattern {
            prefix: null(),
            mask: null(),
            relevance: 0,
        },
    ];
    const MIME_TYPES: [*const gchar; 2] = [c_str(b"image/x-hvif\0"), null()];
    const EXTENSIONS: [*const gchar; 2] = [c_str(b"hvif\0"), null()];

    let mut info = Box::from_raw(info);

    info.name = c_str(b"hvif\0");
    info.signature = SIGNATURE.as_ptr();
    info.description = c_str(b"Haiku Vector Icon Format\0");
    info.mime_types = MIME_TYPES.as_ptr();
    info.extensions = EXTENSIONS.as_ptr();
    info.flags = GdkPixbufFormatFlags::THREADSAFE;
    info.license = c_str(b"BSD\0");
    info.disabled = gboolean::FALSE;

    Box::into_raw(info);
}
