/// Cross-platform signal numbers defined by the GDB Remote Serial Protocol.
///
/// Transcribed from <https://github.com/bminor/binutils-gdb/blob/master/include/gdb/signals.def>
#[repr(u8)]
#[derive(Debug, Copy, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
#[allow(clippy::upper_case_acronyms)]
#[allow(non_camel_case_types)]
#[rustfmt::skip]
#[non_exhaustive]
pub enum Signal {
    #[doc = "Signal 0 (shouldn't be used)"]    SIGZERO = 0,
    #[doc = "Hangup"]                          SIGHUP = 1,
    #[doc = "Interrupt"]                       SIGINT = 2,
    #[doc = "Quit"]                            SIGQUIT = 3,
    #[doc = "Illegal instruction"]             SIGILL = 4,
    #[doc = "Trace/breakpoint trap"]           SIGTRAP = 5,
    #[doc = "Aborted"]                         SIGABRT = 6,
    #[doc = "Emulation trap"]                  SIGEMT = 7,
    #[doc = "Arithmetic exception"]            SIGFPE = 8,
    #[doc = "Killed"]                          SIGKILL = 9,
    #[doc = "Bus error"]                       SIGBUS = 10,
    #[doc = "Segmentation fault"]              SIGSEGV = 11,
    #[doc = "Bad system call"]                 SIGSYS = 12,
    #[doc = "Broken pipe"]                     SIGPIPE = 13,
    #[doc = "Alarm clock"]                     SIGALRM = 14,
    #[doc = "Terminated"]                      SIGTERM = 15,
    #[doc = "Urgent I/O condition"]            SIGURG = 16,
    #[doc = "Stopped (signal)"]                SIGSTOP = 17,
    #[doc = "Stopped (user)"]                  SIGTSTP = 18,
    #[doc = "Continued"]                       SIGCONT = 19,
    #[doc = "Child status changed"]            SIGCHLD = 20,
    #[doc = "Stopped (tty input)"]             SIGTTIN = 21,
    #[doc = "Stopped (tty output)"]            SIGTTOU = 22,
    #[doc = "I/O possible"]                    SIGIO = 23,
    #[doc = "CPU time limit exceeded"]         SIGXCPU = 24,
    #[doc = "File size limit exceeded"]        SIGXFSZ = 25,
    #[doc = "Virtual timer expired"]           SIGVTALRM = 26,
    #[doc = "Profiling timer expired"]         SIGPROF = 27,
    #[doc = "Window size changed"]             SIGWINCH = 28,
    #[doc = "Resource lost"]                   SIGLOST = 29,
    #[doc = "User defined signal 1"]           SIGUSR1 = 30,
    #[doc = "User defined signal 2"]           SIGUSR2 = 31,
    #[doc = "Power fail/restart"]              SIGPWR = 32,
    /* Similar to SIGIO.  Perhaps they should have the same number. */
    #[doc = "Pollable event occurred"]         SIGPOLL = 33,
    #[doc = "SIGWIND"]                         SIGWIND = 34,
    #[doc = "SIGPHONE"]                        SIGPHONE = 35,
    #[doc = "Process's LWPs are blocked"]      SIGWAITING = 36,
    #[doc = "Signal LWP"]                      SIGLWP = 37,
    #[doc = "Swap space dangerously low"]      SIGDANGER = 38,
    #[doc = "Monitor mode granted"]            SIGGRANT = 39,
    #[doc = "Need to relinquish monitor mode"] SIGRETRACT = 40,
    #[doc = "Monitor mode data available"]     SIGMSG = 41,
    #[doc = "Sound completed"]                 SIGSOUND = 42,
    #[doc = "Secure attention"]                SIGSAK = 43,
    #[doc = "SIGPRIO"]                         SIGPRIO = 44,
    #[doc = "Real-time event 33"]              SIG33 = 45,
    #[doc = "Real-time event 34"]              SIG34 = 46,
    #[doc = "Real-time event 35"]              SIG35 = 47,
    #[doc = "Real-time event 36"]              SIG36 = 48,
    #[doc = "Real-time event 37"]              SIG37 = 49,
    #[doc = "Real-time event 38"]              SIG38 = 50,
    #[doc = "Real-time event 39"]              SIG39 = 51,
    #[doc = "Real-time event 40"]              SIG40 = 52,
    #[doc = "Real-time event 41"]              SIG41 = 53,
    #[doc = "Real-time event 42"]              SIG42 = 54,
    #[doc = "Real-time event 43"]              SIG43 = 55,
    #[doc = "Real-time event 44"]              SIG44 = 56,
    #[doc = "Real-time event 45"]              SIG45 = 57,
    #[doc = "Real-time event 46"]              SIG46 = 58,
    #[doc = "Real-time event 47"]              SIG47 = 59,
    #[doc = "Real-time event 48"]              SIG48 = 60,
    #[doc = "Real-time event 49"]              SIG49 = 61,
    #[doc = "Real-time event 50"]              SIG50 = 62,
    #[doc = "Real-time event 51"]              SIG51 = 63,
    #[doc = "Real-time event 52"]              SIG52 = 64,
    #[doc = "Real-time event 53"]              SIG53 = 65,
    #[doc = "Real-time event 54"]              SIG54 = 66,
    #[doc = "Real-time event 55"]              SIG55 = 67,
    #[doc = "Real-time event 56"]              SIG56 = 68,
    #[doc = "Real-time event 57"]              SIG57 = 69,
    #[doc = "Real-time event 58"]              SIG58 = 70,
    #[doc = "Real-time event 59"]              SIG59 = 71,
    #[doc = "Real-time event 60"]              SIG60 = 72,
    #[doc = "Real-time event 61"]              SIG61 = 73,
    #[doc = "Real-time event 62"]              SIG62 = 74,
    #[doc = "Real-time event 63"]              SIG63 = 75,
    /* Used internally by Solaris threads.  See signal(5) on Solaris. */
    #[doc = "LWP internal signal"]             SIGCANCEL = 76,
    /* Yes, this pains me, too.  But LynxOS didn't have SIG32, and now
    GNU/Linux does, and we can't disturb the numbering, since it's
    part of the remote protocol.  Note that in some GDB's
    GDB_SIGNAL_REALTIME_32 is number 76.  */
    #[doc = "Real-time event 32"]              SIG32 = 77,
    /* Yet another pain, IRIX 6 has SIG64. */
    #[doc = "Real-time event 64"]              SIG64 = 78,
    /* Yet another pain, GNU/Linux MIPS might go up to 128. */
    #[doc = "Real-time event 65"]              SIG65 = 79,
    #[doc = "Real-time event 66"]              SIG66 = 80,
    #[doc = "Real-time event 67"]              SIG67 = 81,
    #[doc = "Real-time event 68"]              SIG68 = 82,
    #[doc = "Real-time event 69"]              SIG69 = 83,
    #[doc = "Real-time event 70"]              SIG70 = 84,
    #[doc = "Real-time event 71"]              SIG71 = 85,
    #[doc = "Real-time event 72"]              SIG72 = 86,
    #[doc = "Real-time event 73"]              SIG73 = 87,
    #[doc = "Real-time event 74"]              SIG74 = 88,
    #[doc = "Real-time event 75"]              SIG75 = 89,
    #[doc = "Real-time event 76"]              SIG76 = 90,
    #[doc = "Real-time event 77"]              SIG77 = 91,
    #[doc = "Real-time event 78"]              SIG78 = 92,
    #[doc = "Real-time event 79"]              SIG79 = 93,
    #[doc = "Real-time event 80"]              SIG80 = 94,
    #[doc = "Real-time event 81"]              SIG81 = 95,
    #[doc = "Real-time event 82"]              SIG82 = 96,
    #[doc = "Real-time event 83"]              SIG83 = 97,
    #[doc = "Real-time event 84"]              SIG84 = 98,
    #[doc = "Real-time event 85"]              SIG85 = 99,
    #[doc = "Real-time event 86"]              SIG86 = 100,
    #[doc = "Real-time event 87"]              SIG87 = 101,
    #[doc = "Real-time event 88"]              SIG88 = 102,
    #[doc = "Real-time event 89"]              SIG89 = 103,
    #[doc = "Real-time event 90"]              SIG90 = 104,
    #[doc = "Real-time event 91"]              SIG91 = 105,
    #[doc = "Real-time event 92"]              SIG92 = 106,
    #[doc = "Real-time event 93"]              SIG93 = 107,
    #[doc = "Real-time event 94"]              SIG94 = 108,
    #[doc = "Real-time event 95"]              SIG95 = 109,
    #[doc = "Real-time event 96"]              SIG96 = 110,
    #[doc = "Real-time event 97"]              SIG97 = 111,
    #[doc = "Real-time event 98"]              SIG98 = 112,
    #[doc = "Real-time event 99"]              SIG99 = 113,
    #[doc = "Real-time event 100"]             SIG100 = 114,
    #[doc = "Real-time event 101"]             SIG101 = 115,
    #[doc = "Real-time event 102"]             SIG102 = 116,
    #[doc = "Real-time event 103"]             SIG103 = 117,
    #[doc = "Real-time event 104"]             SIG104 = 118,
    #[doc = "Real-time event 105"]             SIG105 = 119,
    #[doc = "Real-time event 106"]             SIG106 = 120,
    #[doc = "Real-time event 107"]             SIG107 = 121,
    #[doc = "Real-time event 108"]             SIG108 = 122,
    #[doc = "Real-time event 109"]             SIG109 = 123,
    #[doc = "Real-time event 110"]             SIG110 = 124,
    #[doc = "Real-time event 111"]             SIG111 = 125,
    #[doc = "Real-time event 112"]             SIG112 = 126,
    #[doc = "Real-time event 113"]             SIG113 = 127,
    #[doc = "Real-time event 114"]             SIG114 = 128,
    #[doc = "Real-time event 115"]             SIG115 = 129,
    #[doc = "Real-time event 116"]             SIG116 = 130,
    #[doc = "Real-time event 117"]             SIG117 = 131,
    #[doc = "Real-time event 118"]             SIG118 = 132,
    #[doc = "Real-time event 119"]             SIG119 = 133,
    #[doc = "Real-time event 120"]             SIG120 = 134,
    #[doc = "Real-time event 121"]             SIG121 = 135,
    #[doc = "Real-time event 122"]             SIG122 = 136,
    #[doc = "Real-time event 123"]             SIG123 = 137,
    #[doc = "Real-time event 124"]             SIG124 = 138,
    #[doc = "Real-time event 125"]             SIG125 = 139,
    #[doc = "Real-time event 126"]             SIG126 = 140,
    #[doc = "Real-time event 127"]             SIG127 = 141,

    #[doc = "Information request"]             SIGINFO = 142,

    /* Some signal we don't know about. */
    #[doc = "Unknown signal"]                  UNKNOWN = 143,

    /* Use whatever signal we use when one is not specifically specified
    (for passing to proceed and so on).  */
    #[doc = "Internal error: printing GDB_SIGNAL_DEFAULT"] INTERNAL_DEFAULT = 144,

    /* Mach exceptions.  In versions of GDB before 5.2, these were just before
    GDB_SIGNAL_INFO if you were compiling on a Mach host (and missing
    otherwise).  */
    #[doc = "Could not access memory"]         EXC_BAD_ACCESS = 145,
    #[doc = "Illegal instruction/operand"]     EXC_BAD_INSTRUCTION = 146,
    #[doc = "Arithmetic exception"]            EXC_ARITHMETIC = 147,
    #[doc = "Emulation instruction"]           EXC_EMULATION = 148,
    #[doc = "Software generated exception"]    EXC_SOFTWARE = 149,
    #[doc = "Breakpoint"]                      EXC_BREAKPOINT = 150,

    #[doc = "librt internal signal"]           SIGLIBRT = 151,
}

impl core::fmt::Display for Signal {
    fn fmt(&self, f: &mut core::fmt::Formatter<'_>) -> core::fmt::Result {
        #[rustfmt::skip]
        let s = match self {
            Signal::SIGZERO => "SIGZERO - Signal 0",
            Signal::SIGHUP => "SIGHUP - Hangup",
            Signal::SIGINT => "SIGINT - Interrupt",
            Signal::SIGQUIT => "SIGQUIT - Quit",
            Signal::SIGILL => "SIGILL - Illegal instruction",
            Signal::SIGTRAP => "SIGTRAP - Trace/breakpoint trap",
            Signal::SIGABRT => "SIGABRT - Aborted",
            Signal::SIGEMT => "SIGEMT - Emulation trap",
            Signal::SIGFPE => "SIGFPE - Arithmetic exception",
            Signal::SIGKILL => "SIGKILL - Killed",
            Signal::SIGBUS => "SIGBUS - Bus error",
            Signal::SIGSEGV => "SIGSEGV - Segmentation fault",
            Signal::SIGSYS => "SIGSYS - Bad system call",
            Signal::SIGPIPE => "SIGPIPE - Broken pipe",
            Signal::SIGALRM => "SIGALRM - Alarm clock",
            Signal::SIGTERM => "SIGTERM - Terminated",
            Signal::SIGURG => "SIGURG - Urgent I/O condition",
            Signal::SIGSTOP => "SIGSTOP - Stopped (signal)",
            Signal::SIGTSTP => "SIGTSTP - Stopped (user)",
            Signal::SIGCONT => "SIGCONT - Continued",
            Signal::SIGCHLD => "SIGCHLD - Child status changed",
            Signal::SIGTTIN => "SIGTTIN - Stopped (tty input)",
            Signal::SIGTTOU => "SIGTTOU - Stopped (tty output)",
            Signal::SIGIO => "SIGIO - I/O possible",
            Signal::SIGXCPU => "SIGXCPU - CPU time limit exceeded",
            Signal::SIGXFSZ => "SIGXFSZ - File size limit exceeded",
            Signal::SIGVTALRM => "SIGVTALRM - Virtual timer expired",
            Signal::SIGPROF => "SIGPROF - Profiling timer expired",
            Signal::SIGWINCH => "SIGWINCH - Window size changed",
            Signal::SIGLOST => "SIGLOST - Resource lost",
            Signal::SIGUSR1 => "SIGUSR1 - User defined signal 1",
            Signal::SIGUSR2 => "SIGUSR2 - User defined signal 2",
            Signal::SIGPWR => "SIGPWR - Power fail/restart",
            Signal::SIGPOLL => "SIGPOLL - Pollable event occurred",
            Signal::SIGWIND => "SIGWIND - SIGWIND",
            Signal::SIGPHONE => "SIGPHONE - SIGPHONE",
            Signal::SIGWAITING => "SIGWAITING - Process's LWPs are blocked",
            Signal::SIGLWP => "SIGLWP - Signal LWP",
            Signal::SIGDANGER => "SIGDANGER - Swap space dangerously low",
            Signal::SIGGRANT => "SIGGRANT - Monitor mode granted",
            Signal::SIGRETRACT => "SIGRETRACT - Need to relinquish monitor mode",
            Signal::SIGMSG => "SIGMSG - Monitor mode data available",
            Signal::SIGSOUND => "SIGSOUND - Sound completed",
            Signal::SIGSAK => "SIGSAK - Secure attention",
            Signal::SIGPRIO => "SIGPRIO - SIGPRIO",
            Signal::SIG33 => "SIG33 - Real-time event 33",
            Signal::SIG34 => "SIG34 - Real-time event 34",
            Signal::SIG35 => "SIG35 - Real-time event 35",
            Signal::SIG36 => "SIG36 - Real-time event 36",
            Signal::SIG37 => "SIG37 - Real-time event 37",
            Signal::SIG38 => "SIG38 - Real-time event 38",
            Signal::SIG39 => "SIG39 - Real-time event 39",
            Signal::SIG40 => "SIG40 - Real-time event 40",
            Signal::SIG41 => "SIG41 - Real-time event 41",
            Signal::SIG42 => "SIG42 - Real-time event 42",
            Signal::SIG43 => "SIG43 - Real-time event 43",
            Signal::SIG44 => "SIG44 - Real-time event 44",
            Signal::SIG45 => "SIG45 - Real-time event 45",
            Signal::SIG46 => "SIG46 - Real-time event 46",
            Signal::SIG47 => "SIG47 - Real-time event 47",
            Signal::SIG48 => "SIG48 - Real-time event 48",
            Signal::SIG49 => "SIG49 - Real-time event 49",
            Signal::SIG50 => "SIG50 - Real-time event 50",
            Signal::SIG51 => "SIG51 - Real-time event 51",
            Signal::SIG52 => "SIG52 - Real-time event 52",
            Signal::SIG53 => "SIG53 - Real-time event 53",
            Signal::SIG54 => "SIG54 - Real-time event 54",
            Signal::SIG55 => "SIG55 - Real-time event 55",
            Signal::SIG56 => "SIG56 - Real-time event 56",
            Signal::SIG57 => "SIG57 - Real-time event 57",
            Signal::SIG58 => "SIG58 - Real-time event 58",
            Signal::SIG59 => "SIG59 - Real-time event 59",
            Signal::SIG60 => "SIG60 - Real-time event 60",
            Signal::SIG61 => "SIG61 - Real-time event 61",
            Signal::SIG62 => "SIG62 - Real-time event 62",
            Signal::SIG63 => "SIG63 - Real-time event 63",
            Signal::SIGCANCEL => "SIGCANCEL - LWP internal signal",
            Signal::SIG32 => "SIG32 - Real-time event 32",
            Signal::SIG64 => "SIG64 - Real-time event 64",
            Signal::SIG65 => "SIG65 - Real-time event 65",
            Signal::SIG66 => "SIG66 - Real-time event 66",
            Signal::SIG67 => "SIG67 - Real-time event 67",
            Signal::SIG68 => "SIG68 - Real-time event 68",
            Signal::SIG69 => "SIG69 - Real-time event 69",
            Signal::SIG70 => "SIG70 - Real-time event 70",
            Signal::SIG71 => "SIG71 - Real-time event 71",
            Signal::SIG72 => "SIG72 - Real-time event 72",
            Signal::SIG73 => "SIG73 - Real-time event 73",
            Signal::SIG74 => "SIG74 - Real-time event 74",
            Signal::SIG75 => "SIG75 - Real-time event 75",
            Signal::SIG76 => "SIG76 - Real-time event 76",
            Signal::SIG77 => "SIG77 - Real-time event 77",
            Signal::SIG78 => "SIG78 - Real-time event 78",
            Signal::SIG79 => "SIG79 - Real-time event 79",
            Signal::SIG80 => "SIG80 - Real-time event 80",
            Signal::SIG81 => "SIG81 - Real-time event 81",
            Signal::SIG82 => "SIG82 - Real-time event 82",
            Signal::SIG83 => "SIG83 - Real-time event 83",
            Signal::SIG84 => "SIG84 - Real-time event 84",
            Signal::SIG85 => "SIG85 - Real-time event 85",
            Signal::SIG86 => "SIG86 - Real-time event 86",
            Signal::SIG87 => "SIG87 - Real-time event 87",
            Signal::SIG88 => "SIG88 - Real-time event 88",
            Signal::SIG89 => "SIG89 - Real-time event 89",
            Signal::SIG90 => "SIG90 - Real-time event 90",
            Signal::SIG91 => "SIG91 - Real-time event 91",
            Signal::SIG92 => "SIG92 - Real-time event 92",
            Signal::SIG93 => "SIG93 - Real-time event 93",
            Signal::SIG94 => "SIG94 - Real-time event 94",
            Signal::SIG95 => "SIG95 - Real-time event 95",
            Signal::SIG96 => "SIG96 - Real-time event 96",
            Signal::SIG97 => "SIG97 - Real-time event 97",
            Signal::SIG98 => "SIG98 - Real-time event 98",
            Signal::SIG99 => "SIG99 - Real-time event 99",
            Signal::SIG100 => "SIG100 - Real-time event 100",
            Signal::SIG101 => "SIG101 - Real-time event 101",
            Signal::SIG102 => "SIG102 - Real-time event 102",
            Signal::SIG103 => "SIG103 - Real-time event 103",
            Signal::SIG104 => "SIG104 - Real-time event 104",
            Signal::SIG105 => "SIG105 - Real-time event 105",
            Signal::SIG106 => "SIG106 - Real-time event 106",
            Signal::SIG107 => "SIG107 - Real-time event 107",
            Signal::SIG108 => "SIG108 - Real-time event 108",
            Signal::SIG109 => "SIG109 - Real-time event 109",
            Signal::SIG110 => "SIG110 - Real-time event 110",
            Signal::SIG111 => "SIG111 - Real-time event 111",
            Signal::SIG112 => "SIG112 - Real-time event 112",
            Signal::SIG113 => "SIG113 - Real-time event 113",
            Signal::SIG114 => "SIG114 - Real-time event 114",
            Signal::SIG115 => "SIG115 - Real-time event 115",
            Signal::SIG116 => "SIG116 - Real-time event 116",
            Signal::SIG117 => "SIG117 - Real-time event 117",
            Signal::SIG118 => "SIG118 - Real-time event 118",
            Signal::SIG119 => "SIG119 - Real-time event 119",
            Signal::SIG120 => "SIG120 - Real-time event 120",
            Signal::SIG121 => "SIG121 - Real-time event 121",
            Signal::SIG122 => "SIG122 - Real-time event 122",
            Signal::SIG123 => "SIG123 - Real-time event 123",
            Signal::SIG124 => "SIG124 - Real-time event 124",
            Signal::SIG125 => "SIG125 - Real-time event 125",
            Signal::SIG126 => "SIG126 - Real-time event 126",
            Signal::SIG127 => "SIG127 - Real-time event 127",
            Signal::SIGINFO => "SIGINFO - Information request",
            Signal::UNKNOWN => "UNKNOWN - Unknown signal",
            Signal::INTERNAL_DEFAULT => "INTERNAL_DEFAULT - Internal error: printing GDB_SIGNAL_DEFAULT",
            Signal::EXC_BAD_ACCESS => "EXC_BAD_ACCESS - Could not access memory",
            Signal::EXC_BAD_INSTRUCTION => "EXC_BAD_INSTRUCTION - Illegal instruction/operand",
            Signal::EXC_ARITHMETIC => "EXC_ARITHMETIC - Arithmetic exception",
            Signal::EXC_EMULATION => "EXC_EMULATION - Emulation instruction",
            Signal::EXC_SOFTWARE => "EXC_SOFTWARE - Software generated exception",
            Signal::EXC_BREAKPOINT => "EXC_BREAKPOINT - Breakpoint",
            Signal::SIGLIBRT => "SIGLIBRT - librt internal signal",
        };

        write!(f, "{}", s)
    }
}

impl Signal {
    #[cfg(not(feature = "paranoid_unsafe"))]
    pub(crate) fn from_protocol_u8(val: u8) -> Signal {
        if val <= 151 {
            // SAFETY: Signal is repr(u8), and `val` was confirmed to fall in valid range
            unsafe { core::mem::transmute(val) }
        } else {
            Signal::UNKNOWN
        }
    }

    #[cfg(feature = "paranoid_unsafe")]
    pub(crate) fn from_protocol_u8(val: u8) -> Signal {
        match val {
            0 => Signal::SIGZERO,
            1 => Signal::SIGHUP,
            2 => Signal::SIGINT,
            3 => Signal::SIGQUIT,
            4 => Signal::SIGILL,
            5 => Signal::SIGTRAP,
            6 => Signal::SIGABRT,
            7 => Signal::SIGEMT,
            8 => Signal::SIGFPE,
            9 => Signal::SIGKILL,
            10 => Signal::SIGBUS,
            11 => Signal::SIGSEGV,
            12 => Signal::SIGSYS,
            13 => Signal::SIGPIPE,
            14 => Signal::SIGALRM,
            15 => Signal::SIGTERM,
            16 => Signal::SIGURG,
            17 => Signal::SIGSTOP,
            18 => Signal::SIGTSTP,
            19 => Signal::SIGCONT,
            20 => Signal::SIGCHLD,
            21 => Signal::SIGTTIN,
            22 => Signal::SIGTTOU,
            23 => Signal::SIGIO,
            24 => Signal::SIGXCPU,
            25 => Signal::SIGXFSZ,
            26 => Signal::SIGVTALRM,
            27 => Signal::SIGPROF,
            28 => Signal::SIGWINCH,
            29 => Signal::SIGLOST,
            30 => Signal::SIGUSR1,
            31 => Signal::SIGUSR2,
            32 => Signal::SIGPWR,
            33 => Signal::SIGPOLL,
            34 => Signal::SIGWIND,
            35 => Signal::SIGPHONE,
            36 => Signal::SIGWAITING,
            37 => Signal::SIGLWP,
            38 => Signal::SIGDANGER,
            39 => Signal::SIGGRANT,
            40 => Signal::SIGRETRACT,
            41 => Signal::SIGMSG,
            42 => Signal::SIGSOUND,
            43 => Signal::SIGSAK,
            44 => Signal::SIGPRIO,
            45 => Signal::SIG33,
            46 => Signal::SIG34,
            47 => Signal::SIG35,
            48 => Signal::SIG36,
            49 => Signal::SIG37,
            50 => Signal::SIG38,
            51 => Signal::SIG39,
            52 => Signal::SIG40,
            53 => Signal::SIG41,
            54 => Signal::SIG42,
            55 => Signal::SIG43,
            56 => Signal::SIG44,
            57 => Signal::SIG45,
            58 => Signal::SIG46,
            59 => Signal::SIG47,
            60 => Signal::SIG48,
            61 => Signal::SIG49,
            62 => Signal::SIG50,
            63 => Signal::SIG51,
            64 => Signal::SIG52,
            65 => Signal::SIG53,
            66 => Signal::SIG54,
            67 => Signal::SIG55,
            68 => Signal::SIG56,
            69 => Signal::SIG57,
            70 => Signal::SIG58,
            71 => Signal::SIG59,
            72 => Signal::SIG60,
            73 => Signal::SIG61,
            74 => Signal::SIG62,
            75 => Signal::SIG63,
            76 => Signal::SIGCANCEL,
            77 => Signal::SIG32,
            78 => Signal::SIG64,
            79 => Signal::SIG65,
            80 => Signal::SIG66,
            81 => Signal::SIG67,
            82 => Signal::SIG68,
            83 => Signal::SIG69,
            84 => Signal::SIG70,
            85 => Signal::SIG71,
            86 => Signal::SIG72,
            87 => Signal::SIG73,
            88 => Signal::SIG74,
            89 => Signal::SIG75,
            90 => Signal::SIG76,
            91 => Signal::SIG77,
            92 => Signal::SIG78,
            93 => Signal::SIG79,
            94 => Signal::SIG80,
            95 => Signal::SIG81,
            96 => Signal::SIG82,
            97 => Signal::SIG83,
            98 => Signal::SIG84,
            99 => Signal::SIG85,
            100 => Signal::SIG86,
            101 => Signal::SIG87,
            102 => Signal::SIG88,
            103 => Signal::SIG89,
            104 => Signal::SIG90,
            105 => Signal::SIG91,
            106 => Signal::SIG92,
            107 => Signal::SIG93,
            108 => Signal::SIG94,
            109 => Signal::SIG95,
            110 => Signal::SIG96,
            111 => Signal::SIG97,
            112 => Signal::SIG98,
            113 => Signal::SIG99,
            114 => Signal::SIG100,
            115 => Signal::SIG101,
            116 => Signal::SIG102,
            117 => Signal::SIG103,
            118 => Signal::SIG104,
            119 => Signal::SIG105,
            120 => Signal::SIG106,
            121 => Signal::SIG107,
            122 => Signal::SIG108,
            123 => Signal::SIG109,
            124 => Signal::SIG110,
            125 => Signal::SIG111,
            126 => Signal::SIG112,
            127 => Signal::SIG113,
            128 => Signal::SIG114,
            129 => Signal::SIG115,
            130 => Signal::SIG116,
            131 => Signal::SIG117,
            132 => Signal::SIG118,
            133 => Signal::SIG119,
            134 => Signal::SIG120,
            135 => Signal::SIG121,
            136 => Signal::SIG122,
            137 => Signal::SIG123,
            138 => Signal::SIG124,
            139 => Signal::SIG125,
            140 => Signal::SIG126,
            141 => Signal::SIG127,
            142 => Signal::SIGINFO,
            143 => Signal::UNKNOWN,
            144 => Signal::INTERNAL_DEFAULT,
            145 => Signal::EXC_BAD_ACCESS,
            146 => Signal::EXC_BAD_INSTRUCTION,
            147 => Signal::EXC_ARITHMETIC,
            148 => Signal::EXC_EMULATION,
            149 => Signal::EXC_SOFTWARE,
            150 => Signal::EXC_BREAKPOINT,
            151 => Signal::SIGLIBRT,

            _ => Signal::UNKNOWN,
        }
    }
}
