use anyhow::Result;
use clap::{App, Arg, ArgMatches, crate_authors, crate_version};

use gcd_cli::config::Config;
use gcd_cli::constants::*;
use gcd_cli::db::Database;

fn main() -> Result<()> {
    let config = Config::new()?;
    let default_database_file = config.database_file();

    let matches = App::new("gcd-mv")
        .version(&crate_version!()[..])
        .author(crate_authors!())
        .about("Move a project to a new location.")
        .arg(
            Arg::with_name(DATABASE_FILE)
                .short("d")
                .long(DATABASE_FILE)
                .env(DATABASE_FILE)
                .value_name(DATABASE_FILE_VALUE_NAME)
                .default_value(&default_database_file)
                .help(DATABASE_FILE_HELP)
                .required(false)
                .takes_value(true),
        )
        .arg(
            Arg::with_name(FROM_LOCATION)
                .help("Current project location")
                .required(true),
        )
        .arg(
            Arg::with_name(TO_LOCATION)
                .help("New project location")
                .required(true),
        )
        .get_matches();

    let database_file = matches
        .value_of(DATABASE_FILE)
        .unwrap_or(&default_database_file);

    let database = Database::new(database_file)?;

    let (cur_loc, new_loc) = safe_get_required(matches);
    database.move_project(cur_loc.as_str(), new_loc.as_str())?;
    Ok(())
}

fn safe_get_required(matches: ArgMatches) -> (String, String) {
    (matches.value_of(FROM_LOCATION).unwrap().to_owned(), matches.value_of(TO_LOCATION).unwrap().to_owned())
}
