use anyhow::Result;
use clap::{crate_authors, crate_version, App, Arg};

use gcd_cli::config::Config;
use gcd_cli::constants::*;
use gcd_cli::db::Database;
use gcd_cli::lib::current_dir;

const ALIAS: &'static str = "alias";

fn main() -> Result<()> {
    let config = Config::new()?;
    let default_database_file = config.database_file();

    let matches = App::new("gcd-alias")
        .version(&crate_version!()[..])
        .author(crate_authors!())
        .about("Add an alias for the current folder to the database.")
        .arg(
            Arg::with_name("list")
                .short("l")
                .long("list")
                .help("list all aliases")
                .required(false)
                .takes_value(false),
        )
        .arg(
            Arg::with_name("remove")
                .short("r")
                .long("remove")
                .help("Removes alias for current folder")
                .required(false)
                .takes_value(false)
                .conflicts_with("list"),
        )
        .arg(
            Arg::with_name(DATABASE_FILE)
                .short("d")
                .long(DATABASE_FILE)
                .env(DATABASE_FILE)
                .value_name(DATABASE_FILE_VALUE_NAME)
                .default_value(&default_database_file)
                .help(DATABASE_FILE_HELP)
                .required(false)
                .takes_value(true),
        )
        .arg(
            Arg::with_name(ALIAS)
                .help("alias for current project")
                .required_unless_one(&["list", "remove"]),
        )
        .get_matches();

    let database_file = matches
        .value_of(DATABASE_FILE)
        .unwrap_or(&default_database_file);
    let database = Database::new(database_file)?;
    if matches.is_present("list") {
        list(database.all_aliased()?);
    }
    if matches.is_present("remove") {
        if let Some(alias) = matches.value_of(ALIAS) {
            database.remove_alias_by_alias(alias)?;
        } else {
            database.remove_alias(current_dir()?.as_str())?;
        }
    } else {
        if let Some(alias) = matches.value_of(ALIAS) {
            database.alias(current_dir()?.as_str(), alias)?;
        }
    }
    Ok(())
}

fn list(projects: Vec<(String, String)>) {
    for project in projects {
        println!("{:>15} - {}", project.0, project.1);
    }
}

