//! # gbt32960-parser
//!
//! `gbt32960-parser` 是一个国标GB/T32960协议解析库的rust实现.

use chrono::{NaiveDate, NaiveDateTime};
use nom::bytes;
use nom::number;
use nom::IResult;
use num_enum::TryFromPrimitive;
use std::str;
extern crate encoding;
use encoding::all::GB18030;
use encoding::{DecoderTrap, Encoding};

///解析日期时间
pub fn parse_date(input: &[u8]) -> IResult<&[u8], NaiveDateTime> {
    let (input, year) = number::streaming::be_u8(input)?;
    let (input, month) = number::streaming::be_u8(input)?;
    let (input, day) = number::streaming::be_u8(input)?;
    let (input, hour) = number::streaming::be_u8(input)?;
    let (input, minute) = number::streaming::be_u8(input)?;
    let (input, second) = number::streaming::be_u8(input)?;

    let date_time: NaiveDateTime = NaiveDate::from_ymd(
        2000 + (year as i32),
        month as u32,
        day as u32,
    )
    .and_hms(hour as u32, minute as u32, second as u32);
    println!("date_time: {:?}", date_time);
    Ok((input, date_time))
}

///计算bcc校验值
pub fn bcc(input: &[u8]) -> u8 {
    let mut bcc = 0x00;
    for i in 2..input.len() {
        bcc ^= input[i];
        //println!("{:?} {:?}", i, bcc)
    }
    bcc
}

#[derive(Debug)]
///数据包
pub struct Frame {
    ///数据包头部
    pub header: Header,
    ///数据单元
    pub payload: Command,
    ///校验码
    pub check_code: u8,
}

impl Frame {
    ///解析字节数组
    pub fn parse_from_bytes(input: &[u8]) -> IResult<&[u8], Frame> {
        let (input, header) = Header::parse_from_bytes(input)?;
        println!(
            "【parse Header: {:?}】【input remain: {:?}】",
            header,
            input.len()
        );
        let (input, payload) =
            Command::parse_from_bytes(RequestType::try_from(header.request_type).unwrap(), input)?;
        let (input, check_code) = number::streaming::be_u8(input)?;

        Ok((
            input,
            Frame {
                header,
                payload,
                check_code,
            },
        ))
    }

    ///获取数据单元长度
    pub fn get_payload_len(input: &[u8]) -> u16 {
        if input.len() < 25 {
            return 0;
        }
        let b = input[22];
        let e = input[23];
        u16::from_be_bytes([b, e]) + 25
    }
}

#[derive(Debug)]
///数据包头部
pub struct Header {
    ///起始符
    pub begin_symbol: String,
    ///命令标识
    pub request_type: u8,
    ///应答标志
    pub response_tag: u8,
    ///唯一识别码
    pub vin: String,
    ///加密标志
    pub encryption_type: u8,
    ///数据单元长度
    pub payload_length: u16,
}

impl Header {
    ///解析字节数组
    pub fn parse_from_bytes(input: &[u8]) -> IResult<&[u8], Header> {
        let (input, begin_symbol_bytes) = bytes::streaming::take(2usize)(input)?;
        let begin_symbol = GB18030
            .decode(begin_symbol_bytes, DecoderTrap::Ignore)
            .unwrap();
        let (input, request_type) = number::streaming::be_u8(input)?;
        let (input, response_tag) = number::streaming::be_u8(input)?;
        let (input, vin_bytes) = bytes::streaming::take(17usize)(input)?;
        let vin = GB18030.decode(vin_bytes, DecoderTrap::Ignore).unwrap();
        let (input, encryption_type) = number::streaming::be_u8(input)?;
        let (input, payload_length) = number::streaming::be_u16(input)?;

        Ok((
            input,
            Header {
                begin_symbol,
                request_type,
                response_tag,
                vin,
                encryption_type,
                payload_length,
            },
        ))
    }
}

#[derive(Debug, TryFromPrimitive)]
#[repr(u8)]
///命令标识
pub enum RequestType {
    ///车辆登入
    Login = 0x01,
    ///实时信息上报
    RealTime = 0x02,
    ///补发信息上报
    Reissue = 0x03,
    ///车辆登出
    Logout = 0x04,
    ///平台登入
    PlatLogin = 0x05,
    //PlatOut = 0x06,   //平台登出
    ///心跳
    Heartbeat = 0x07,
    //CLOCK_CORRECT = 0x08, //终端校时
    //CONFIG_QUERY = 0x80,  //参数查询命令
    //CONFIG_SETUP = 0x81,  //参数设置命令
    //CONTROL = 0x82,       //车载终端控制命令
}

#[derive(Debug, TryFromPrimitive)]
#[repr(u8)]
///应答标志
pub enum ResponseTag {
    ///成功；接收到的消息正确
    Success = 0x01,
    ///错误；设置未成功
    Failed = 0x02,
    ///VIN重复；VIN重复错误
    VinDup = 0x03,
    ///命令；表示数据包为命令包，而非应答包
    Command = 0xFE,
}

#[derive(Debug, TryFromPrimitive)]
#[repr(u8)]
///加密类型
pub enum Encrypt {
    ///数据不加密
    None = 0x01,
    ///数据经过RSA算法加密
    Rsa = 0x02,
    ///数据经过AES128位算法加密
    Aes128 = 0x03,
    ///表示异常
    Excep = 0xFE,
}


#[derive(Debug)]
///数据单元
pub enum Command {
    ///车辆登入
    VehicleLogin {
        record_time: NaiveDateTime,
        serial_number: u16,
        iccid: String,
        subsystem_num: u8,
        subsystem_num_length: u8,
        subsystem_code: String,
    },
    ///车辆登出
    VehicleLogout {
        record_time: NaiveDateTime,
        serial_number: u16,
    },
    ///平台登入
    PlatformLogin {
        ///登入时间
        login_time: NaiveDateTime,
        ///流水号
        serial_number: u16,
        ///平台用户名
        plat_name: String,
        ///平台密码
        plat_password: String,
        ///加密规则
        encryption_rule: u8,
    },
    ///平台登出
    PlatformLogout {
        ///登出时间
        logout_time: NaiveDateTime,
        ///流水号
        serial_number: u16,
    },
    ///实时信息上报
    RealTimeReport {
        record_time: NaiveDateTime,
        reports: Vec<Report>,
    },
    Time {},
}

impl Command {
    ///解析字节数组
    pub fn parse_from_bytes(request_type: RequestType, input: &[u8]) -> IResult<&[u8], Command> {
        match request_type {
            RequestType::PlatLogin => {
                let (input, login_time) = parse_date(input)?;
                let (input, serial_number) = number::streaming::be_u16(input)?;
                let (input, plat_name_bytes) = bytes::streaming::take(12usize)(input)?;
                let plat_name = GB18030
                    .decode(plat_name_bytes, DecoderTrap::Ignore)
                    .unwrap();
                let (input, plat_password_bytes) = bytes::streaming::take(20usize)(input)?;
                let plat_password = GB18030
                    .decode(plat_password_bytes, DecoderTrap::Ignore)
                    .unwrap();
                let (input, encryption_rule) = number::streaming::be_u8(input)?;
                Ok((
                    input,
                    Command::PlatformLogin {
                        login_time,
                        serial_number,
                        plat_name,
                        plat_password,
                        encryption_rule,
                    },
                ))
            }
            RequestType::RealTime => {
                let (input, record_time) = parse_date(input)?;
                let (input, reports) = Report::parse_from_bytes(input)?;
                Ok((
                    input,
                    Command::RealTimeReport {
                        record_time,
                        reports,
                    },
                ))
            }
            RequestType::Reissue => {
                let (input, record_time) = parse_date(input)?;
                let (input, reports) = Report::parse_from_bytes(input)?;
                Ok((
                    input,
                    Command::RealTimeReport {
                        record_time,
                        reports,
                    },
                ))
            }
            RequestType::Login => {
                let (input, record_time) = parse_date(input)?;
                let (input, serial_number) = number::streaming::be_u16(input)?;
                let (input, iccid_bytes) = bytes::streaming::take(20usize)(input)?;
                let iccid = GB18030.decode(iccid_bytes, DecoderTrap::Ignore).unwrap();
                let (input, subsystem_num) = number::streaming::be_u8(input)?;
                println!("subsystem_num: {:?}", subsystem_num);
                let (input, subsystem_num_length) = number::streaming::be_u8(input)?;
                println!("subsystem_num_length: {:?}", subsystem_num_length);
                let rs = subsystem_num * subsystem_num_length;
                println!("rs: {:?} input: {:?}", rs, input);
                let (input, subsystem_code_bytes) = if rs > 0 {
                    bytes::streaming::take(rs as usize)(input)?
                } else {
                    (input, &b""[..])
                };

                let subsystem_code = GB18030
                    .decode(subsystem_code_bytes, DecoderTrap::Ignore)
                    .unwrap();
                println!("subsystem_code_res: {:?}", subsystem_code);

                Ok((
                    input,
                    Command::VehicleLogin {
                        record_time,
                        serial_number,
                        iccid,
                        subsystem_num,
                        subsystem_num_length,
                        subsystem_code,
                    },
                ))
            }
            RequestType::Logout => {
                let (input, record_time) = parse_date(input)?;
                let (input, serial_number) = number::streaming::be_u16(input)?;
                Ok((
                    input,
                    Command::VehicleLogout {
                        record_time,
                        serial_number,
                    },
                ))
            }
            RequestType::Heartbeat => Ok((input, Command::Time {})),
        }
    }
}

#[derive(Debug)]
///实时信息
pub struct Report {
    ///信息类型标志
    report_type: u8,
    ///信息体
    report_body: ReportInfo,
}

impl Report {
    ///解析字节数组
    pub fn parse_from_bytes(input: &[u8]) -> IResult<&[u8], Vec<Report>> {
        let mut reports = Vec::new();

        let (mut data, report) = Report::parse_from_bytes_reportinfo(input)?;
        reports.push(report);

        loop {
            if data.len() == 1 {
                break;
            }
            let (data1, report1) = Report::parse_from_bytes_reportinfo(data)?;
            data = data1;
            reports.push(report1);
        }

        Ok((data, reports))
    }

    ///解析字节数组
    pub fn parse_from_bytes_reportinfo(input: &[u8]) -> IResult<&[u8], Report> {
        let (input, report_type) = number::streaming::be_u8(input)?;
        println!(
            "【parse ReportInfo-header: {:?}】【input remain: {:?}】",
            report_type,
            input.len()
        );
        let (input, report_body) =
            ReportInfo::parse_from_bytes(ReportType::try_from(report_type).unwrap(), input)?;
        Ok((
            input,
            Report {
                report_type,
                report_body,
            },
        ))
    }
}

#[derive(Debug, TryFromPrimitive)]
#[repr(u8)]
///信息类型标志
pub enum ReportType {
    ///整车数据
    Vehicle = 0x01,
    ///电机数据
    Motor = 0x02,
    ///燃料电池数据
    FuelCell = 0x03,
    ///发动机数据
    Engine = 0x04,
    ///位置数据
    Location = 0x05,
    ///极值数据
    Extreme = 0x06,
    ///报警数据
    Alarm = 0x07,
    ///可充电储能装置电压数据
    RessVoltage = 0x08,
    ///可充电储能装置温度数据
    RessTemperature = 0x09,
}

#[derive(Debug)]
///信息体
pub enum ReportInfo {
    ///整车数据
    Vehicle {
        ///车辆状态
        status: u8,
        ///充电状态
        charge_status: u8,
        ///能源类型
        mode: u8,
        ///车速
        speed: u16,
        ///累计里程
        mileage: u32,
        ///总电压
        voltage: u16,
        ///总电流
        current: u16,
        ///SOC
        soc: u8,
        ///DC-DC 状态
        dc_status: u8,
        ///档位
        shift: u8,
        ///绝缘电阻
        resistance: u16,
        ///预留位
        reserved: u16,
    },
    ///驱动电机数据
    Motor {
        count: u8,
        ///驱动电机个数
        motors: Vec<MotorData>,
    },
    ///燃料电池数据
    FuelCell {
        voltage: u16,
        ///电压
        current: u16,
        ///电流
        con_rate: u16,
        ///消耗率
        probe_count: u16,
        ///温度探针总数
        ntcs: Vec<u8>,
        ///探针温度值
        max_ntc: u16,
        ///氢系统中最高温度
        max_ntc_no: u8,
        ///氢系统中最高温度探针代号
        hc_of_h: u16,
        ///氢气最高浓度 Highest concentration of hydrogen
        hc_of_h_no: u8,
        ///氢气最高浓度传感器代号 Highest concentration of hydrogen
        max_voltage_of_h: u16,
        ///氢气最高压力
        max_voltage_of_h_no: u8,
        ///氢气最高压力传感器代号
        ///高压DC/DC状态
        dc: u8,
    },
    ///发动机数据
    Engine {
        state: u8,
        ///发动机状态
        cs_speed: u16,
        ///曲轴转速
        ///燃料消耗率
        fc_rate: u16,
    },
    ///车辆位置数据
    Location {
        state: u8,
        ///定位状态
        lng: u32,
        ///经度
        ///纬度
        lat: u32,
    },
    ///极值数据
    Extreme {
        max_voltage_sub_sys_no: u8,
        ///最高电压电池子系统号
        max_voltage_sing_no: u8,
        ///最高电压电池单体代号
        max_voltage: u16,
        ///电池单体电压最高值
        min_voltage_sub_sys_no: u8,
        ///最低电压电池子系统号
        min_voltage_sing_no: u8,
        ///最高电压电池单体代号
        min_voltage: u16,
        ///电池单体电压最低值
        max_ntc_sub_sys_no: u8,
        ///最高温度子系统号
        max_ntc_no: u8,
        ///最高温度探针序号
        max_ntc: u8,
        ///最高单体电池温度值
        min_ntc_sub_sys_no: u8,
        ///最低温度子系统号
        min_ntc_no: u8,
        ///最低温度探针序号
        ///最低温度值
        min_ntc: u8,
    },
    ///报警数据
    Alarm {
        max_level: u8,
        ///最高报警等级
        uas: u32,
        ///通用报警标志
        ress_len: u8,
        ///可充电储能装置故障总数N1
        ress_list: Vec<u32>,
        ///可充电储能装置故障代码列表
        mortor_len: u8,
        ///驱动电机故障总数N2
        mortor_list: Vec<u32>,
        ///驱动电机故障代码列表
        engine_len: u8,
        ///发动机故障总数N3
        engine_list: Vec<u32>,
        ///发动机故障代码列表
        other_len: u8,
        ///其他故障总数N4
        ///其他故障列表
        other_list: Vec<u32>,
    },
    ///可充电储能子系统电压
    RessVoltage {
        ///可充电储能子系统个数
        sub_count: u8,
        ///可充电储能子系统电压信息列表
        sub_voltage_data: Vec<VoltageData>,
    },
    ///可充电储能子系统温度
    RessTemperature {
        ///可充电储能子系统个数
        sub_count: u8,
        ///可充电储能子系统温度信息列表
        sub_temperature_data: Vec<TemperatureData>,
    },
}

#[derive(Debug)]
///可充电储能子系统电压
pub struct VoltageData {
    ///可充电储能子系统编号
    no: u8,
    ///可充电储能装置电压
    voltage: u16,
    ///可充电储能装置电流
    current: u16,
    ///单体电池总数
    battery_count: u16,
    ///本帧起始电池序号
    frame_start_battery_no: u16,
    ///本帧单体电池总数
    frame_battery_count: u8,
    ///单体电池电压
    battery_vols: Vec<u16>,
}

impl VoltageData {
    ///解析字节数组
    pub fn parse_from_bytes(input: &[u8]) -> IResult<&[u8], VoltageData> {
        let (input, no) = number::streaming::be_u8(input)?;
        let (input, voltage) = number::streaming::be_u16(input)?;
        let (input, current) = number::streaming::be_u16(input)?;
        let (input, battery_count) = number::streaming::be_u16(input)?;
        let (input, frame_start_battery_no) = number::streaming::be_u16(input)?;
        let (input, frame_battery_count) = number::streaming::be_u8(input)?;

        let mut battery_vols = Vec::new();
        let mut input_tmp = input;
        for i in 0..frame_battery_count {
            let (input1, data) = number::streaming::be_u16(input_tmp)?;
            input_tmp = input1;
            battery_vols.push(data);
        }

        Ok((
            input_tmp,
            VoltageData {
                no,
                voltage,
                current,
                battery_count,
                frame_start_battery_no,
                frame_battery_count,
                battery_vols,
            },
        ))
    }
}

#[derive(Debug)]
///可充电储能子系统温度
pub struct TemperatureData {
    ///可充电储能子系统编号
    no: u8,
    ///可充电储能温度探针个数
    probe_count: u16,
    ///单体电池温度
    temps: Vec<u8>,
}

impl TemperatureData {
    ///解析字节数组
    pub fn parse_from_bytes(input: &[u8]) -> IResult<&[u8], TemperatureData> {
        let (input, no) = number::streaming::be_u8(input)?;
        let (input, probe_count) = number::streaming::be_u16(input)?;

        let mut temps = Vec::new();
        let mut input_tmp = input;
        for i in 0..probe_count {
            let (input1, data) = number::streaming::be_u8(input_tmp)?;
            input_tmp = input1;
            temps.push(data);
        }

        Ok((
            input_tmp,
            TemperatureData {
                no,
                probe_count,
                temps,
            },
        ))
    }
}

impl ReportInfo {
    ///解析字节数组
    pub fn parse_from_bytes(report_type: ReportType, input: &[u8]) -> IResult<&[u8], ReportInfo> {
        match report_type {
            ReportType::Vehicle => {
                let (input, status) = number::streaming::be_u8(input)?;
                let (input, charge_status) = number::streaming::be_u8(input)?;
                let (input, mode) = number::streaming::be_u8(input)?;
                let (input, speed) = number::streaming::be_u16(input)?;
                let (input, mileage) = number::streaming::be_u32(input)?;
                let (input, voltage) = number::streaming::be_u16(input)?;
                let (input, current) = number::streaming::be_u16(input)?;
                let (input, soc) = number::streaming::be_u8(input)?;
                let (input, dc_status) = number::streaming::be_u8(input)?;
                let (input, shift) = number::streaming::be_u8(input)?;
                let (input, resistance) = number::streaming::be_u16(input)?;
                let (input, reserved) = number::streaming::be_u16(input)?;
                Ok((
                    input,
                    ReportInfo::Vehicle {
                        status,
                        charge_status,
                        mode,
                        speed,
                        mileage,
                        voltage,
                        current,
                        soc,
                        dc_status,
                        shift,
                        resistance,
                        reserved,
                    },
                ))
            }
            ReportType::Motor => {
                let (input, count) = number::streaming::be_u8(input)?;
                let mut motors = Vec::new();

                let mut input_tmp = input;

                for i in 0..count {
                    let (input_tmp2, data) = MotorData::parse_from_bytes(input_tmp)?;
                    input_tmp = input_tmp2;
                    motors.push(data);
                }
                Ok((input_tmp, ReportInfo::Motor { count, motors }))
            }
            ReportType::FuelCell => {
                let (input, voltage) = number::streaming::be_u16(input)?;
                let (input, current) = number::streaming::be_u16(input)?;
                let (input, con_rate) = number::streaming::be_u16(input)?;
                let (input, probe_count) = number::streaming::be_u16(input)?;
                let mut ntcs = Vec::new();
                let mut input_temp = input;
                for i in 0..probe_count {
                    let (input1, data) = number::streaming::be_u8(input_temp)?;
                    input_temp = input1;
                    ntcs.push(data);
                }

                let (input_temp, max_ntc) = number::streaming::be_u16(input_temp)?;
                let (input_temp, max_ntc_no) = number::streaming::be_u8(input_temp)?;
                let (input_temp, hc_of_h) = number::streaming::be_u16(input_temp)?;
                let (input_temp, hc_of_h_no) = number::streaming::be_u8(input_temp)?;
                let (input_temp, max_voltage_of_h) = number::streaming::be_u16(input_temp)?;
                let (input_temp, max_voltage_of_h_no) = number::streaming::be_u8(input_temp)?;
                let (input_temp, dc) = number::streaming::be_u8(input_temp)?;
                Ok((
                    input_temp,
                    ReportInfo::FuelCell {
                        voltage,
                        current,
                        con_rate,
                        probe_count,
                        ntcs,
                        max_ntc,
                        max_ntc_no,
                        hc_of_h,
                        hc_of_h_no,
                        max_voltage_of_h,
                        max_voltage_of_h_no,
                        dc,
                    },
                ))
            }
            ReportType::Engine => {
                let (input, state) = number::streaming::be_u8(input)?;
                let (input, cs_speed) = number::streaming::be_u16(input)?;
                let (input, fc_rate) = number::streaming::be_u16(input)?;
                Ok((
                    input,
                    ReportInfo::Engine {
                        state,
                        cs_speed,
                        fc_rate,
                    },
                ))
            }
            ReportType::Location => {
                let (input, state) = number::streaming::be_u8(input)?;
                let (input, lng) = number::streaming::be_u32(input)?;
                let (input, lat) = number::streaming::be_u32(input)?;
                Ok((input, ReportInfo::Location { state, lng, lat }))
            }
            ReportType::Extreme => {
                let (input, max_voltage_sub_sys_no) = number::streaming::be_u8(input)?;
                let (input, max_voltage_sing_no) = number::streaming::be_u8(input)?;
                let (input, max_voltage) = number::streaming::be_u16(input)?;
                let (input, min_voltage_sub_sys_no) = number::streaming::be_u8(input)?;
                let (input, min_voltage_sing_no) = number::streaming::be_u8(input)?;
                let (input, min_voltage) = number::streaming::be_u16(input)?;
                let (input, max_ntc_sub_sys_no) = number::streaming::be_u8(input)?;
                let (input, max_ntc_no) = number::streaming::be_u8(input)?;
                let (input, max_ntc) = number::streaming::be_u8(input)?;
                let (input, min_ntc_sub_sys_no) = number::streaming::be_u8(input)?;
                let (input, min_ntc_no) = number::streaming::be_u8(input)?;
                let (input, min_ntc) = number::streaming::be_u8(input)?;
                Ok((
                    input,
                    ReportInfo::Extreme {
                        max_voltage_sub_sys_no,
                        max_voltage_sing_no,
                        max_voltage,
                        min_voltage_sub_sys_no,
                        min_voltage_sing_no,
                        min_voltage,
                        max_ntc_sub_sys_no,
                        max_ntc_no,
                        max_ntc,
                        min_ntc_sub_sys_no,
                        min_ntc_no,
                        min_ntc,
                    },
                ))
            }
            ReportType::Alarm => {
                let (input, max_level) = number::streaming::be_u8(input)?;
                let (input, uas) = number::streaming::be_u32(input)?;

                let (input, ress_len) = number::streaming::be_u8(input)?;
                let mut buf = input;
                let mut ress_list = Vec::new();
                for i in 0..ress_len {
                    let (input1, data) = number::streaming::be_u32(buf)?;
                    buf = input1;
                    ress_list.push(data);
                }

                let (buf, mortor_len) = number::streaming::be_u8(buf)?;
                let mut buf2 = buf;
                let mut mortor_list = Vec::new();
                for i in 0..mortor_len {
                    let (input1, data) = number::streaming::be_u32(buf2)?;
                    buf2 = input1;
                    mortor_list.push(data);
                }

                let (buf2, engine_len) = number::streaming::be_u8(buf2)?;
                let mut buf3 = buf2;
                let mut engine_list = Vec::new();
                for i in 0..engine_len {
                    let (input1, data) = number::streaming::be_u32(buf3)?;
                    buf3 = input1;
                    engine_list.push(data);
                }

                let (buf3, other_len) = number::streaming::be_u8(buf3)?;
                let mut buf4 = buf3;
                let mut other_list = Vec::new();
                for i in 0..other_len {
                    let (input1, data) = number::streaming::be_u32(buf4)?;
                    buf4 = input1;
                    other_list.push(data);
                }
                Ok((
                    buf4,
                    ReportInfo::Alarm {
                        max_level,
                        uas,
                        ress_len,
                        ress_list,
                        mortor_len,
                        mortor_list,
                        engine_len,
                        engine_list,
                        other_len,
                        other_list,
                    },
                ))
            }
            ReportType::RessVoltage => {
                let (input, sub_count) = number::streaming::be_u8(input)?;

                let mut buf = input;
                let mut sub_voltage_data = Vec::new();
                for i in 0..sub_count {
                    let (input1, data) = VoltageData::parse_from_bytes(input)?;
                    buf = input1;
                    sub_voltage_data.push(data);
                }

                Ok((
                    buf,
                    ReportInfo::RessVoltage {
                        sub_count,
                        sub_voltage_data,
                    },
                ))
            }
            ReportType::RessTemperature => {
                let (input, sub_count) = number::streaming::be_u8(input)?;

                let mut buf = input;
                let mut sub_temperature_data = Vec::new();
                for i in 0..sub_count {
                    let (input1, data) = TemperatureData::parse_from_bytes(input)?;
                    buf = input1;
                    sub_temperature_data.push(data);
                }

                Ok((
                    buf,
                    ReportInfo::RessTemperature {
                        sub_count,
                        sub_temperature_data,
                    },
                ))
            }
        }
    }
}

#[derive(Debug)]
///驱动电机数据
pub struct MotorData {
    ///驱动电机序号
    no: u8,
    ///驱动电机状态
    status: u8,
    ///驱动电机控制器温度
    control_temp: u8,
    ///驱动电机转速
    speed: u16,
    ///驱动电机转矩 (-2000 ~ 4553.1)
    torque: u16,
    ///驱动电机温度
    temp: u8,
    ///电机控制器输入电压
    voltage: u16,
    ///电机控制器直流母线电流
    current: u16,
}

impl MotorData {
    ///解析字节数组
    pub fn parse_from_bytes(input: &[u8]) -> IResult<&[u8], MotorData> {
        let (input, no) = number::streaming::be_u8(input)?;
        let (input, status) = number::streaming::be_u8(input)?;
        let (input, control_temp) = number::streaming::be_u8(input)?;
        let (input, speed) = number::streaming::be_u16(input)?;
        let (input, torque) = number::streaming::be_u16(input)?;
        let (input, temp) = number::streaming::be_u8(input)?;
        let (input, voltage) = number::streaming::be_u16(input)?;
        let (input, current) = number::streaming::be_u16(input)?;

        Ok((
            input,
            MotorData {
                no,
                status,
                control_temp,
                speed,
                torque,
                temp,
                voltage,
                current,
            },
        ))
    }
}

// #[derive(Debug)]
// pub enum VEHICLE_STATUS {
//     ON = 0x01,
//     OFF = 0x02,
//     OTHER = 0x03,
// }

// ///充电状态
// #[derive(Debug)]
// pub enum CHARGE_STATUS {
//     PARK_CHARGING = 0x01,
//     ///停车充电
//     MOVE_CHARGING = 0x02,
//     ///行驶充电
//     UNCHARGED = 0x03,
//     ///未充电状态
//     ///充电完成
//     CHARGED = 0x04,
// }
// #[derive(Debug)]
// pub enum VEHICLE_MODE {
//     ///车辆模式: 纯电, 混合, 燃油
//     ELECTRIC = 0x01,
//     MIXED = 0x02,
//     FUEL = 0x03,
// }
// #[derive(Debug)]
// pub enum DC_STATUS {
//     ///DC-DC 状态: 工作, 断开
//     ON = 0x01,
//     OFF = 0x02,
// }
// #[derive(Debug)]
// pub enum ENGINE_STATUS {
//     ///发动机状态: 工作, 断开
//     ON = 0x01,
//     OFF = 0x02,
// }

// ///车辆档位
// #[derive(Debug)]
// pub enum VEHICLE_SHIFT {
//     N = 0x01,
//     ///  1=0x01,
//     ///  2",
//     ///  3",
//     ///  4",
//     ///  5",
//     ///  6",
//     ///  7",
//     ///  8",
//     ///  9",
//     ///  10",
//     ///  11",
//     ///  12",
//     R = 0x02,
//     D = 0x03,
//     P = 0x04,
// }

#[cfg(test)]
mod tests {
    #[test]
    fn it_works() {
        let result = 2 + 2;
        assert_eq!(result, 4);
    }
}
