use std::path::PathBuf;

/// Result convenience type for the Error type
pub type Result<T, E = Error> = std::result::Result<T, E>;

/// All errors that could happen
#[derive(Debug, Error)]
#[allow(missing_docs)]
pub enum Error {
    #[error("IO error: {0}")]
    Io(#[from] std::io::Error),

    #[error("Could not open file {0:?}: {1}")]
    OpenFile(PathBuf, std::io::Error),

    #[error("Walkdir error: {0}")]
    WalkDir(#[from] walkdir::Error),

    #[error("Mount error: {0}")]
    Mount(#[from] libmount::mountinfo::ParseError),

    #[error("Strip prefix error: {0}")]
    StripPrefix(#[from] std::path::StripPrefixError),

    #[error("Bad .trashinfo file at {0:?}: {1}")]
    BadTrashInfo(PathBuf, TrashInfoError),

    #[error("Date parsing error: {0}")]
    ParseDate(#[from] chrono::format::ParseError),

    #[error("No trash directory exists at: {0:?}")]
    TrashDirDoesntExist(PathBuf),

    #[error("No files in the trash directory: {0:?}")]
    NoFilesInThisDirectory(PathBuf),

    #[error("Put error: {0}")]
    Put(#[from] crate::ops::put::PutError),
}

/// Errors related to .trashinfo files
#[derive(Debug, Error)]
#[allow(missing_docs)]
pub enum TrashInfoError {
    #[error("Missing [TrashInfo] header")]
    MissingHeader,

    #[error("Missing path attribute")]
    MissingPath,

    #[error("Missing date attribute")]
    MissingDate,
}
