use std::env;
use std::path::PathBuf;

use crate::dir::TrashDir;
use crate::errors::Result;
use crate::list;

/// Options to pass to list
#[derive(StructOpt)]
pub struct ListOptions {
    /// The path to the trash directory to list.
    /// By default, this is your home directory's trash ($XDG_DATA_HOME/Trash)
    #[structopt(long = "trash-dir", parse(from_os_str))]
    trash_dir: Option<PathBuf>,

    /// List all files in the trash (by default, only files in the current
    /// directory are listed)
    #[structopt(short = "a", long = "all")]
    all: bool,
}

/// List the contents of a trash directory
pub fn list(options: ListOptions) -> Result<()> {
    let trash_dir = TrashDir::from_opt(options.trash_dir.as_ref());

    let current_dir = env::current_dir()?;

    let mut files = trash_dir
        .iter()?
        .collect::<Result<Vec<_>>>()?
        .into_iter()
        .filter_map(|info| {
            if !options.all && !info.path.starts_with(&current_dir) {
                None
            } else {
                Some(info)
            }
        })
        .collect::<Vec<_>>();
    files.sort_unstable_by_key(|info| info.deletion_date);

    list::print_files_list(files.iter(), false);

    Ok(())
}
