#![deny(warnings)]

use garbage::{
    ops::{self, EmptyOptions, ListOptions, PutOptions, RestoreOptions},
    Result,
};
use structopt::StructOpt;

#[derive(StructOpt)]
struct Opt {
    #[structopt(subcommand)]
    command: Command,

    /// Verbosity (-v, -vv, -vvv, etc)
    #[structopt(short = "v", long = "verbose", parse(from_occurrences))]
    verbose: usize,
}

#[derive(StructOpt)]
enum Command {
    /// Empty a trash directory.
    #[structopt(name = "empty")]
    Empty(EmptyOptions),

    /// List the contents of a trash directory.
    #[structopt(name = "list")]
    List(ListOptions),

    /// Puts files into the trash. (also 'garbage rm')
    ///
    /// If a trash directory isn't specified, the best strategy is picked
    /// for each file that's deleted (after shell glob expansion). The
    /// algorithm for deciding a strategy is specified in the FreeDesktop
    /// Trash spec.
    #[structopt(name = "put", alias = "rm")]
    Put(PutOptions),

    /// Restores files from the trash.
    #[structopt(name = "restore")]
    Restore(RestoreOptions),
}

fn run(cmd: Command) -> Result<()> {
    match cmd {
        Command::Empty(options) => ops::empty(options),
        Command::List(options) => ops::list(options),
        Command::Put(options) => ops::put(options),
        Command::Restore(options) => ops::restore(options),
    }
}

fn main() {
    let opt = Opt::from_args();

    stderrlog::new()
        .module(module_path!())
        .verbosity(opt.verbose)
        .init()
        .unwrap();

    match run(opt.command) {
        Ok(_) => (),
        Err(err) => {
            eprintln!("Error: {}", err);
        }
    }
}
