use futures::AsyncReadExt;

use crate::anyhow::Result;
use crate::runtime::AsyncKernel;
use crate::runtime::Block;
use crate::runtime::BlockMeta;
use crate::runtime::BlockMetaBuilder;
use crate::runtime::MessageIo;
use crate::runtime::MessageIoBuilder;
use crate::runtime::StreamIo;
use crate::runtime::StreamIoBuilder;
use crate::runtime::WorkIo;

pub struct FileSource<T: Send + 'static> {
    file_name: String,
    file: Option<async_fs::File>,
    _type: std::marker::PhantomData<T>,
}

impl<T: Send + 'static> FileSource<T> {
    pub fn new<S: Into<String>>(file_name: S) -> Block {
        Block::new_async(
            BlockMetaBuilder::new("FileSource").build(),
            StreamIoBuilder::new()
                .add_output("out", std::mem::size_of::<T>())
                .build(),
            MessageIoBuilder::new().build(),
            FileSource::<T> {
                file_name: file_name.into(),
                file: None,
                _type: std::marker::PhantomData,
            },
        )
    }
}

#[async_trait]
impl<T: Send + 'static> AsyncKernel for FileSource<T> {
    async fn work(
        &mut self,
        io: &mut WorkIo,
        sio: &mut StreamIo,
        _mio: &mut MessageIo<Self>,
        _meta: &mut BlockMeta,
    ) -> Result<()> {
        let out = sio.output(0).slice::<u8>();
        let item_size = std::mem::size_of::<T>();

        let mut i = 0;

        while i < out.len() {
            match self.file.as_mut().unwrap().read(&mut out[i..]).await {
                Ok(0) => {
                    io.finished = true;
                    break;
                }
                Ok(written) => {
                    i += written;
                }
                Err(e) => panic!("FileSource: Error reading from file: {:?}", e),
            }
        }

        sio.output(0).produce(i / item_size);

        Ok(())
    }

    async fn init(
        &mut self,
        _sio: &mut StreamIo,
        _mio: &mut MessageIo<Self>,
        _meta: &mut BlockMeta,
    ) -> Result<()> {
        self.file = Some(async_fs::File::open(self.file_name.clone()).await.unwrap());
        Ok(())
    }
}
