use super::create::{self, CompletableFuture, CompleteHandle};

/// Creates a new [`CompletableFuture`] and the associated [`CompleteHandle`] inside a scope.
///
/// An alternative API to [`create`], that while being less ergonomic is harder to deadlock (see
/// [`create`] documentation).
///
/// ```rust
/// # use future_handles::sync;
/// # async fn func() -> u32 {
/// let future = sync::scoped(|handle| {
///     func_with_callback(|res| {
///         handle.complete(res);
///     });
/// });
///
/// match future.await {
///     // The callback was invoked and the result set via the handle.
///     Ok(res) => res,
///     // The callback was never invoked, but the handle has been dropped.
///     Err(err) => panic!("Handle was dropped without setting a value")
/// }
/// # }
/// # fn func_with_callback<F>(func: F)
/// #    where F: FnOnce(u32) {
/// #    func(1);
/// # }
/// ```
///
/// [`create`]: super::create::create
/// [`CompletableFuture`]: super::CompletableFuture
/// [`CompleteHandle`]: super::CompleteHandle
pub fn scoped<T, F>(func: F) -> CompletableFuture<T>
where
    T: Send,
    F: FnOnce(CompleteHandle<T>),
{
    let (fut, handle) = create::create();
    func(handle);
    fut
}

// Doesn't execute the closure until polled for the first time.
// pub fn scoped_lazy()
