//! A non thread-safe [`CompletableFuture`] and [`CompleteHandle`] implementation.
//!
/// [`CompletableFuture`]: unsync/struct.CompletableFuture.html
/// [`CompleteHandle`]: unsync/struct.CompleteHandle.html
use std::cell::RefCell;
use std::future::Future;
use std::pin::Pin;
use std::rc::{Rc, Weak};
use std::task::{Context, Poll};

use crate::state::State;
use crate::{HandleError, HandleResult};

/// Creates a new [`CompletableFuture`] and it's associated [`CompleteHandle`].
///
/// [`CompletableFuture`]: struct.CompletableFuture.html
/// [`CompleteHandle`]: struct.CompleteHandle.html
pub fn create<T>() -> (CompletableFuture<T>, CompleteHandle<T>) {
    let payload = Rc::new(RefCell::new(State::new()));
    let weak = Rc::downgrade(&payload);
    (
        CompletableFuture { inner: payload },
        CompleteHandle { inner: weak },
    )
}

/// A handle to complete the associated [`CompletableFuture`].
/// It can be safely dropped without setting a value.
///
/// [`CompletableFuture`]: struct.CompletableFuture.html
pub struct CompleteHandle<T> {
    inner: Weak<RefCell<State<HandleResult<T>>>>,
}

impl<T> CompleteHandle<T> {
    fn try_set(&self, value: HandleResult<T>) {
        if let Some(rc) = self.inner.upgrade() {
            if let Some(waker) = rc.borrow_mut().try_set_result(value) {
                waker.wake();
            }
        }
    }

    /// Complete the future consuming the handle.
    pub fn complete(self, value: T) {
        self.try_set(Ok(value));
    }
}

impl<T> Drop for CompleteHandle<T> {
    fn drop(&mut self) {
        self.try_set(Err(HandleError::DroppedBeforeComplete));
    }
}

/// A future that can only be completed by the associated [`CompleteHandle`].
///
/// [`CompleteHandle`]: struct.CompleteHandle.html
pub struct CompletableFuture<T> {
    inner: Rc<RefCell<State<HandleResult<T>>>>,
}

impl<T> Future for CompletableFuture<T> {
    /// The associated [`CompleteHandle`] may be dropped before setting a result, as such, it must
    /// be handled by returning a [`HandleResult`].
    type Output = HandleResult<T>;

    fn poll(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Self::Output> {
        let mut state = self.inner.borrow_mut();
        match state.try_consume() {
            None => {
                state.update_waker(cx.waker().clone());
                Poll::Pending
            }
            Some(res) => Poll::Ready(res),
        }
    }
}

#[cfg(test)]
mod tests {
    use std::mem;

    use futures::executor::LocalPool;
    use futures::task::LocalSpawnExt;

    use super::*;

    fn get_test_objects() -> (
        LocalPool,
        impl Future<Output = HandleResult<()>>,
        CompleteHandle<()>,
    ) {
        let (fut, comp) = create::<()>();
        (LocalPool::new(), fut, comp)
    }

    #[test]
    fn return_value() {
        let (mut pool, fut, comp) = get_test_objects();

        pool.spawner()
            .spawn_local(async move {
                fut.await.unwrap();
            })
            .unwrap();

        pool.spawner()
            .spawn_local(async {
                comp.complete(());
            })
            .unwrap();

        pool.run();
    }

    #[test]
    fn drop() {
        let (mut pool, fut, comp) = get_test_objects();

        pool.spawner()
            .spawn_local(async move {
                match fut.await {
                    Err(err) => {
                        match err {
                            HandleError::DroppedBeforeComplete => {} // Ok
                        }
                    }
                    Ok(_) => panic!("Completer dropped but future returned Ok"),
                }
            })
            .unwrap();

        pool.spawner()
            .spawn_local(async {
                mem::drop(comp);
            })
            .unwrap();

        pool.run();
    }

    #[test]
    fn complete_before_await() {
        let (mut pool, fut, comp) = get_test_objects();

        comp.complete(());
        pool.spawner().spawn_local(async move {
            fut.await.unwrap();
        }).unwrap();
        pool.run();
    }
}
