//! A thread-safe [`CompletableFuture`] and [`CompleteHandle`] implementation.
//!
//! ### Implementation notes
//! Currently it uses a spin lock to make the implementation thread-safe.
//!
/// [`CompletableFuture`]: sync/struct.CompletableFuture.html
/// [`CompleteHandle`]: sync/struct.CompleteHandle.html
use std::future::Future;
use std::pin::Pin;
use std::sync::{Arc, Weak};
use std::task::{Context, Poll};

use spinning_top::Spinlock;

use crate::state::State;
use crate::{HandleResult, HandleError};

/// Creates a new [`CompletableFuture`] and it's associated [`CompleteHandle`].
///
/// [`CompletableFuture`]: struct.CompletableFuture.html
/// [`CompleteHandle`]: struct.CompleteHandle.html
pub fn create<T: Send>() -> (CompletableFuture<T>, CompleteHandle<T>) {
    let payload = Arc::new(Spinlock::new(State::new()));
    let weak = Arc::downgrade(&payload);
    (
        CompletableFuture { inner: payload },
        CompleteHandle { inner: weak },
    )
}

/// A handle to complete the associated [`CompletableFuture`].
/// It can be safely dropped without setting a value.
///
/// [`CompletableFuture`]: struct.CompletableFuture.html
pub struct CompleteHandle<T> {
    inner: Weak<Spinlock<State<HandleResult<T>>>>,
}

impl<T> CompleteHandle<T> {
    fn try_set(&self, value: HandleResult<T>) {
        if let Some(arc) = self.inner.upgrade() {
            if let Some(waker) = {
                let mut guard = arc.lock();
                guard.try_set_result(value)
            } {
                waker.wake();
            }
        }
    }

    /// Complete the future consuming the handle.
    pub fn complete(self, value: T) {
        self.try_set(Ok(value));
    }
}

impl<T> Drop for CompleteHandle<T> {
    fn drop(&mut self) {
        self.try_set(Err(HandleError::DroppedBeforeComplete));
    }
}

/// A future that can only be completed by the associated [`CompleteHandle`].
///
/// [`CompleteHandle`]: struct.CompleteHandle.html
pub struct CompletableFuture<T> {
    inner: Arc<Spinlock<State<HandleResult<T>>>>,
}

impl<T: Send> Future for CompletableFuture<T> {
    /// The associated [`CompleteHandle`] may be dropped before setting a result, as such, it must
    /// be handled by returning a [`HandleResult`].
    type Output = HandleResult<T>;

    fn poll(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Self::Output> {
        let mut guard = self.inner.lock();
        match guard.try_consume() {
            None => {
                guard.update_waker(cx.waker().clone());
                Poll::Pending
            }
            Some(res) => Poll::Ready(res),
        }
    }
}

#[cfg(test)]
mod tests {
    use std::mem;

    use futures::executor::ThreadPool;
    use futures::task::SpawnExt;

    use super::*;

    fn get_test_objects() -> (
        ThreadPool,
        impl Future<Output = HandleResult<()>>,
        CompleteHandle<()>,
    ) {
        let (fut, comp) = create::<()>();
        (ThreadPool::new().unwrap(), fut, comp)
    }

    #[test]
    fn return_value() {
        let (pool, fut, comp) = get_test_objects();

        pool.spawn(async move {
            fut.await.unwrap();
        })
        .unwrap();

        pool.spawn(async {
            comp.complete(());
        })
        .unwrap();
    }

    #[test]
    fn drop() {
        let (pool, fut, comp) = get_test_objects();

        pool.spawn(async move {
            match fut.await {
                Err(err) => {
                    match err {
                        HandleError::DroppedBeforeComplete => {} // Ok
                    }
                }
                Ok(_) => panic!("Completer dropped but future returned Ok"),
            }
        })
        .unwrap();

        pool.spawn(async {
            mem::drop(comp);
        })
        .unwrap();
    }

    #[test]
    fn complete_before_await() {
        let (pool, fut, comp) = get_test_objects();

        comp.complete(());
        pool.spawn(async move {
            fut.await.unwrap();
        }).unwrap();
    }
}
