//! # Overview
//!
//! A crate to complete futures via a remote handle.
//!
//! Although it's design shouldn't restrain it's usages, this crate is mainly intended as an utility
//! to interface asynchronous functions running callbacks at competition with rust's async/await
//! paradigm.
//!
//! ```rust
//! # use futures::task::LocalSpawnExt;
//! # use futures::executor::LocalPool;
//! #
//! # use future_handles::unsync;
//! #
//! async fn func() -> Option<u32> {
//!     let (future, handle) = unsync::create();
//!
//!     func_with_callback(|| {
//!         handle.complete(1);
//!     });
//!
//!     match future.await {
//!         // The callback was invoked and the result set via the handle.
//!         Ok(res) => Some(res),
//!         // The callback was never invoked, but the handle has been dropped.
//!         Err(_) => None
//!     }
//! }
//! #
//! # fn func_with_callback<F>(func: F)
//! #    where F: FnOnce() {
//! #    func();
//! # }
//! #
//! # fn main() {
//! #    let mut pool = LocalPool::new();
//! #    pool.spawner().spawn_local(async { let _ = func(); }).unwrap();
//! #    pool.run();
//! # }
//! ```
//!
//! This crate comes a non thread-safe [`unsync`] implementation by default. To make the [`sync`]
//! thread-safe implementation available, enable the `sync` feature.
//!
//! [`unsync`]: unsync.html
//! [`sync`]: sync.html
//!
//! # Danger!
//!
//! Do **NOT** do this!
//! ```rust,ignore
//! async fn func() {
//!     let (future, handle) = unsync::create();
//!
//!     // Start awaiting here...
//!     future.await.unwrap();
//!     // Now we'll never be able set the result!
//!     handle.complete(1);
//! }
//! ```
//! Awaiting a `CompletableFuture` before setting the result or dropping the associated
//! `CompleteHandle` or will cause a **deadlock**!

mod state;
#[cfg(feature = "sync")]
pub mod sync;
pub mod unsync;

use thiserror::Error;

/// The error returned by `CompletableFuture`s.
#[derive(Error, Debug)]
pub enum HandleError {
    /// A `Completer` handle may always be dropped before setting a value as result.
    #[error("the completer has been dropped before it was assigned a value")]
    DroppedBeforeComplete,
}

pub type HandleResult<T> = Result<T, HandleError>;
