use std::os::raw::c_char;

/// Event enum.
#[repr(C)]
#[allow(dead_code)]
pub enum FFIEvent {
    /// Mouse cursor enters window area.
    MouseEnter,

    /// Mouse cursor leaves window area.
    MouseLeave,

    /// Mouse button press/release..
    MouseButton {
        state: FFIElementState,
        button: FFIMouseButton,
    },

    /// Mouse move.
    MouseMove { position: FFIPosition },

    /// Mouse scroll wheel rolled or touchpad scroll gesture.
    ScrollWheel { delta: FFIScrollDelta },

    /// Key pressed/released.
    KeyEvent {
        state: FFIElementState,
        keycode: i32,
        is_repeat: bool,
        modifiers: FFIKeyModifiers,
        text: *const c_char,
    },

    /// Window resized.
    Resize { width: i32, height: i32 },
}

/// Element state.
#[repr(C)]
#[allow(dead_code)]
pub enum FFIElementState {
    Pressed,
    Released,
}

/// Mouse button enumeration.
#[repr(C)]
#[allow(dead_code)]
pub enum FFIMouseButton {
    Left,
    Right,
    Middle,
    Other(u8),
}

/// Position.
#[repr(C)]
#[allow(dead_code)]
pub struct FFIPosition {
    pub x: f32,
    pub y: f32,
}

/// Scroll delta enum.
#[repr(C)]
#[allow(dead_code)]
pub enum FFIScrollDelta {
    /// Amount of lines to scroll horizontally and vertically.
    /// This is generated by mouse wheel.
    LineDelta(f32, f32),

    /// Amount of pixels to scroll horizontally and vertically.
    /// This is generated by touchpad.
    PixelDelta(f32, f32),
}

#[repr(C)]
#[allow(dead_code)]
pub struct FFIKeyModifiers {
    pub shift: bool,
    pub ctrl: bool,
    pub alt: bool,
    pub win: bool,
    pub keypad: bool,
}

impl FFIEvent {
    /// Allocates new Event enum. Can be called from C.
    #[no_mangle]
    pub extern "C" fn alloc_ffi_event() -> *mut FFIEvent {
        Box::into_raw(Box::new(FFIEvent::MouseEnter))
    }

    /// Free Event enum. Can be called from C.
    #[no_mangle]
    pub extern "C" fn free_ffi_event(event: *mut FFIEvent) {
        unsafe {
            Box::from_raw(event);
        }
    }
}
