macro_rules! duplicate {
    ($file: ty) => {
        pub fn duplicate(file: &$file) -> Result<File> {
            unsafe {
                let mut handle = ptr::null_mut();
                let current_process = GetCurrentProcess();
                let ret = DuplicateHandle(current_process,
                                          file.as_raw_handle(),
                                          current_process,
                                          &mut handle,
                                          0,
                                          true as BOOL,
                                          DUPLICATE_SAME_ACCESS);
                if ret == 0 {
                    Err(Error::last_os_error())
                } else {
                    Ok(<$file>::from_raw_handle(handle))
                }
            }
        }
    };
}

macro_rules! lock_impl {
    ($file: ty) => {
        pub fn lock_shared(file: &$file) -> Result<()> {
            lock_file(file, 0)
        }

        pub fn lock_exclusive(file: &$file) -> Result<()> {
            lock_file(file, LOCKFILE_EXCLUSIVE_LOCK)
        }

        pub fn try_lock_shared(file: &$file) -> Result<()> {
            lock_file(file, LOCKFILE_FAIL_IMMEDIATELY)
        }

        pub fn try_lock_exclusive(file: &$file) -> Result<()> {
            lock_file(file, LOCKFILE_EXCLUSIVE_LOCK | LOCKFILE_FAIL_IMMEDIATELY)
        }

        pub fn unlock(file: &$file) -> Result<()> {
            unsafe {
                let ret = UnlockFile(file.as_raw_handle(), 0, 0, !0, !0);
                if ret == 0 { Err(Error::last_os_error()) } else { Ok(()) }
            }
        }

        fn lock_file(file: &$file, flags: DWORD) -> Result<()> {
            unsafe {
                let mut overlapped = mem::zeroed();
                let ret = LockFileEx(file.as_raw_handle(), flags, 0, !0, !0, &mut overlapped);
                if ret == 0 { Err(Error::last_os_error()) } else { Ok(()) }
            }
        }
    };
}

#[cfg(feature = "sync")]
pub(crate) mod sync_impl;
#[cfg(any(feature = "smol-async", feature = "std-async", feature = "tokio-async"))]
pub(crate) mod async_impl;

use std::io::{Error, Result};
use std::path::Path;
use winapi::shared::winerror::ERROR_LOCK_VIOLATION;
use winapi::shared::minwindef::DWORD;
use winapi::um::fileapi::GetDiskFreeSpaceW;
use winapi::um::fileapi::GetVolumePathNameW;
use crate::FsStats;

pub fn lock_error() -> Error {
    Error::from_raw_os_error(ERROR_LOCK_VIOLATION as i32)
}

fn volume_path(path: &Path, volume_path: &mut [u16]) -> Result<()> {
    let path_utf8: Vec<u16> = path.as_os_str().encode_wide().chain(Some(0)).collect();
    unsafe {
        let ret = GetVolumePathNameW(path_utf8.as_ptr(),
                                     volume_path.as_mut_ptr(),
                                     volume_path.len() as DWORD);
        if ret == 0 { Err(Error::last_os_error()) } else { Ok(())
        }
    }
}

pub fn statvfs(path: &Path) -> Result<FsStats> {
    let root_path: &mut [u16] = &mut [0; 261];
    volume_path(path, root_path)?;
    unsafe {

        let mut sectors_per_cluster = 0;
        let mut bytes_per_sector = 0;
        let mut number_of_free_clusters = 0;
        let mut total_number_of_clusters = 0;
        let ret = GetDiskFreeSpaceW(root_path.as_ptr(),
                                    &mut sectors_per_cluster,
                                    &mut bytes_per_sector,
                                    &mut number_of_free_clusters,
                                    &mut total_number_of_clusters);
        if ret == 0 {
            Err(Error::last_os_error())
        } else {
            let bytes_per_cluster = sectors_per_cluster as u64 * bytes_per_sector as u64;
            let free_space = bytes_per_cluster * number_of_free_clusters as u64;
            let total_space = bytes_per_cluster * total_number_of_clusters as u64;
            Ok(FsStats {
                free_space,
                available_space: free_space,
                total_space,
                allocation_granularity: bytes_per_cluster,
            })
        }
    }
}