extern crate chrono;
extern crate rusqlite;

pub use super::beans::*;
use chrono::NaiveDateTime;
use rusqlite::{params, Connection, Result, ToSql};
use uuid::Uuid;

///
/// Roast data structure
///
#[derive(Debug, Clone, PartialEq, PartialOrd)]
pub struct Roast {
    pub id: String,
    pub bean_name: Bean,
    pub starting_weight: f32,
    pub beans_in: NaiveDateTime,
    pub first_crack_start: NaiveDateTime,
    pub first_crack_finish: NaiveDateTime,
    pub second_crack_start: Option<NaiveDateTime>,
    pub second_crack_finish: Option<NaiveDateTime>,
    pub beans_out: NaiveDateTime,
}

///
/// Methods for roast struct
///
impl Roast {
    pub fn new(bean: Bean) -> Self {
        let now = chrono::Utc::now().naive_utc();
        Roast {
            id: Uuid::new_v4().to_string(),
            bean_name: bean,
            starting_weight: 0.000,
            beans_in: now,
            first_crack_start: now,
            first_crack_finish: now,
            second_crack_start: None,
            second_crack_finish: None,
            beans_out: now,
        }
    }

    ///
    /// Insert new roast into database
    ///
    pub fn insert(&mut self, db_file: &'static str) -> Result<Self> {
        // Open connection to database
        let mut conn = Connection::open(db_file)?;

        // Begin database transaction
        let tx = conn.transaction()?;
        let mut qry = tx.prepare(
            "
                    INSERT INTO roasts (
                        id, 
                        bean_name, 
                        starting_weight,
                        beans_in, 
                        first_crack_start, 
                        first_crack_finish, 
                        second_crack_start, 
                        second_crack_finish, 
                        beans_out)
                    VALUES (?1, ?2, ?3, ?4, ?5, ?6, ?7, ?8, ?9);

                    ",
        )?;

        qry.execute(params![
            self.id,
            self.bean_name.bean_nickname,
            self.starting_weight,
            self.beans_in.timestamp(),
            self.first_crack_start.timestamp(),
            self.first_crack_finish.timestamp().to_sql().unwrap(),
            match self.second_crack_start {
                Some(scs) => {
                    Some(scs.timestamp())
                }
                None => None,
            },
            match self.second_crack_finish {
                Some(scf) => {
                    Some(scf.timestamp())
                }
                None => None,
            },
            self.beans_out.timestamp().to_sql().unwrap(),
        ])?;

        qry.finalize()?;

        tx.execute(
            "
                UPDATE beans 
                SET amount_in_stock = ?1 - ?2
                WHERE bean_nickname = ?3;
            ",
            params![
                self.bean_name.amount_in_stock,
                self.starting_weight,
                self.bean_name.bean_nickname,
            ],
        )?;

        tx.commit()?;

        self.bean_name.amount_in_stock -= self.starting_weight;

        Ok(self.to_owned())
    }

    ///
    /// Delete roast from database
    ///
    pub fn delete(&self, db_file: &'static str) -> Result<()> {
        let conn = Connection::open(db_file).expect("Problem opening database");

        conn.execute("DELETE FROM roasts WHERE ?1 like \"id\" ", params![self.id])
            .expect("Problem deleting roast");
        Ok(())
    }
}
