//!
//! Data structures and some methods for beans
//!
extern crate rusqlite;

use rusqlite::{params, Connection, Result, ToSql};

/// Data structure for beans in stock
#[derive(Debug, Clone, Default, PartialEq, PartialOrd)]
pub struct Bean {
    pub id: String,
    pub bean_nickname: String,
    pub country_of_origin: String,
    pub region: String,
    pub grower: String,
    pub amount_in_stock: f32,
}

/// Bean mappings for database
impl Bean {
    /// Insert new bean into database
    pub fn insert(&self, db_file: &'static str) -> Result<()> {
        let mut conn = Connection::open(db_file)?;
        let tx = conn.transaction()?;
        tx.execute(
            "
                      INSERT INTO beans (id, \
                      bean_nickname, \
                      country_of_origin, \
                      region, grower, \
                      amount_in_stock)
                      VALUES (?1, ?2, ?3, ?4, ?5, ?6);
                      ",
            params![
                self.id.to_sql()?,
                self.bean_nickname.to_sql()?,
                self.country_of_origin.to_sql()?,
                self.region.to_sql()?,
                self.grower.to_sql()?,
                self.amount_in_stock,
            ],
        )?;

        // Commit transaction to database
        tx.commit()?;

        Ok(())
    }

    /// Delete bean from database
    pub fn delete(self, db_file: &'static str) -> Result<()> {
        let conn = Connection::open(db_file).unwrap();

        conn.execute(
            "DELETE FROM beans WHERE ?1 LIKE id;",
            params![self.id.as_str(),],
        )?;

        conn.close().expect("Problem closing database connection");

        Ok(())
    }

    /// Get bean info from database
    pub fn get(&mut self, db_file: &'static str) -> Result<Self> {
        let mut conn = Connection::open(db_file)?;
        let tx = conn.transaction()?;
        let mut qry = tx
            .prepare(
                "
            SELECT id, bean_nickname, country_of_origin, region, grower, amount_in_stock
            FROM beans 
            WHERE bean_nickname LIKE ?1;
            ",
            )
            .expect("Problem preparing statement");
        let bean_rows = qry.query_map(params![self.clone().bean_nickname], |row| {
            Ok({
                self.id = row.get(0)?;
                self.clone().bean_nickname = row.get(1)?;
                self.country_of_origin = row.get(2)?;
                self.region = row.get(3)?;
                self.grower = row.get(4)?;
                self.amount_in_stock = row.get(5)?;
            })
        })?;

        for row in bean_rows {
            row.unwrap()
        }

        Ok(self.to_owned())
    }
}
