//!
//! Data structures and some methods for beans
//!
extern crate rusqlite;

use rusqlite::{params, Connection, Result};

/// Data structure for beans in stock
#[derive(Debug, Clone, Default)]
pub struct Bean {
    pub id: String,
    pub bean_nickname: String,
    pub country_of_origin: String,
    pub region: String,
    pub grower: String,
}

/// Bean mappings for database
impl Bean {
    /// Insert new bean into database
    pub fn insert(&self, db_file: &'static str) -> Result<()> {
        let mut conn = Connection::open(db_file)?;
        let tx = conn.transaction()?;
        tx.execute(
            "
                      INSERT INTO beans (id, bean_nickname, country_of_origin, region, grower)
                      VALUES (?1, ?2, ?3, ?4, ?5)
                      ",
            params![
                self.id,
                self.bean_nickname,
                self.country_of_origin,
                self.region,
                self.grower,
            ],
        )?;

        // Commit transaction to database
        tx.commit()?;

        Ok(())
    }

    /// Delete bean from database
    pub fn delete(self, db_file: &'static str) -> rusqlite::Result<()> {
        let conn = Connection::open(db_file).unwrap();

        conn.execute(
            "DELETE FROM beans WHERE ?1 LIKE id; COMMIT;",
            params![self.id.as_str(),],
        )?;

        conn.close().expect("Problem closing database connection");

        Ok(())
    }

    pub fn get(&mut self, db_file: &'static str) -> rusqlite::Result<Bean> {
        let mut conn = Connection::open(db_file)?;
        let tx = conn.transaction()?;
        let mut qry = tx.prepare(
            "
                SELECT id, bean_nickname, country_of_origin, region, grower
                FROM beans WHERE ? = 'id';
            ",
        )?;
        let mut bean_row = qry.query(params![self.id])?;
        while let Some(row) = bean_row.next()? {
            self.id = row.get(0)?;
        }
        Ok(self.to_owned())
    }
}
