extern crate chrono;
extern crate rusqlite;

pub use super::beans::*;
use chrono::NaiveDateTime;
use rusqlite::{params, Connection, Result};
use uuid::Uuid;

///
/// Roast data structure
///
#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct Roast {
    pub id: String,
    pub bean_name: String,
    pub starting_weight: i32,
    pub beans_in: NaiveDateTime,
    pub first_crack_start: NaiveDateTime,
    pub first_crack_finish: NaiveDateTime,
    pub second_crack_start: Option<NaiveDateTime>,
    pub second_crack_finish: Option<NaiveDateTime>,
    pub beans_out: NaiveDateTime,
}

///
/// Methods for roast struct
///
impl Roast {
    pub fn new(bean: Bean) -> Roast {
        let now = chrono::Utc::now().naive_utc();
        Roast {
            id: Uuid::new_v4().to_string(),
            bean_name: bean.bean_nickname,
            starting_weight: 0,
            beans_in: now,
            first_crack_start: now,
            first_crack_finish: now,
            second_crack_start: None,
            second_crack_finish: None,
            beans_out: now,
        }
    }

    ///
    /// Insert new roast into database
    ///
    pub fn insert(&self, db_file: &'static str) -> Result<()> {
        // Open connection to database
        let mut conn = Connection::open(db_file)?;

        // Begin database transaction
        let tx = conn.transaction()?;
        tx.execute(
            "
                    INSERT INTO roasts (
                        id, 
                        bean_name, 
                        starting_weight,
                        beans_in, 
                        first_crack_start, 
                        first_crack_finish, 
                        second_crack_start, 
                        second_crack_finish, 
                        beans_out)
                    VALUES (?1, ?2, ?3, ?4, ?5, ?6, ?7, ?8, ?9);
                    ",
            params![
                self.id,
                self.bean_name,
                self.starting_weight,
                self.beans_in.timestamp(),
                self.first_crack_start.timestamp(),
                self.first_crack_finish.timestamp(),
                self.second_crack_start,
                self.second_crack_finish,
                self.beans_out.timestamp(),
            ],
        )?;

        // Commit transaction to database
        tx.commit()?;

        Ok(())
    }

    ///
    /// Delete roast from database
    ///
    pub fn delete(&self, db_file: &'static str) -> Result<()> {
        let conn = Connection::open(db_file).expect("Problem opening database");

        conn.execute("DELETE FROM roasts WHERE ?1 like \"id\" ", params![self.id])
            .expect("Problem deleting roast");
        Ok(())
    }

    pub fn get(&mut self, db_file: String) -> Result<()> {
        let mut conn = Connection::open(db_file)?;
        let tx = conn.transaction()?;
        let mut qry = tx
            .prepare("SELECT * FROM beans WHERE bean_nickname = \"?1\";")
            .unwrap();
        self.bean_name = qry.execute(params![self.bean_name])?.to_string();
        Ok(())
    }
}
