use serde::{Deserialize, Serialize};
use std::path::PathBuf;

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
#[serde(untagged)]
pub enum InputMessageContent {
    InputTextMessageContentVariant(InputTextMessageContent),
    InputLocationMessageContentVariant(InputLocationMessageContent),
    InputVenueMessageContentVariant(InputLocationMessageContent),
    InputContactMessageContentVariant(InputLocationMessageContent),
    InputInvoiceMessageContentVariant(InputInvoiceMessageContent),
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InputFile {
    path: PathBuf,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct VoiceChatStarted {}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct VoiceChatScheduled {
    start_date: isize,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct CallbackGame {}

#[derive(Clone, Debug, Serialize, Deserialize, PartialEq)]
#[serde(untagged)]
pub enum FileEnum {
    InputFileVariant(InputFile),
    StringVariant(String),
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Update {
    update_id: isize,

    #[serde(skip_serializing_if = "Option::is_none")]
    message: Option<Message>,

    #[serde(skip_serializing_if = "Option::is_none")]
    edited_message: Option<Message>,

    #[serde(skip_serializing_if = "Option::is_none")]
    channel_post: Option<Message>,

    #[serde(skip_serializing_if = "Option::is_none")]
    edited_channel_post: Option<Message>,

    #[serde(skip_serializing_if = "Option::is_none")]
    inline_query: Option<InlineQuery>,

    #[serde(skip_serializing_if = "Option::is_none")]
    chosen_inline_result: Option<ChosenInlineResult>,

    #[serde(skip_serializing_if = "Option::is_none")]
    callback_query: Option<CallbackQuery>,

    #[serde(skip_serializing_if = "Option::is_none")]
    shipping_query: Option<ShippingQuery>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pre_checkout_query: Option<PreCheckoutQuery>,

    #[serde(skip_serializing_if = "Option::is_none")]
    poll: Option<Poll>,

    #[serde(skip_serializing_if = "Option::is_none")]
    poll_answer: Option<PollAnswer>,

    #[serde(skip_serializing_if = "Option::is_none")]
    my_chat_member: Option<ChatMemberUpdated>,

    #[serde(skip_serializing_if = "Option::is_none")]
    chat_member: Option<ChatMemberUpdated>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct WebhookInfo {
    url: String,

    has_custom_certificate: bool,

    pending_update_count: isize,

    #[serde(skip_serializing_if = "Option::is_none")]
    ip_address: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    last_error_date: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    last_error_message: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    max_connections: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    allowed_updates: Option<Vec<String>>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct User {
    id: isize,

    is_bot: bool,

    first_name: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    last_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    username: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    language_code: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_join_groups: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_read_all_group_messages: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    supports_inline_queries: Option<bool>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Chat {
    id: isize,
    #[serde(rename = "type")]
    type_field: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    title: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    username: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    first_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    last_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    photo: Option<ChatPhoto>,

    #[serde(skip_serializing_if = "Option::is_none")]
    bio: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    description: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    invite_link: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pinned_message: Option<Box<Message>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    permissions: Option<ChatPermissions>,

    #[serde(skip_serializing_if = "Option::is_none")]
    slow_mode_delay: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    message_auto_delete_time: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    sticker_set_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_set_sticker_set: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    linked_chat_id: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    location: Option<ChatLocation>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Message {
    message_id: isize,

    #[serde(skip_serializing_if = "Option::is_none")]
    from: Option<User>,

    #[serde(skip_serializing_if = "Option::is_none")]
    sender_chat: Option<Chat>,

    date: isize,

    chat: Chat,

    #[serde(skip_serializing_if = "Option::is_none")]
    forward_from: Option<User>,

    #[serde(skip_serializing_if = "Option::is_none")]
    forward_from_chat: Option<Chat>,

    #[serde(skip_serializing_if = "Option::is_none")]
    forward_from_message_id: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    forward_signature: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    forward_sender_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    forward_date: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_to_message: Option<Box<Message>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    via_bot: Option<User>,

    #[serde(skip_serializing_if = "Option::is_none")]
    edit_date: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    media_group_id: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    author_signature: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    text: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    animation: Option<Animation>,

    #[serde(skip_serializing_if = "Option::is_none")]
    audio: Option<Audio>,

    #[serde(skip_serializing_if = "Option::is_none")]
    document: Option<Document>,

    #[serde(skip_serializing_if = "Option::is_none")]
    photo: Option<Vec<PhotoSize>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    sticker: Option<Sticker>,

    #[serde(skip_serializing_if = "Option::is_none")]
    video: Option<Video>,

    #[serde(skip_serializing_if = "Option::is_none")]
    video_note: Option<VideoNote>,

    #[serde(skip_serializing_if = "Option::is_none")]
    voice: Option<Voice>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    contact: Option<Contact>,

    #[serde(skip_serializing_if = "Option::is_none")]
    dice: Option<Dice>,

    #[serde(skip_serializing_if = "Option::is_none")]
    game: Option<Game>,

    #[serde(skip_serializing_if = "Option::is_none")]
    poll: Option<Poll>,

    #[serde(skip_serializing_if = "Option::is_none")]
    venue: Option<Venue>,

    #[serde(skip_serializing_if = "Option::is_none")]
    location: Option<Location>,

    #[serde(skip_serializing_if = "Option::is_none")]
    new_chat_members: Option<Vec<User>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    left_chat_member: Option<User>,

    #[serde(skip_serializing_if = "Option::is_none")]
    new_chat_title: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    new_chat_photo: Option<Vec<PhotoSize>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    delete_chat_photo: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    group_chat_created: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    supergroup_chat_created: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    channel_chat_created: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    message_auto_delete_timer_changed: Option<MessageAutoDeleteTimerChanged>,

    #[serde(skip_serializing_if = "Option::is_none")]
    migrate_to_chat_id: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    migrate_from_chat_id: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pinned_message: Option<Box<Message>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    invoice: Option<Invoice>,

    #[serde(skip_serializing_if = "Option::is_none")]
    successful_payment: Option<SuccessfulPayment>,

    #[serde(skip_serializing_if = "Option::is_none")]
    connected_website: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    passport_data: Option<PassportData>,

    #[serde(skip_serializing_if = "Option::is_none")]
    proximity_alert_triggered: Option<ProximityAlertTriggered>,

    #[serde(skip_serializing_if = "Option::is_none")]
    voice_chat_started: Option<VoiceChatStarted>,

    #[serde(skip_serializing_if = "Option::is_none")]
    voice_chat_ended: Option<VoiceChatEnded>,

    #[serde(skip_serializing_if = "Option::is_none")]
    voice_chat_scheduled: Option<VoiceChatScheduled>,

    #[serde(skip_serializing_if = "Option::is_none")]
    voice_chat_participants_invited: Option<VoiceChatParticipantsInvited>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct MessageId {
    message_id: isize,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct MessageEntity {
    #[serde(rename = "type")]
    type_field: String,

    offset: isize,

    length: isize,

    #[serde(skip_serializing_if = "Option::is_none")]
    url: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    user: Option<User>,

    #[serde(skip_serializing_if = "Option::is_none")]
    language: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PhotoSize {
    file_id: String,

    file_unique_id: String,

    width: isize,

    height: isize,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_size: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Animation {
    file_id: String,

    file_unique_id: String,

    width: isize,

    height: isize,

    duration: isize,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb: Option<PhotoSize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    mime_type: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_size: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Audio {
    file_id: String,

    file_unique_id: String,

    duration: isize,

    #[serde(skip_serializing_if = "Option::is_none")]
    performer: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    title: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    mime_type: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_size: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb: Option<PhotoSize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Document {
    file_id: String,

    file_unique_id: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb: Option<PhotoSize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    mime_type: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_size: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Video {
    file_id: String,

    file_unique_id: String,

    width: isize,

    height: isize,

    duration: isize,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb: Option<PhotoSize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    mime_type: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_size: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct VideoNote {
    file_id: String,

    file_unique_id: String,

    length: isize,

    duration: isize,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb: Option<PhotoSize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_size: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Voice {
    file_id: String,

    file_unique_id: String,

    duration: isize,

    #[serde(skip_serializing_if = "Option::is_none")]
    mime_type: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_size: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Contact {
    phone_number: String,

    first_name: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    last_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    user_id: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    vcard: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Dice {
    emoji: String,

    value: isize,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PollOption {
    text: String,

    voter_count: isize,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PollAnswer {
    poll_id: String,

    user: User,

    option_ids: Vec<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Poll {
    id: String,

    question: String,

    options: Vec<PollOption>,

    total_voter_count: isize,

    is_closed: bool,

    is_anonymous: bool,
    #[serde(rename = "type")]
    type_field: String,

    allows_multiple_answers: bool,

    #[serde(skip_serializing_if = "Option::is_none")]
    correct_option_id: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    explanation: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    explanation_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    open_period: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    close_date: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Location {
    longitude: f64,

    latitude: f64,

    #[serde(skip_serializing_if = "Option::is_none")]
    horizontal_accuracy: Option<f64>,

    #[serde(skip_serializing_if = "Option::is_none")]
    live_period: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    heading: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    proximity_alert_radius: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Venue {
    location: Location,

    title: String,

    address: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    foursquare_id: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    foursquare_type: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    google_place_id: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    google_place_type: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ProximityAlertTriggered {
    traveler: User,

    watcher: User,

    distance: isize,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct MessageAutoDeleteTimerChanged {
    message_auto_delete_time: isize,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct VoiceChatEnded {
    duration: isize,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct VoiceChatParticipantsInvited {
    #[serde(skip_serializing_if = "Option::is_none")]
    users: Option<Vec<User>>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct UserProfilePhotos {
    total_count: isize,

    photos: Vec<Vec<PhotoSize>>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct File {
    file_id: String,

    file_unique_id: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_size: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_path: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ReplyKeyboardMarkup {
    keyboard: Vec<KeyboardButton>,

    #[serde(skip_serializing_if = "Option::is_none")]
    resize_keyboard: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    one_time_keyboard: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    selective: Option<bool>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct KeyboardButton {
    text: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    request_contact: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    request_location: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    request_poll: Option<KeyboardButtonPollType>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct KeyboardButtonPollType {
    #[serde(rename = "type")]
    #[serde(skip_serializing_if = "Option::is_none")]
    type_field: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ReplyKeyboardRemove {
    remove_keyboard: bool,

    #[serde(skip_serializing_if = "Option::is_none")]
    selective: Option<bool>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineKeyboardMarkup {
    inline_keyboard: Vec<InlineKeyboardButton>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineKeyboardButton {
    text: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    url: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    login_url: Option<LoginUrl>,

    #[serde(skip_serializing_if = "Option::is_none")]
    callback_data: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    switch_inline_query: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    switch_inline_query_current_chat: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    callback_game: Option<CallbackGame>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pay: Option<bool>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct LoginUrl {
    url: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    forward_text: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    bot_username: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    request_write_access: Option<bool>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct CallbackQuery {
    id: String,

    from: User,

    #[serde(skip_serializing_if = "Option::is_none")]
    message: Option<Message>,

    #[serde(skip_serializing_if = "Option::is_none")]
    inline_message_id: Option<String>,

    chat_instance: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    data: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    game_short_name: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ForceReply {
    force_reply: bool,

    #[serde(skip_serializing_if = "Option::is_none")]
    selective: Option<bool>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ChatPhoto {
    small_file_id: String,

    small_file_unique_id: String,

    big_file_id: String,

    big_file_unique_id: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ChatInviteLink {
    invite_link: String,

    creator: User,

    is_primary: bool,

    is_revoked: bool,

    #[serde(skip_serializing_if = "Option::is_none")]
    expire_date: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    member_limit: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ChatMember {
    user: User,

    status: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    custom_title: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    is_anonymous: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_be_edited: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_manage_chat: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_post_messages: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_edit_messages: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_delete_messages: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_manage_voice_chats: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_restrict_members: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_promote_members: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_change_info: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_invite_users: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_pin_messages: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    is_member: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_send_messages: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_send_media_messages: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_send_polls: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_send_other_messages: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_add_web_page_previews: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    until_date: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ChatMemberUpdated {
    chat: Chat,

    from: User,

    date: isize,

    old_chat_member: ChatMember,

    new_chat_member: ChatMember,

    #[serde(skip_serializing_if = "Option::is_none")]
    invite_link: Option<ChatInviteLink>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ChatPermissions {
    #[serde(skip_serializing_if = "Option::is_none")]
    can_send_messages: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_send_media_messages: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_send_polls: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_send_other_messages: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_add_web_page_previews: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_change_info: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_invite_users: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    can_pin_messages: Option<bool>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ChatLocation {
    location: Location,

    address: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct BotCommand {
    command: String,

    description: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ResponseParameters {
    #[serde(skip_serializing_if = "Option::is_none")]
    migrate_to_chat_id: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    retry_after: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Sticker {
    file_id: String,

    file_unique_id: String,

    width: isize,

    height: isize,

    is_animated: bool,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb: Option<PhotoSize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    emoji: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    set_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    mask_position: Option<MaskPosition>,

    #[serde(skip_serializing_if = "Option::is_none")]
    file_size: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct StickerSet {
    name: String,

    title: String,

    is_animated: bool,

    contains_masks: bool,

    stickers: Vec<Sticker>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb: Option<PhotoSize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct MaskPosition {
    point: String,

    x_shift: f64,

    y_shift: f64,

    scale: f64,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQuery {
    id: String,

    from: User,

    #[serde(skip_serializing_if = "Option::is_none")]
    location: Option<Location>,

    #[serde(skip_serializing_if = "Option::is_none")]
    chat_type: Option<String>,

    query: String,

    offset: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultArticle {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    title: String,

    input_message_content: InputMessageContent,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    url: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    hide_url: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    description: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_url: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_width: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_height: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultPhoto {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    photo_url: String,

    thumb_url: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    photo_width: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    photo_height: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    title: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    description: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultGif {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    gif_url: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    gif_width: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    gif_height: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    gif_duration: Option<isize>,

    thumb_url: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_mime_type: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    title: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultMpeg4Gif {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    mpeg4_url: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    mpeg4_width: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    mpeg4_height: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    mpeg4_duration: Option<isize>,

    thumb_url: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_mime_type: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    title: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultVideo {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    video_url: String,

    mime_type: String,

    thumb_url: String,

    title: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    video_width: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    video_height: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    video_duration: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    description: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultAudio {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    audio_url: String,

    title: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    performer: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    audio_duration: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultVoice {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    voice_url: String,

    title: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    voice_duration: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultDocument {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    title: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    document_url: String,

    mime_type: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    description: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_url: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_width: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_height: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultLocation {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    latitude: f64,

    longitude: f64,

    title: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    horizontal_accuracy: Option<f64>,

    #[serde(skip_serializing_if = "Option::is_none")]
    live_period: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    heading: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    proximity_alert_radius: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_url: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_width: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_height: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultVenue {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    latitude: f64,

    longitude: f64,

    title: String,

    address: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    foursquare_id: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    foursquare_type: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    google_place_id: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    google_place_type: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_url: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_width: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_height: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultContact {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    phone_number: String,

    first_name: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    last_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    vcard: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_url: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_width: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    thumb_height: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultGame {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    game_short_name: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultCachedPhoto {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    photo_file_id: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    title: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    description: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultCachedGif {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    gif_file_id: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    title: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultCachedMpeg4Gif {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    mpeg4_file_id: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    title: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultCachedSticker {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    sticker_file_id: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultCachedDocument {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    title: String,

    document_file_id: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    description: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultCachedVideo {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    video_file_id: String,

    title: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    description: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultCachedVoice {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    voice_file_id: String,

    title: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InlineQueryResultCachedAudio {
    #[serde(rename = "type")]
    type_field: String,

    id: String,

    audio_file_id: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    caption_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reply_markup: Option<InlineKeyboardMarkup>,

    #[serde(skip_serializing_if = "Option::is_none")]
    input_message_content: Option<InputMessageContent>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InputTextMessageContent {
    message_text: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    parse_mode: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    disable_web_page_preview: Option<bool>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InputLocationMessageContent {
    latitude: f64,

    longitude: f64,

    #[serde(skip_serializing_if = "Option::is_none")]
    horizontal_accuracy: Option<f64>,

    #[serde(skip_serializing_if = "Option::is_none")]
    live_period: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    heading: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    proximity_alert_radius: Option<isize>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InputInvoiceMessageContent {
    title: String,

    description: String,

    payload: String,

    provider_token: String,

    currency: String,

    prices: Vec<LabeledPrice>,

    #[serde(skip_serializing_if = "Option::is_none")]
    max_tip_amount: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    suggested_tip_amounts: Option<Vec<isize>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    provider_data: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    photo_url: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    photo_size: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    photo_width: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    photo_height: Option<isize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    need_name: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    need_phone_number: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    need_email: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    need_shipping_address: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    send_phone_number_to_provider: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    send_email_to_provider: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    is_flexible: Option<bool>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InputVenueMessageContent {
    latitude: f64,

    longitude: f64,

    title: String,

    address: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    foursquare_id: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    foursquare_type: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    google_place_id: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    google_place_type: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct InputContactMessageContent {
    phone_number: String,

    first_name: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    last_name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    vcard: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ChosenInlineResult {
    result_id: String,

    from: User,

    #[serde(skip_serializing_if = "Option::is_none")]
    location: Option<Location>,

    #[serde(skip_serializing_if = "Option::is_none")]
    inline_message_id: Option<String>,

    query: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct LabeledPrice {
    label: String,

    amount: isize,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Invoice {
    title: String,

    description: String,

    start_parameter: String,

    currency: String,

    total_amount: isize,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ShippingAddress {
    country_code: String,

    state: String,

    city: String,

    street_line1: String,

    street_line2: String,

    post_code: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct OrderInfo {
    #[serde(skip_serializing_if = "Option::is_none")]
    name: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    phone_number: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    email: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    shipping_address: Option<ShippingAddress>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ShippingOption {
    id: String,

    title: String,

    prices: Vec<LabeledPrice>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct SuccessfulPayment {
    currency: String,

    total_amount: isize,

    invoice_payload: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    shipping_option_id: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    order_info: Option<OrderInfo>,

    telegram_payment_charge_id: String,

    provider_payment_charge_id: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct ShippingQuery {
    id: String,

    from: User,

    invoice_payload: String,

    shipping_address: ShippingAddress,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PreCheckoutQuery {
    id: String,

    from: User,

    currency: String,

    total_amount: isize,

    invoice_payload: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    shipping_option_id: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    order_info: Option<OrderInfo>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PassportData {
    data: Vec<EncryptedPassportElement>,

    credentials: EncryptedCredentials,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PassportFile {
    file_id: String,

    file_unique_id: String,

    file_size: isize,

    file_date: isize,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct EncryptedPassportElement {
    #[serde(rename = "type")]
    type_field: String,

    #[serde(skip_serializing_if = "Option::is_none")]
    data: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    phone_number: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    email: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    files: Option<Vec<PassportFile>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    front_side: Option<PassportFile>,

    #[serde(skip_serializing_if = "Option::is_none")]
    reverse_side: Option<PassportFile>,

    #[serde(skip_serializing_if = "Option::is_none")]
    selfie: Option<PassportFile>,

    #[serde(skip_serializing_if = "Option::is_none")]
    translation: Option<Vec<PassportFile>>,

    hash: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct EncryptedCredentials {
    data: String,

    hash: String,

    secret: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PassportElementErrorDataField {
    source: String,
    #[serde(rename = "type")]
    type_field: String,

    field_name: String,

    data_hash: String,

    message: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PassportElementErrorFrontSide {
    source: String,
    #[serde(rename = "type")]
    type_field: String,

    file_hash: String,

    message: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PassportElementErrorReverseSide {
    source: String,
    #[serde(rename = "type")]
    type_field: String,

    file_hash: String,

    message: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PassportElementErrorSelfie {
    source: String,
    #[serde(rename = "type")]
    type_field: String,

    file_hash: String,

    message: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PassportElementErrorFile {
    source: String,
    #[serde(rename = "type")]
    type_field: String,

    file_hash: String,

    message: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PassportElementErrorFiles {
    source: String,
    #[serde(rename = "type")]
    type_field: String,

    file_hashes: Vec<String>,

    message: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PassportElementErrorTranslationFile {
    source: String,
    #[serde(rename = "type")]
    type_field: String,

    file_hash: String,

    message: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PassportElementErrorTranslationFiles {
    source: String,
    #[serde(rename = "type")]
    type_field: String,

    file_hashes: Vec<String>,

    message: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct PassportElementErrorUnspecified {
    source: String,
    #[serde(rename = "type")]
    type_field: String,

    element_hash: String,

    message: String,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Game {
    title: String,

    description: String,

    photo: Vec<PhotoSize>,

    #[serde(skip_serializing_if = "Option::is_none")]
    text: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    text_entities: Option<Vec<MessageEntity>>,

    #[serde(skip_serializing_if = "Option::is_none")]
    animation: Option<Animation>,
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct GameHighScore {
    position: isize,

    user: User,

    score: isize,
}

impl Update {
    pub fn new(update_id: isize) -> Self {
        Self {
            update_id,
            message: None,
            edited_message: None,
            channel_post: None,
            edited_channel_post: None,
            inline_query: None,
            chosen_inline_result: None,
            callback_query: None,
            shipping_query: None,
            pre_checkout_query: None,
            poll: None,
            poll_answer: None,
            my_chat_member: None,
            chat_member: None,
        }
    }

    pub fn set_update_id(&mut self, update_id: isize) {
        self.update_id = update_id;
    }

    pub fn set_message(&mut self, message: Option<Message>) {
        self.message = message;
    }

    pub fn set_edited_message(&mut self, edited_message: Option<Message>) {
        self.edited_message = edited_message;
    }

    pub fn set_channel_post(&mut self, channel_post: Option<Message>) {
        self.channel_post = channel_post;
    }

    pub fn set_edited_channel_post(&mut self, edited_channel_post: Option<Message>) {
        self.edited_channel_post = edited_channel_post;
    }

    pub fn set_inline_query(&mut self, inline_query: Option<InlineQuery>) {
        self.inline_query = inline_query;
    }

    pub fn set_chosen_inline_result(&mut self, chosen_inline_result: Option<ChosenInlineResult>) {
        self.chosen_inline_result = chosen_inline_result;
    }

    pub fn set_callback_query(&mut self, callback_query: Option<CallbackQuery>) {
        self.callback_query = callback_query;
    }

    pub fn set_shipping_query(&mut self, shipping_query: Option<ShippingQuery>) {
        self.shipping_query = shipping_query;
    }

    pub fn set_pre_checkout_query(&mut self, pre_checkout_query: Option<PreCheckoutQuery>) {
        self.pre_checkout_query = pre_checkout_query;
    }

    pub fn set_poll(&mut self, poll: Option<Poll>) {
        self.poll = poll;
    }

    pub fn set_poll_answer(&mut self, poll_answer: Option<PollAnswer>) {
        self.poll_answer = poll_answer;
    }

    pub fn set_my_chat_member(&mut self, my_chat_member: Option<ChatMemberUpdated>) {
        self.my_chat_member = my_chat_member;
    }

    pub fn set_chat_member(&mut self, chat_member: Option<ChatMemberUpdated>) {
        self.chat_member = chat_member;
    }

    pub fn update_id(&self) -> isize {
        self.update_id
    }

    pub fn message(&self) -> Option<Message> {
        self.message.clone()
    }

    pub fn edited_message(&self) -> Option<Message> {
        self.edited_message.clone()
    }

    pub fn channel_post(&self) -> Option<Message> {
        self.channel_post.clone()
    }

    pub fn edited_channel_post(&self) -> Option<Message> {
        self.edited_channel_post.clone()
    }

    pub fn inline_query(&self) -> Option<InlineQuery> {
        self.inline_query.clone()
    }

    pub fn chosen_inline_result(&self) -> Option<ChosenInlineResult> {
        self.chosen_inline_result.clone()
    }

    pub fn callback_query(&self) -> Option<CallbackQuery> {
        self.callback_query.clone()
    }

    pub fn shipping_query(&self) -> Option<ShippingQuery> {
        self.shipping_query.clone()
    }

    pub fn pre_checkout_query(&self) -> Option<PreCheckoutQuery> {
        self.pre_checkout_query.clone()
    }

    pub fn poll(&self) -> Option<Poll> {
        self.poll.clone()
    }

    pub fn poll_answer(&self) -> Option<PollAnswer> {
        self.poll_answer.clone()
    }

    pub fn my_chat_member(&self) -> Option<ChatMemberUpdated> {
        self.my_chat_member.clone()
    }

    pub fn chat_member(&self) -> Option<ChatMemberUpdated> {
        self.chat_member.clone()
    }
}

impl WebhookInfo {
    pub fn new(url: String, has_custom_certificate: bool, pending_update_count: isize) -> Self {
        Self {
            url,
            has_custom_certificate,
            pending_update_count,
            ip_address: None,
            last_error_date: None,
            last_error_message: None,
            max_connections: None,
            allowed_updates: None,
        }
    }

    pub fn set_url(&mut self, url: String) {
        self.url = url;
    }

    pub fn set_has_custom_certificate(&mut self, has_custom_certificate: bool) {
        self.has_custom_certificate = has_custom_certificate;
    }

    pub fn set_pending_update_count(&mut self, pending_update_count: isize) {
        self.pending_update_count = pending_update_count;
    }

    pub fn set_ip_address(&mut self, ip_address: Option<String>) {
        self.ip_address = ip_address;
    }

    pub fn set_last_error_date(&mut self, last_error_date: Option<isize>) {
        self.last_error_date = last_error_date;
    }

    pub fn set_last_error_message(&mut self, last_error_message: Option<String>) {
        self.last_error_message = last_error_message;
    }

    pub fn set_max_connections(&mut self, max_connections: Option<isize>) {
        self.max_connections = max_connections;
    }

    pub fn set_allowed_updates(&mut self, allowed_updates: Option<Vec<String>>) {
        self.allowed_updates = allowed_updates;
    }

    pub fn url(&self) -> String {
        self.url.clone()
    }

    pub fn has_custom_certificate(&self) -> bool {
        self.has_custom_certificate
    }

    pub fn pending_update_count(&self) -> isize {
        self.pending_update_count
    }

    pub fn ip_address(&self) -> Option<String> {
        self.ip_address.clone()
    }

    pub fn last_error_date(&self) -> Option<isize> {
        self.last_error_date
    }

    pub fn last_error_message(&self) -> Option<String> {
        self.last_error_message.clone()
    }

    pub fn max_connections(&self) -> Option<isize> {
        self.max_connections
    }

    pub fn allowed_updates(&self) -> Option<Vec<String>> {
        self.allowed_updates.clone()
    }
}

impl User {
    pub fn new(id: isize, is_bot: bool, first_name: String) -> Self {
        Self {
            id,
            is_bot,
            first_name,
            last_name: None,
            username: None,
            language_code: None,
            can_join_groups: None,
            can_read_all_group_messages: None,
            supports_inline_queries: None,
        }
    }

    pub fn set_id(&mut self, id: isize) {
        self.id = id;
    }

    pub fn set_is_bot(&mut self, is_bot: bool) {
        self.is_bot = is_bot;
    }

    pub fn set_first_name(&mut self, first_name: String) {
        self.first_name = first_name;
    }

    pub fn set_last_name(&mut self, last_name: Option<String>) {
        self.last_name = last_name;
    }

    pub fn set_username(&mut self, username: Option<String>) {
        self.username = username;
    }

    pub fn set_language_code(&mut self, language_code: Option<String>) {
        self.language_code = language_code;
    }

    pub fn set_can_join_groups(&mut self, can_join_groups: Option<bool>) {
        self.can_join_groups = can_join_groups;
    }

    pub fn set_can_read_all_group_messages(&mut self, can_read_all_group_messages: Option<bool>) {
        self.can_read_all_group_messages = can_read_all_group_messages;
    }

    pub fn set_supports_inline_queries(&mut self, supports_inline_queries: Option<bool>) {
        self.supports_inline_queries = supports_inline_queries;
    }

    pub fn id(&self) -> isize {
        self.id
    }

    pub fn is_bot(&self) -> bool {
        self.is_bot
    }

    pub fn first_name(&self) -> String {
        self.first_name.clone()
    }

    pub fn last_name(&self) -> Option<String> {
        self.last_name.clone()
    }

    pub fn username(&self) -> Option<String> {
        self.username.clone()
    }

    pub fn language_code(&self) -> Option<String> {
        self.language_code.clone()
    }

    pub fn can_join_groups(&self) -> Option<bool> {
        self.can_join_groups
    }

    pub fn can_read_all_group_messages(&self) -> Option<bool> {
        self.can_read_all_group_messages
    }

    pub fn supports_inline_queries(&self) -> Option<bool> {
        self.supports_inline_queries
    }
}

impl Chat {
    pub fn new(id: isize, type_field: String) -> Self {
        Self {
            id,
            type_field,
            title: None,
            username: None,
            first_name: None,
            last_name: None,
            photo: None,
            bio: None,
            description: None,
            invite_link: None,
            pinned_message: None,
            permissions: None,
            slow_mode_delay: None,
            message_auto_delete_time: None,
            sticker_set_name: None,
            can_set_sticker_set: None,
            linked_chat_id: None,
            location: None,
        }
    }

    pub fn set_id(&mut self, id: isize) {
        self.id = id;
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_title(&mut self, title: Option<String>) {
        self.title = title;
    }

    pub fn set_username(&mut self, username: Option<String>) {
        self.username = username;
    }

    pub fn set_first_name(&mut self, first_name: Option<String>) {
        self.first_name = first_name;
    }

    pub fn set_last_name(&mut self, last_name: Option<String>) {
        self.last_name = last_name;
    }

    pub fn set_photo(&mut self, photo: Option<ChatPhoto>) {
        self.photo = photo;
    }

    pub fn set_bio(&mut self, bio: Option<String>) {
        self.bio = bio;
    }

    pub fn set_description(&mut self, description: Option<String>) {
        self.description = description;
    }

    pub fn set_invite_link(&mut self, invite_link: Option<String>) {
        self.invite_link = invite_link;
    }

    pub fn set_pinned_message(&mut self, pinned_message: Option<Box<Message>>) {
        self.pinned_message = pinned_message;
    }

    pub fn set_permissions(&mut self, permissions: Option<ChatPermissions>) {
        self.permissions = permissions;
    }

    pub fn set_slow_mode_delay(&mut self, slow_mode_delay: Option<isize>) {
        self.slow_mode_delay = slow_mode_delay;
    }

    pub fn set_message_auto_delete_time(&mut self, message_auto_delete_time: Option<isize>) {
        self.message_auto_delete_time = message_auto_delete_time;
    }

    pub fn set_sticker_set_name(&mut self, sticker_set_name: Option<String>) {
        self.sticker_set_name = sticker_set_name;
    }

    pub fn set_can_set_sticker_set(&mut self, can_set_sticker_set: Option<bool>) {
        self.can_set_sticker_set = can_set_sticker_set;
    }

    pub fn set_linked_chat_id(&mut self, linked_chat_id: Option<isize>) {
        self.linked_chat_id = linked_chat_id;
    }

    pub fn set_location(&mut self, location: Option<ChatLocation>) {
        self.location = location;
    }

    pub fn id(&self) -> isize {
        self.id
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn title(&self) -> Option<String> {
        self.title.clone()
    }

    pub fn username(&self) -> Option<String> {
        self.username.clone()
    }

    pub fn first_name(&self) -> Option<String> {
        self.first_name.clone()
    }

    pub fn last_name(&self) -> Option<String> {
        self.last_name.clone()
    }

    pub fn photo(&self) -> Option<ChatPhoto> {
        self.photo.clone()
    }

    pub fn bio(&self) -> Option<String> {
        self.bio.clone()
    }

    pub fn description(&self) -> Option<String> {
        self.description.clone()
    }

    pub fn invite_link(&self) -> Option<String> {
        self.invite_link.clone()
    }

    pub fn pinned_message(&self) -> Option<Box<Message>> {
        self.pinned_message.clone()
    }

    pub fn permissions(&self) -> Option<ChatPermissions> {
        self.permissions.clone()
    }

    pub fn slow_mode_delay(&self) -> Option<isize> {
        self.slow_mode_delay
    }

    pub fn message_auto_delete_time(&self) -> Option<isize> {
        self.message_auto_delete_time
    }

    pub fn sticker_set_name(&self) -> Option<String> {
        self.sticker_set_name.clone()
    }

    pub fn can_set_sticker_set(&self) -> Option<bool> {
        self.can_set_sticker_set
    }

    pub fn linked_chat_id(&self) -> Option<isize> {
        self.linked_chat_id
    }

    pub fn location(&self) -> Option<ChatLocation> {
        self.location.clone()
    }
}

impl Message {
    pub fn new(message_id: isize, date: isize, chat: Chat) -> Self {
        Self {
            message_id,
            date,
            chat,
            from: None,
            sender_chat: None,
            forward_from: None,
            forward_from_chat: None,
            forward_from_message_id: None,
            forward_signature: None,
            forward_sender_name: None,
            forward_date: None,
            reply_to_message: None,
            via_bot: None,
            edit_date: None,
            media_group_id: None,
            author_signature: None,
            text: None,
            entities: None,
            animation: None,
            audio: None,
            document: None,
            photo: None,
            sticker: None,
            video: None,
            video_note: None,
            voice: None,
            caption: None,
            caption_entities: None,
            contact: None,
            dice: None,
            game: None,
            poll: None,
            venue: None,
            location: None,
            new_chat_members: None,
            left_chat_member: None,
            new_chat_title: None,
            new_chat_photo: None,
            delete_chat_photo: None,
            group_chat_created: None,
            supergroup_chat_created: None,
            channel_chat_created: None,
            message_auto_delete_timer_changed: None,
            migrate_to_chat_id: None,
            migrate_from_chat_id: None,
            pinned_message: None,
            invoice: None,
            successful_payment: None,
            connected_website: None,
            passport_data: None,
            proximity_alert_triggered: None,
            voice_chat_started: None,
            voice_chat_ended: None,
            voice_chat_scheduled: None,
            voice_chat_participants_invited: None,
            reply_markup: None,
        }
    }

    pub fn set_message_id(&mut self, message_id: isize) {
        self.message_id = message_id;
    }

    pub fn set_date(&mut self, date: isize) {
        self.date = date;
    }

    pub fn set_chat(&mut self, chat: Chat) {
        self.chat = chat;
    }

    pub fn set_from(&mut self, from: Option<User>) {
        self.from = from;
    }

    pub fn set_sender_chat(&mut self, sender_chat: Option<Chat>) {
        self.sender_chat = sender_chat;
    }

    pub fn set_forward_from(&mut self, forward_from: Option<User>) {
        self.forward_from = forward_from;
    }

    pub fn set_forward_from_chat(&mut self, forward_from_chat: Option<Chat>) {
        self.forward_from_chat = forward_from_chat;
    }

    pub fn set_forward_from_message_id(&mut self, forward_from_message_id: Option<isize>) {
        self.forward_from_message_id = forward_from_message_id;
    }

    pub fn set_forward_signature(&mut self, forward_signature: Option<String>) {
        self.forward_signature = forward_signature;
    }

    pub fn set_forward_sender_name(&mut self, forward_sender_name: Option<String>) {
        self.forward_sender_name = forward_sender_name;
    }

    pub fn set_forward_date(&mut self, forward_date: Option<isize>) {
        self.forward_date = forward_date;
    }

    pub fn set_reply_to_message(&mut self, reply_to_message: Option<Box<Message>>) {
        self.reply_to_message = reply_to_message;
    }

    pub fn set_via_bot(&mut self, via_bot: Option<User>) {
        self.via_bot = via_bot;
    }

    pub fn set_edit_date(&mut self, edit_date: Option<isize>) {
        self.edit_date = edit_date;
    }

    pub fn set_media_group_id(&mut self, media_group_id: Option<String>) {
        self.media_group_id = media_group_id;
    }

    pub fn set_author_signature(&mut self, author_signature: Option<String>) {
        self.author_signature = author_signature;
    }

    pub fn set_text(&mut self, text: Option<String>) {
        self.text = text;
    }

    pub fn set_entities(&mut self, entities: Option<Vec<MessageEntity>>) {
        self.entities = entities;
    }

    pub fn set_animation(&mut self, animation: Option<Animation>) {
        self.animation = animation;
    }

    pub fn set_audio(&mut self, audio: Option<Audio>) {
        self.audio = audio;
    }

    pub fn set_document(&mut self, document: Option<Document>) {
        self.document = document;
    }

    pub fn set_photo(&mut self, photo: Option<Vec<PhotoSize>>) {
        self.photo = photo;
    }

    pub fn set_sticker(&mut self, sticker: Option<Sticker>) {
        self.sticker = sticker;
    }

    pub fn set_video(&mut self, video: Option<Video>) {
        self.video = video;
    }

    pub fn set_video_note(&mut self, video_note: Option<VideoNote>) {
        self.video_note = video_note;
    }

    pub fn set_voice(&mut self, voice: Option<Voice>) {
        self.voice = voice;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_contact(&mut self, contact: Option<Contact>) {
        self.contact = contact;
    }

    pub fn set_dice(&mut self, dice: Option<Dice>) {
        self.dice = dice;
    }

    pub fn set_game(&mut self, game: Option<Game>) {
        self.game = game;
    }

    pub fn set_poll(&mut self, poll: Option<Poll>) {
        self.poll = poll;
    }

    pub fn set_venue(&mut self, venue: Option<Venue>) {
        self.venue = venue;
    }

    pub fn set_location(&mut self, location: Option<Location>) {
        self.location = location;
    }

    pub fn set_new_chat_members(&mut self, new_chat_members: Option<Vec<User>>) {
        self.new_chat_members = new_chat_members;
    }

    pub fn set_left_chat_member(&mut self, left_chat_member: Option<User>) {
        self.left_chat_member = left_chat_member;
    }

    pub fn set_new_chat_title(&mut self, new_chat_title: Option<String>) {
        self.new_chat_title = new_chat_title;
    }

    pub fn set_new_chat_photo(&mut self, new_chat_photo: Option<Vec<PhotoSize>>) {
        self.new_chat_photo = new_chat_photo;
    }

    pub fn set_delete_chat_photo(&mut self, delete_chat_photo: Option<bool>) {
        self.delete_chat_photo = delete_chat_photo;
    }

    pub fn set_group_chat_created(&mut self, group_chat_created: Option<bool>) {
        self.group_chat_created = group_chat_created;
    }

    pub fn set_supergroup_chat_created(&mut self, supergroup_chat_created: Option<bool>) {
        self.supergroup_chat_created = supergroup_chat_created;
    }

    pub fn set_channel_chat_created(&mut self, channel_chat_created: Option<bool>) {
        self.channel_chat_created = channel_chat_created;
    }

    pub fn set_message_auto_delete_timer_changed(
        &mut self,
        message_auto_delete_timer_changed: Option<MessageAutoDeleteTimerChanged>,
    ) {
        self.message_auto_delete_timer_changed = message_auto_delete_timer_changed;
    }

    pub fn set_migrate_to_chat_id(&mut self, migrate_to_chat_id: Option<isize>) {
        self.migrate_to_chat_id = migrate_to_chat_id;
    }

    pub fn set_migrate_from_chat_id(&mut self, migrate_from_chat_id: Option<isize>) {
        self.migrate_from_chat_id = migrate_from_chat_id;
    }

    pub fn set_pinned_message(&mut self, pinned_message: Option<Box<Message>>) {
        self.pinned_message = pinned_message;
    }

    pub fn set_invoice(&mut self, invoice: Option<Invoice>) {
        self.invoice = invoice;
    }

    pub fn set_successful_payment(&mut self, successful_payment: Option<SuccessfulPayment>) {
        self.successful_payment = successful_payment;
    }

    pub fn set_connected_website(&mut self, connected_website: Option<String>) {
        self.connected_website = connected_website;
    }

    pub fn set_passport_data(&mut self, passport_data: Option<PassportData>) {
        self.passport_data = passport_data;
    }

    pub fn set_proximity_alert_triggered(
        &mut self,
        proximity_alert_triggered: Option<ProximityAlertTriggered>,
    ) {
        self.proximity_alert_triggered = proximity_alert_triggered;
    }

    pub fn set_voice_chat_started(&mut self, voice_chat_started: Option<VoiceChatStarted>) {
        self.voice_chat_started = voice_chat_started;
    }

    pub fn set_voice_chat_ended(&mut self, voice_chat_ended: Option<VoiceChatEnded>) {
        self.voice_chat_ended = voice_chat_ended;
    }

    pub fn set_voice_chat_participants_invited(
        &mut self,
        voice_chat_participants_invited: Option<VoiceChatParticipantsInvited>,
    ) {
        self.voice_chat_participants_invited = voice_chat_participants_invited;
    }

    pub fn set_voice_chat_scheduled(&mut self, voice_chat_scheduled: Option<VoiceChatScheduled>) {
        self.voice_chat_scheduled = voice_chat_scheduled;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn voice_chat_scheduled(&self) -> Option<VoiceChatScheduled> {
        self.voice_chat_scheduled.clone()
    }

    pub fn message_id(&self) -> isize {
        self.message_id
    }

    pub fn date(&self) -> isize {
        self.date
    }

    pub fn chat(&self) -> Chat {
        self.chat.clone()
    }

    pub fn from(&self) -> Option<User> {
        self.from.clone()
    }

    pub fn sender_chat(&self) -> Option<Chat> {
        self.sender_chat.clone()
    }

    pub fn forward_from(&self) -> Option<User> {
        self.forward_from.clone()
    }

    pub fn forward_from_chat(&self) -> Option<Chat> {
        self.forward_from_chat.clone()
    }

    pub fn forward_from_message_id(&self) -> Option<isize> {
        self.forward_from_message_id
    }

    pub fn forward_signature(&self) -> Option<String> {
        self.forward_signature.clone()
    }

    pub fn forward_sender_name(&self) -> Option<String> {
        self.forward_sender_name.clone()
    }

    pub fn forward_date(&self) -> Option<isize> {
        self.forward_date
    }

    pub fn reply_to_message(&self) -> Option<Box<Message>> {
        self.reply_to_message.clone()
    }

    pub fn via_bot(&self) -> Option<User> {
        self.via_bot.clone()
    }

    pub fn edit_date(&self) -> Option<isize> {
        self.edit_date
    }

    pub fn media_group_id(&self) -> Option<String> {
        self.media_group_id.clone()
    }

    pub fn author_signature(&self) -> Option<String> {
        self.author_signature.clone()
    }

    pub fn text(&self) -> Option<String> {
        self.text.clone()
    }

    pub fn entities(&self) -> Option<Vec<MessageEntity>> {
        self.entities.clone()
    }

    pub fn animation(&self) -> Option<Animation> {
        self.animation.clone()
    }

    pub fn audio(&self) -> Option<Audio> {
        self.audio.clone()
    }

    pub fn document(&self) -> Option<Document> {
        self.document.clone()
    }

    pub fn photo(&self) -> Option<Vec<PhotoSize>> {
        self.photo.clone()
    }

    pub fn sticker(&self) -> Option<Sticker> {
        self.sticker.clone()
    }

    pub fn video(&self) -> Option<Video> {
        self.video.clone()
    }

    pub fn video_note(&self) -> Option<VideoNote> {
        self.video_note.clone()
    }

    pub fn voice(&self) -> Option<Voice> {
        self.voice.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn contact(&self) -> Option<Contact> {
        self.contact.clone()
    }

    pub fn dice(&self) -> Option<Dice> {
        self.dice.clone()
    }

    pub fn game(&self) -> Option<Game> {
        self.game.clone()
    }

    pub fn poll(&self) -> Option<Poll> {
        self.poll.clone()
    }

    pub fn venue(&self) -> Option<Venue> {
        self.venue.clone()
    }

    pub fn location(&self) -> Option<Location> {
        self.location.clone()
    }

    pub fn new_chat_members(&self) -> Option<Vec<User>> {
        self.new_chat_members.clone()
    }

    pub fn left_chat_member(&self) -> Option<User> {
        self.left_chat_member.clone()
    }

    pub fn new_chat_title(&self) -> Option<String> {
        self.new_chat_title.clone()
    }

    pub fn new_chat_photo(&self) -> Option<Vec<PhotoSize>> {
        self.new_chat_photo.clone()
    }

    pub fn delete_chat_photo(&self) -> Option<bool> {
        self.delete_chat_photo
    }

    pub fn group_chat_created(&self) -> Option<bool> {
        self.group_chat_created
    }

    pub fn supergroup_chat_created(&self) -> Option<bool> {
        self.supergroup_chat_created
    }

    pub fn channel_chat_created(&self) -> Option<bool> {
        self.channel_chat_created
    }

    pub fn message_auto_delete_timer_changed(&self) -> Option<MessageAutoDeleteTimerChanged> {
        self.message_auto_delete_timer_changed.clone()
    }

    pub fn migrate_to_chat_id(&self) -> Option<isize> {
        self.migrate_to_chat_id
    }

    pub fn migrate_from_chat_id(&self) -> Option<isize> {
        self.migrate_from_chat_id
    }

    pub fn pinned_message(&self) -> Option<Box<Message>> {
        self.pinned_message.clone()
    }

    pub fn invoice(&self) -> Option<Invoice> {
        self.invoice.clone()
    }

    pub fn successful_payment(&self) -> Option<SuccessfulPayment> {
        self.successful_payment.clone()
    }

    pub fn connected_website(&self) -> Option<String> {
        self.connected_website.clone()
    }

    pub fn passport_data(&self) -> Option<PassportData> {
        self.passport_data.clone()
    }

    pub fn proximity_alert_triggered(&self) -> Option<ProximityAlertTriggered> {
        self.proximity_alert_triggered.clone()
    }

    pub fn voice_chat_started(&self) -> Option<VoiceChatStarted> {
        self.voice_chat_started.clone()
    }

    pub fn voice_chat_ended(&self) -> Option<VoiceChatEnded> {
        self.voice_chat_ended.clone()
    }

    pub fn voice_chat_participants_invited(&self) -> Option<VoiceChatParticipantsInvited> {
        self.voice_chat_participants_invited.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }
}

impl MessageId {
    pub fn new(message_id: isize) -> Self {
        Self { message_id }
    }

    pub fn set_message_id(&mut self, message_id: isize) {
        self.message_id = message_id;
    }

    pub fn message_id(&self) -> isize {
        self.message_id
    }
}

impl MessageEntity {
    pub fn new(type_field: String, offset: isize, length: isize) -> Self {
        Self {
            type_field,
            offset,
            length,
            url: None,
            user: None,
            language: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_offset(&mut self, offset: isize) {
        self.offset = offset;
    }

    pub fn set_length(&mut self, length: isize) {
        self.length = length;
    }

    pub fn set_url(&mut self, url: Option<String>) {
        self.url = url;
    }

    pub fn set_user(&mut self, user: Option<User>) {
        self.user = user;
    }

    pub fn set_language(&mut self, language: Option<String>) {
        self.language = language;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn offset(&self) -> isize {
        self.offset
    }

    pub fn length(&self) -> isize {
        self.length
    }

    pub fn url(&self) -> Option<String> {
        self.url.clone()
    }

    pub fn user(&self) -> Option<User> {
        self.user.clone()
    }

    pub fn language(&self) -> Option<String> {
        self.language.clone()
    }
}

impl PhotoSize {
    pub fn new(file_id: String, file_unique_id: String, width: isize, height: isize) -> Self {
        Self {
            file_id,
            file_unique_id,
            width,
            height,
            file_size: None,
        }
    }

    pub fn set_file_id(&mut self, file_id: String) {
        self.file_id = file_id;
    }

    pub fn set_file_unique_id(&mut self, file_unique_id: String) {
        self.file_unique_id = file_unique_id;
    }

    pub fn set_width(&mut self, width: isize) {
        self.width = width;
    }

    pub fn set_height(&mut self, height: isize) {
        self.height = height;
    }

    pub fn set_file_size(&mut self, file_size: Option<isize>) {
        self.file_size = file_size;
    }

    pub fn file_id(&self) -> String {
        self.file_id.clone()
    }

    pub fn file_unique_id(&self) -> String {
        self.file_unique_id.clone()
    }

    pub fn width(&self) -> isize {
        self.width
    }

    pub fn height(&self) -> isize {
        self.height
    }

    pub fn file_size(&self) -> Option<isize> {
        self.file_size
    }
}

impl Animation {
    pub fn new(
        file_id: String,
        file_unique_id: String,
        width: isize,
        height: isize,
        duration: isize,
    ) -> Self {
        Self {
            file_id,
            file_unique_id,
            width,
            height,
            duration,
            thumb: None,
            file_name: None,
            mime_type: None,
            file_size: None,
        }
    }

    pub fn set_file_id(&mut self, file_id: String) {
        self.file_id = file_id;
    }

    pub fn set_file_unique_id(&mut self, file_unique_id: String) {
        self.file_unique_id = file_unique_id;
    }

    pub fn set_width(&mut self, width: isize) {
        self.width = width;
    }

    pub fn set_height(&mut self, height: isize) {
        self.height = height;
    }

    pub fn set_duration(&mut self, duration: isize) {
        self.duration = duration;
    }

    pub fn set_thumb(&mut self, thumb: Option<PhotoSize>) {
        self.thumb = thumb;
    }

    pub fn set_file_name(&mut self, file_name: Option<String>) {
        self.file_name = file_name;
    }

    pub fn set_mime_type(&mut self, mime_type: Option<String>) {
        self.mime_type = mime_type;
    }

    pub fn set_file_size(&mut self, file_size: Option<isize>) {
        self.file_size = file_size;
    }

    pub fn file_id(&self) -> String {
        self.file_id.clone()
    }

    pub fn file_unique_id(&self) -> String {
        self.file_unique_id.clone()
    }

    pub fn width(&self) -> isize {
        self.width
    }

    pub fn height(&self) -> isize {
        self.height
    }

    pub fn duration(&self) -> isize {
        self.duration
    }

    pub fn thumb(&self) -> Option<PhotoSize> {
        self.thumb.clone()
    }

    pub fn file_name(&self) -> Option<String> {
        self.file_name.clone()
    }

    pub fn mime_type(&self) -> Option<String> {
        self.mime_type.clone()
    }

    pub fn file_size(&self) -> Option<isize> {
        self.file_size
    }
}

impl Audio {
    pub fn new(file_id: String, file_unique_id: String, duration: isize) -> Self {
        Self {
            file_id,
            file_unique_id,
            duration,
            performer: None,
            title: None,
            file_name: None,
            mime_type: None,
            file_size: None,
            thumb: None,
        }
    }

    pub fn set_file_id(&mut self, file_id: String) {
        self.file_id = file_id;
    }

    pub fn set_file_unique_id(&mut self, file_unique_id: String) {
        self.file_unique_id = file_unique_id;
    }

    pub fn set_duration(&mut self, duration: isize) {
        self.duration = duration;
    }

    pub fn set_performer(&mut self, performer: Option<String>) {
        self.performer = performer;
    }

    pub fn set_title(&mut self, title: Option<String>) {
        self.title = title;
    }

    pub fn set_file_name(&mut self, file_name: Option<String>) {
        self.file_name = file_name;
    }

    pub fn set_mime_type(&mut self, mime_type: Option<String>) {
        self.mime_type = mime_type;
    }

    pub fn set_file_size(&mut self, file_size: Option<isize>) {
        self.file_size = file_size;
    }

    pub fn set_thumb(&mut self, thumb: Option<PhotoSize>) {
        self.thumb = thumb;
    }

    pub fn file_id(&self) -> String {
        self.file_id.clone()
    }

    pub fn file_unique_id(&self) -> String {
        self.file_unique_id.clone()
    }

    pub fn duration(&self) -> isize {
        self.duration
    }

    pub fn performer(&self) -> Option<String> {
        self.performer.clone()
    }

    pub fn title(&self) -> Option<String> {
        self.title.clone()
    }

    pub fn file_name(&self) -> Option<String> {
        self.file_name.clone()
    }

    pub fn mime_type(&self) -> Option<String> {
        self.mime_type.clone()
    }

    pub fn file_size(&self) -> Option<isize> {
        self.file_size
    }

    pub fn thumb(&self) -> Option<PhotoSize> {
        self.thumb.clone()
    }
}

impl Document {
    pub fn new(file_id: String, file_unique_id: String) -> Self {
        Self {
            file_id,
            file_unique_id,
            thumb: None,
            file_name: None,
            mime_type: None,
            file_size: None,
        }
    }

    pub fn set_file_id(&mut self, file_id: String) {
        self.file_id = file_id;
    }

    pub fn set_file_unique_id(&mut self, file_unique_id: String) {
        self.file_unique_id = file_unique_id;
    }

    pub fn set_thumb(&mut self, thumb: Option<PhotoSize>) {
        self.thumb = thumb;
    }

    pub fn set_file_name(&mut self, file_name: Option<String>) {
        self.file_name = file_name;
    }

    pub fn set_mime_type(&mut self, mime_type: Option<String>) {
        self.mime_type = mime_type;
    }

    pub fn set_file_size(&mut self, file_size: Option<isize>) {
        self.file_size = file_size;
    }

    pub fn file_id(&self) -> String {
        self.file_id.clone()
    }

    pub fn file_unique_id(&self) -> String {
        self.file_unique_id.clone()
    }

    pub fn thumb(&self) -> Option<PhotoSize> {
        self.thumb.clone()
    }

    pub fn file_name(&self) -> Option<String> {
        self.file_name.clone()
    }

    pub fn mime_type(&self) -> Option<String> {
        self.mime_type.clone()
    }

    pub fn file_size(&self) -> Option<isize> {
        self.file_size
    }
}

impl Video {
    pub fn new(
        file_id: String,
        file_unique_id: String,
        width: isize,
        height: isize,
        duration: isize,
    ) -> Self {
        Self {
            file_id,
            file_unique_id,
            width,
            height,
            duration,
            thumb: None,
            file_name: None,
            mime_type: None,
            file_size: None,
        }
    }

    pub fn set_file_id(&mut self, file_id: String) {
        self.file_id = file_id;
    }

    pub fn set_file_unique_id(&mut self, file_unique_id: String) {
        self.file_unique_id = file_unique_id;
    }

    pub fn set_width(&mut self, width: isize) {
        self.width = width;
    }

    pub fn set_height(&mut self, height: isize) {
        self.height = height;
    }

    pub fn set_duration(&mut self, duration: isize) {
        self.duration = duration;
    }

    pub fn set_thumb(&mut self, thumb: Option<PhotoSize>) {
        self.thumb = thumb;
    }

    pub fn set_file_name(&mut self, file_name: Option<String>) {
        self.file_name = file_name;
    }

    pub fn set_mime_type(&mut self, mime_type: Option<String>) {
        self.mime_type = mime_type;
    }

    pub fn set_file_size(&mut self, file_size: Option<isize>) {
        self.file_size = file_size;
    }

    pub fn file_id(&self) -> String {
        self.file_id.clone()
    }

    pub fn file_unique_id(&self) -> String {
        self.file_unique_id.clone()
    }

    pub fn width(&self) -> isize {
        self.width
    }

    pub fn height(&self) -> isize {
        self.height
    }

    pub fn duration(&self) -> isize {
        self.duration
    }

    pub fn thumb(&self) -> Option<PhotoSize> {
        self.thumb.clone()
    }

    pub fn file_name(&self) -> Option<String> {
        self.file_name.clone()
    }

    pub fn mime_type(&self) -> Option<String> {
        self.mime_type.clone()
    }

    pub fn file_size(&self) -> Option<isize> {
        self.file_size
    }
}

impl VideoNote {
    pub fn new(file_id: String, file_unique_id: String, length: isize, duration: isize) -> Self {
        Self {
            file_id,
            file_unique_id,
            length,
            duration,
            thumb: None,
            file_size: None,
        }
    }

    pub fn set_file_id(&mut self, file_id: String) {
        self.file_id = file_id;
    }

    pub fn set_file_unique_id(&mut self, file_unique_id: String) {
        self.file_unique_id = file_unique_id;
    }

    pub fn set_length(&mut self, length: isize) {
        self.length = length;
    }

    pub fn set_duration(&mut self, duration: isize) {
        self.duration = duration;
    }

    pub fn set_thumb(&mut self, thumb: Option<PhotoSize>) {
        self.thumb = thumb;
    }

    pub fn set_file_size(&mut self, file_size: Option<isize>) {
        self.file_size = file_size;
    }

    pub fn file_id(&self) -> String {
        self.file_id.clone()
    }

    pub fn file_unique_id(&self) -> String {
        self.file_unique_id.clone()
    }

    pub fn length(&self) -> isize {
        self.length
    }

    pub fn duration(&self) -> isize {
        self.duration
    }

    pub fn thumb(&self) -> Option<PhotoSize> {
        self.thumb.clone()
    }

    pub fn file_size(&self) -> Option<isize> {
        self.file_size
    }
}

impl Voice {
    pub fn new(file_id: String, file_unique_id: String, duration: isize) -> Self {
        Self {
            file_id,
            file_unique_id,
            duration,
            mime_type: None,
            file_size: None,
        }
    }

    pub fn set_file_id(&mut self, file_id: String) {
        self.file_id = file_id;
    }

    pub fn set_file_unique_id(&mut self, file_unique_id: String) {
        self.file_unique_id = file_unique_id;
    }

    pub fn set_duration(&mut self, duration: isize) {
        self.duration = duration;
    }

    pub fn set_mime_type(&mut self, mime_type: Option<String>) {
        self.mime_type = mime_type;
    }

    pub fn set_file_size(&mut self, file_size: Option<isize>) {
        self.file_size = file_size;
    }

    pub fn file_id(&self) -> String {
        self.file_id.clone()
    }

    pub fn file_unique_id(&self) -> String {
        self.file_unique_id.clone()
    }

    pub fn duration(&self) -> isize {
        self.duration
    }

    pub fn mime_type(&self) -> Option<String> {
        self.mime_type.clone()
    }

    pub fn file_size(&self) -> Option<isize> {
        self.file_size
    }
}

impl Contact {
    pub fn new(phone_number: String, first_name: String) -> Self {
        Self {
            phone_number,
            first_name,
            last_name: None,
            user_id: None,
            vcard: None,
        }
    }

    pub fn set_phone_number(&mut self, phone_number: String) {
        self.phone_number = phone_number;
    }

    pub fn set_first_name(&mut self, first_name: String) {
        self.first_name = first_name;
    }

    pub fn set_last_name(&mut self, last_name: Option<String>) {
        self.last_name = last_name;
    }

    pub fn set_user_id(&mut self, user_id: Option<isize>) {
        self.user_id = user_id;
    }

    pub fn set_vcard(&mut self, vcard: Option<String>) {
        self.vcard = vcard;
    }

    pub fn phone_number(&self) -> String {
        self.phone_number.clone()
    }

    pub fn first_name(&self) -> String {
        self.first_name.clone()
    }

    pub fn last_name(&self) -> Option<String> {
        self.last_name.clone()
    }

    pub fn user_id(&self) -> Option<isize> {
        self.user_id
    }

    pub fn vcard(&self) -> Option<String> {
        self.vcard.clone()
    }
}

impl Dice {
    pub fn new(emoji: String, value: isize) -> Self {
        Self { emoji, value }
    }

    pub fn set_emoji(&mut self, emoji: String) {
        self.emoji = emoji;
    }

    pub fn set_value(&mut self, value: isize) {
        self.value = value;
    }

    pub fn emoji(&self) -> String {
        self.emoji.clone()
    }

    pub fn value(&self) -> isize {
        self.value
    }
}

impl PollOption {
    pub fn new(text: String, voter_count: isize) -> Self {
        Self { text, voter_count }
    }

    pub fn set_text(&mut self, text: String) {
        self.text = text;
    }

    pub fn set_voter_count(&mut self, voter_count: isize) {
        self.voter_count = voter_count;
    }

    pub fn text(&self) -> String {
        self.text.clone()
    }

    pub fn voter_count(&self) -> isize {
        self.voter_count
    }
}

impl PollAnswer {
    pub fn new(poll_id: String, user: User, option_ids: Vec<isize>) -> Self {
        Self {
            poll_id,
            user,
            option_ids,
        }
    }

    pub fn set_poll_id(&mut self, poll_id: String) {
        self.poll_id = poll_id;
    }

    pub fn set_user(&mut self, user: User) {
        self.user = user;
    }

    pub fn set_option_ids(&mut self, option_ids: Vec<isize>) {
        self.option_ids = option_ids;
    }

    pub fn poll_id(&self) -> String {
        self.poll_id.clone()
    }

    pub fn user(&self) -> User {
        self.user.clone()
    }

    pub fn option_ids(&self) -> Vec<isize> {
        self.option_ids.clone()
    }
}

impl Poll {
    pub fn new(
        id: String,
        question: String,
        options: Vec<PollOption>,
        total_voter_count: isize,
        is_closed: bool,
        is_anonymous: bool,
        type_field: String,
        allows_multiple_answers: bool,
    ) -> Self {
        Self {
            id,
            question,
            options,
            total_voter_count,
            is_closed,
            is_anonymous,
            type_field,
            allows_multiple_answers,
            correct_option_id: None,
            explanation: None,
            explanation_entities: None,
            open_period: None,
            close_date: None,
        }
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_question(&mut self, question: String) {
        self.question = question;
    }

    pub fn set_options(&mut self, options: Vec<PollOption>) {
        self.options = options;
    }

    pub fn set_total_voter_count(&mut self, total_voter_count: isize) {
        self.total_voter_count = total_voter_count;
    }

    pub fn set_is_closed(&mut self, is_closed: bool) {
        self.is_closed = is_closed;
    }

    pub fn set_is_anonymous(&mut self, is_anonymous: bool) {
        self.is_anonymous = is_anonymous;
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_allows_multiple_answers(&mut self, allows_multiple_answers: bool) {
        self.allows_multiple_answers = allows_multiple_answers;
    }

    pub fn set_correct_option_id(&mut self, correct_option_id: Option<isize>) {
        self.correct_option_id = correct_option_id;
    }

    pub fn set_explanation(&mut self, explanation: Option<String>) {
        self.explanation = explanation;
    }

    pub fn set_explanation_entities(&mut self, explanation_entities: Option<Vec<MessageEntity>>) {
        self.explanation_entities = explanation_entities;
    }

    pub fn set_open_period(&mut self, open_period: Option<isize>) {
        self.open_period = open_period;
    }

    pub fn set_close_date(&mut self, close_date: Option<isize>) {
        self.close_date = close_date;
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn question(&self) -> String {
        self.question.clone()
    }

    pub fn options(&self) -> Vec<PollOption> {
        self.options.clone()
    }

    pub fn total_voter_count(&self) -> isize {
        self.total_voter_count
    }

    pub fn is_closed(&self) -> bool {
        self.is_closed
    }

    pub fn is_anonymous(&self) -> bool {
        self.is_anonymous
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn allows_multiple_answers(&self) -> bool {
        self.allows_multiple_answers
    }

    pub fn correct_option_id(&self) -> Option<isize> {
        self.correct_option_id
    }

    pub fn explanation(&self) -> Option<String> {
        self.explanation.clone()
    }

    pub fn explanation_entities(&self) -> Option<Vec<MessageEntity>> {
        self.explanation_entities.clone()
    }

    pub fn open_period(&self) -> Option<isize> {
        self.open_period
    }

    pub fn close_date(&self) -> Option<isize> {
        self.close_date
    }
}

impl Location {
    pub fn new(longitude: f64, latitude: f64) -> Self {
        Self {
            longitude,
            latitude,
            horizontal_accuracy: None,
            live_period: None,
            heading: None,
            proximity_alert_radius: None,
        }
    }

    pub fn set_longitude(&mut self, longitude: f64) {
        self.longitude = longitude;
    }

    pub fn set_latitude(&mut self, latitude: f64) {
        self.latitude = latitude;
    }

    pub fn set_horizontal_accuracy(&mut self, horizontal_accuracy: Option<f64>) {
        self.horizontal_accuracy = horizontal_accuracy;
    }

    pub fn set_live_period(&mut self, live_period: Option<isize>) {
        self.live_period = live_period;
    }

    pub fn set_heading(&mut self, heading: Option<isize>) {
        self.heading = heading;
    }

    pub fn set_proximity_alert_radius(&mut self, proximity_alert_radius: Option<isize>) {
        self.proximity_alert_radius = proximity_alert_radius;
    }

    pub fn longitude(&self) -> f64 {
        self.longitude
    }

    pub fn latitude(&self) -> f64 {
        self.latitude
    }

    pub fn horizontal_accuracy(&self) -> Option<f64> {
        self.horizontal_accuracy
    }

    pub fn live_period(&self) -> Option<isize> {
        self.live_period
    }

    pub fn heading(&self) -> Option<isize> {
        self.heading
    }

    pub fn proximity_alert_radius(&self) -> Option<isize> {
        self.proximity_alert_radius
    }
}

impl Venue {
    pub fn new(location: Location, title: String, address: String) -> Self {
        Self {
            location,
            title,
            address,
            foursquare_id: None,
            foursquare_type: None,
            google_place_id: None,
            google_place_type: None,
        }
    }

    pub fn set_location(&mut self, location: Location) {
        self.location = location;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_address(&mut self, address: String) {
        self.address = address;
    }

    pub fn set_foursquare_id(&mut self, foursquare_id: Option<String>) {
        self.foursquare_id = foursquare_id;
    }

    pub fn set_foursquare_type(&mut self, foursquare_type: Option<String>) {
        self.foursquare_type = foursquare_type;
    }

    pub fn set_google_place_id(&mut self, google_place_id: Option<String>) {
        self.google_place_id = google_place_id;
    }

    pub fn set_google_place_type(&mut self, google_place_type: Option<String>) {
        self.google_place_type = google_place_type;
    }

    pub fn location(&self) -> Location {
        self.location.clone()
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn address(&self) -> String {
        self.address.clone()
    }

    pub fn foursquare_id(&self) -> Option<String> {
        self.foursquare_id.clone()
    }

    pub fn foursquare_type(&self) -> Option<String> {
        self.foursquare_type.clone()
    }

    pub fn google_place_id(&self) -> Option<String> {
        self.google_place_id.clone()
    }

    pub fn google_place_type(&self) -> Option<String> {
        self.google_place_type.clone()
    }
}

impl ProximityAlertTriggered {
    pub fn new(traveler: User, watcher: User, distance: isize) -> Self {
        Self {
            traveler,
            watcher,
            distance,
        }
    }

    pub fn set_traveler(&mut self, traveler: User) {
        self.traveler = traveler;
    }

    pub fn set_watcher(&mut self, watcher: User) {
        self.watcher = watcher;
    }

    pub fn set_distance(&mut self, distance: isize) {
        self.distance = distance;
    }

    pub fn traveler(&self) -> User {
        self.traveler.clone()
    }

    pub fn watcher(&self) -> User {
        self.watcher.clone()
    }

    pub fn distance(&self) -> isize {
        self.distance
    }
}

impl MessageAutoDeleteTimerChanged {
    pub fn new(message_auto_delete_time: isize) -> Self {
        Self {
            message_auto_delete_time,
        }
    }

    pub fn set_message_auto_delete_time(&mut self, message_auto_delete_time: isize) {
        self.message_auto_delete_time = message_auto_delete_time;
    }

    pub fn message_auto_delete_time(&self) -> isize {
        self.message_auto_delete_time
    }
}

impl VoiceChatEnded {
    pub fn new(duration: isize) -> Self {
        Self { duration }
    }

    pub fn set_duration(&mut self, duration: isize) {
        self.duration = duration;
    }

    pub fn duration(&self) -> isize {
        self.duration
    }
}

impl VoiceChatScheduled {
    pub fn new(start_date: isize) -> Self {
        Self { start_date }
    }

    pub fn set_start_date(&mut self, start_date: isize) {
        self.start_date = start_date;
    }

    pub fn start_date(&self) -> isize {
        self.start_date
    }
}

impl VoiceChatParticipantsInvited {
    pub fn new() -> Self {
        Self { users: None }
    }

    pub fn set_users(&mut self, users: Option<Vec<User>>) {
        self.users = users;
    }

    pub fn users(&self) -> Option<Vec<User>> {
        self.users.clone()
    }
}

impl UserProfilePhotos {
    pub fn new(total_count: isize, photos: Vec<Vec<PhotoSize>>) -> Self {
        Self {
            total_count,
            photos,
        }
    }

    pub fn set_total_count(&mut self, total_count: isize) {
        self.total_count = total_count;
    }

    pub fn set_photos(&mut self, photos: Vec<Vec<PhotoSize>>) {
        self.photos = photos;
    }

    pub fn total_count(&self) -> isize {
        self.total_count
    }

    pub fn photos(&self) -> Vec<Vec<PhotoSize>> {
        self.photos.clone()
    }
}

impl File {
    pub fn new(file_id: String, file_unique_id: String) -> Self {
        Self {
            file_id,
            file_unique_id,
            file_size: None,
            file_path: None,
        }
    }

    pub fn set_file_id(&mut self, file_id: String) {
        self.file_id = file_id;
    }

    pub fn set_file_unique_id(&mut self, file_unique_id: String) {
        self.file_unique_id = file_unique_id;
    }

    pub fn set_file_size(&mut self, file_size: Option<isize>) {
        self.file_size = file_size;
    }

    pub fn set_file_path(&mut self, file_path: Option<String>) {
        self.file_path = file_path;
    }

    pub fn file_id(&self) -> String {
        self.file_id.clone()
    }

    pub fn file_unique_id(&self) -> String {
        self.file_unique_id.clone()
    }

    pub fn file_size(&self) -> Option<isize> {
        self.file_size
    }

    pub fn file_path(&self) -> Option<String> {
        self.file_path.clone()
    }
}

impl ReplyKeyboardMarkup {
    pub fn new(keyboard: Vec<KeyboardButton>) -> Self {
        Self {
            keyboard,
            resize_keyboard: None,
            one_time_keyboard: None,
            selective: None,
        }
    }

    pub fn set_keyboard(&mut self, keyboard: Vec<KeyboardButton>) {
        self.keyboard = keyboard;
    }

    pub fn set_resize_keyboard(&mut self, resize_keyboard: Option<bool>) {
        self.resize_keyboard = resize_keyboard;
    }

    pub fn set_one_time_keyboard(&mut self, one_time_keyboard: Option<bool>) {
        self.one_time_keyboard = one_time_keyboard;
    }

    pub fn set_selective(&mut self, selective: Option<bool>) {
        self.selective = selective;
    }

    pub fn keyboard(&self) -> Vec<KeyboardButton> {
        self.keyboard.clone()
    }

    pub fn resize_keyboard(&self) -> Option<bool> {
        self.resize_keyboard
    }

    pub fn one_time_keyboard(&self) -> Option<bool> {
        self.one_time_keyboard
    }

    pub fn selective(&self) -> Option<bool> {
        self.selective
    }
}

impl KeyboardButton {
    pub fn new(text: String) -> Self {
        Self {
            text,
            request_contact: None,
            request_location: None,
            request_poll: None,
        }
    }

    pub fn set_text(&mut self, text: String) {
        self.text = text;
    }

    pub fn set_request_contact(&mut self, request_contact: Option<bool>) {
        self.request_contact = request_contact;
    }

    pub fn set_request_location(&mut self, request_location: Option<bool>) {
        self.request_location = request_location;
    }

    pub fn set_request_poll(&mut self, request_poll: Option<KeyboardButtonPollType>) {
        self.request_poll = request_poll;
    }

    pub fn text(&self) -> String {
        self.text.clone()
    }

    pub fn request_contact(&self) -> Option<bool> {
        self.request_contact
    }

    pub fn request_location(&self) -> Option<bool> {
        self.request_location
    }

    pub fn request_poll(&self) -> Option<KeyboardButtonPollType> {
        self.request_poll.clone()
    }
}

impl KeyboardButtonPollType {
    pub fn new() -> Self {
        Self { type_field: None }
    }

    pub fn set_type_field(&mut self, type_field: Option<String>) {
        self.type_field = type_field;
    }

    pub fn type_field(&self) -> Option<String> {
        self.type_field.clone()
    }
}

impl ReplyKeyboardRemove {
    pub fn new(remove_keyboard: bool) -> Self {
        Self {
            remove_keyboard,
            selective: None,
        }
    }

    pub fn set_remove_keyboard(&mut self, remove_keyboard: bool) {
        self.remove_keyboard = remove_keyboard;
    }

    pub fn set_selective(&mut self, selective: Option<bool>) {
        self.selective = selective;
    }

    pub fn remove_keyboard(&self) -> bool {
        self.remove_keyboard
    }

    pub fn selective(&self) -> Option<bool> {
        self.selective
    }
}

impl InlineKeyboardMarkup {
    pub fn new(inline_keyboard: Vec<InlineKeyboardButton>) -> Self {
        Self { inline_keyboard }
    }

    pub fn set_inline_keyboard(&mut self, inline_keyboard: Vec<InlineKeyboardButton>) {
        self.inline_keyboard = inline_keyboard;
    }

    pub fn inline_keyboard(&self) -> Vec<InlineKeyboardButton> {
        self.inline_keyboard.clone()
    }
}

impl InlineKeyboardButton {
    pub fn new(text: String) -> Self {
        Self {
            text,
            url: None,
            login_url: None,
            callback_data: None,
            switch_inline_query: None,
            switch_inline_query_current_chat: None,
            callback_game: None,
            pay: None,
        }
    }

    pub fn set_text(&mut self, text: String) {
        self.text = text;
    }

    pub fn set_url(&mut self, url: Option<String>) {
        self.url = url;
    }

    pub fn set_login_url(&mut self, login_url: Option<LoginUrl>) {
        self.login_url = login_url;
    }

    pub fn set_callback_data(&mut self, callback_data: Option<String>) {
        self.callback_data = callback_data;
    }

    pub fn set_switch_inline_query(&mut self, switch_inline_query: Option<String>) {
        self.switch_inline_query = switch_inline_query;
    }

    pub fn set_switch_inline_query_current_chat(
        &mut self,
        switch_inline_query_current_chat: Option<String>,
    ) {
        self.switch_inline_query_current_chat = switch_inline_query_current_chat;
    }

    pub fn set_callback_game(&mut self, callback_game: Option<CallbackGame>) {
        self.callback_game = callback_game;
    }

    pub fn set_pay(&mut self, pay: Option<bool>) {
        self.pay = pay;
    }

    pub fn text(&self) -> String {
        self.text.clone()
    }

    pub fn url(&self) -> Option<String> {
        self.url.clone()
    }

    pub fn login_url(&self) -> Option<LoginUrl> {
        self.login_url.clone()
    }

    pub fn callback_data(&self) -> Option<String> {
        self.callback_data.clone()
    }

    pub fn switch_inline_query(&self) -> Option<String> {
        self.switch_inline_query.clone()
    }

    pub fn switch_inline_query_current_chat(&self) -> Option<String> {
        self.switch_inline_query_current_chat.clone()
    }

    pub fn callback_game(&self) -> Option<CallbackGame> {
        self.callback_game.clone()
    }

    pub fn pay(&self) -> Option<bool> {
        self.pay
    }
}

impl LoginUrl {
    pub fn new(url: String) -> Self {
        Self {
            url,
            forward_text: None,
            bot_username: None,
            request_write_access: None,
        }
    }

    pub fn set_url(&mut self, url: String) {
        self.url = url;
    }

    pub fn set_forward_text(&mut self, forward_text: Option<String>) {
        self.forward_text = forward_text;
    }

    pub fn set_bot_username(&mut self, bot_username: Option<String>) {
        self.bot_username = bot_username;
    }

    pub fn set_request_write_access(&mut self, request_write_access: Option<bool>) {
        self.request_write_access = request_write_access;
    }

    pub fn url(&self) -> String {
        self.url.clone()
    }

    pub fn forward_text(&self) -> Option<String> {
        self.forward_text.clone()
    }

    pub fn bot_username(&self) -> Option<String> {
        self.bot_username.clone()
    }

    pub fn request_write_access(&self) -> Option<bool> {
        self.request_write_access
    }
}

impl CallbackQuery {
    pub fn new(id: String, from: User, chat_instance: String) -> Self {
        Self {
            id,
            from,
            chat_instance,
            message: None,
            inline_message_id: None,
            data: None,
            game_short_name: None,
        }
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_from(&mut self, from: User) {
        self.from = from;
    }

    pub fn set_chat_instance(&mut self, chat_instance: String) {
        self.chat_instance = chat_instance;
    }

    pub fn set_message(&mut self, message: Option<Message>) {
        self.message = message;
    }

    pub fn set_inline_message_id(&mut self, inline_message_id: Option<String>) {
        self.inline_message_id = inline_message_id;
    }

    pub fn set_data(&mut self, data: Option<String>) {
        self.data = data;
    }

    pub fn set_game_short_name(&mut self, game_short_name: Option<String>) {
        self.game_short_name = game_short_name;
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn from(&self) -> User {
        self.from.clone()
    }

    pub fn chat_instance(&self) -> String {
        self.chat_instance.clone()
    }

    pub fn message(&self) -> Option<Message> {
        self.message.clone()
    }

    pub fn inline_message_id(&self) -> Option<String> {
        self.inline_message_id.clone()
    }

    pub fn data(&self) -> Option<String> {
        self.data.clone()
    }

    pub fn game_short_name(&self) -> Option<String> {
        self.game_short_name.clone()
    }
}

impl ForceReply {
    pub fn new(force_reply: bool) -> Self {
        Self {
            force_reply,
            selective: None,
        }
    }

    pub fn set_force_reply(&mut self, force_reply: bool) {
        self.force_reply = force_reply;
    }

    pub fn set_selective(&mut self, selective: Option<bool>) {
        self.selective = selective;
    }

    pub fn force_reply(&self) -> bool {
        self.force_reply
    }

    pub fn selective(&self) -> Option<bool> {
        self.selective
    }
}

impl ChatPhoto {
    pub fn new(
        small_file_id: String,
        small_file_unique_id: String,
        big_file_id: String,
        big_file_unique_id: String,
    ) -> Self {
        Self {
            small_file_id,
            small_file_unique_id,
            big_file_id,
            big_file_unique_id,
        }
    }

    pub fn set_small_file_id(&mut self, small_file_id: String) {
        self.small_file_id = small_file_id;
    }

    pub fn set_small_file_unique_id(&mut self, small_file_unique_id: String) {
        self.small_file_unique_id = small_file_unique_id;
    }

    pub fn set_big_file_id(&mut self, big_file_id: String) {
        self.big_file_id = big_file_id;
    }

    pub fn set_big_file_unique_id(&mut self, big_file_unique_id: String) {
        self.big_file_unique_id = big_file_unique_id;
    }

    pub fn small_file_id(&self) -> String {
        self.small_file_id.clone()
    }

    pub fn small_file_unique_id(&self) -> String {
        self.small_file_unique_id.clone()
    }

    pub fn big_file_id(&self) -> String {
        self.big_file_id.clone()
    }

    pub fn big_file_unique_id(&self) -> String {
        self.big_file_unique_id.clone()
    }
}

impl ChatInviteLink {
    pub fn new(invite_link: String, creator: User, is_primary: bool, is_revoked: bool) -> Self {
        Self {
            invite_link,
            creator,
            is_primary,
            is_revoked,
            expire_date: None,
            member_limit: None,
        }
    }

    pub fn set_invite_link(&mut self, invite_link: String) {
        self.invite_link = invite_link;
    }

    pub fn set_creator(&mut self, creator: User) {
        self.creator = creator;
    }

    pub fn set_is_primary(&mut self, is_primary: bool) {
        self.is_primary = is_primary;
    }

    pub fn set_is_revoked(&mut self, is_revoked: bool) {
        self.is_revoked = is_revoked;
    }

    pub fn set_expire_date(&mut self, expire_date: Option<isize>) {
        self.expire_date = expire_date;
    }

    pub fn set_member_limit(&mut self, member_limit: Option<isize>) {
        self.member_limit = member_limit;
    }

    pub fn invite_link(&self) -> String {
        self.invite_link.clone()
    }

    pub fn creator(&self) -> User {
        self.creator.clone()
    }

    pub fn is_primary(&self) -> bool {
        self.is_primary
    }

    pub fn is_revoked(&self) -> bool {
        self.is_revoked
    }

    pub fn expire_date(&self) -> Option<isize> {
        self.expire_date
    }

    pub fn member_limit(&self) -> Option<isize> {
        self.member_limit
    }
}

impl ChatMember {
    pub fn new(user: User, status: String) -> Self {
        Self {
            user,
            status,
            custom_title: None,
            is_anonymous: None,
            can_be_edited: None,
            can_manage_chat: None,
            can_post_messages: None,
            can_edit_messages: None,
            can_delete_messages: None,
            can_manage_voice_chats: None,
            can_restrict_members: None,
            can_promote_members: None,
            can_change_info: None,
            can_invite_users: None,
            can_pin_messages: None,
            is_member: None,
            can_send_messages: None,
            can_send_media_messages: None,
            can_send_polls: None,
            can_send_other_messages: None,
            can_add_web_page_previews: None,
            until_date: None,
        }
    }

    pub fn set_user(&mut self, user: User) {
        self.user = user;
    }

    pub fn set_status(&mut self, status: String) {
        self.status = status;
    }

    pub fn set_custom_title(&mut self, custom_title: Option<String>) {
        self.custom_title = custom_title;
    }

    pub fn set_is_anonymous(&mut self, is_anonymous: Option<bool>) {
        self.is_anonymous = is_anonymous;
    }

    pub fn set_can_be_edited(&mut self, can_be_edited: Option<bool>) {
        self.can_be_edited = can_be_edited;
    }

    pub fn set_can_manage_chat(&mut self, can_manage_chat: Option<bool>) {
        self.can_manage_chat = can_manage_chat;
    }

    pub fn set_can_post_messages(&mut self, can_post_messages: Option<bool>) {
        self.can_post_messages = can_post_messages;
    }

    pub fn set_can_edit_messages(&mut self, can_edit_messages: Option<bool>) {
        self.can_edit_messages = can_edit_messages;
    }

    pub fn set_can_delete_messages(&mut self, can_delete_messages: Option<bool>) {
        self.can_delete_messages = can_delete_messages;
    }

    pub fn set_can_manage_voice_chats(&mut self, can_manage_voice_chats: Option<bool>) {
        self.can_manage_voice_chats = can_manage_voice_chats;
    }

    pub fn set_can_restrict_members(&mut self, can_restrict_members: Option<bool>) {
        self.can_restrict_members = can_restrict_members;
    }

    pub fn set_can_promote_members(&mut self, can_promote_members: Option<bool>) {
        self.can_promote_members = can_promote_members;
    }

    pub fn set_can_change_info(&mut self, can_change_info: Option<bool>) {
        self.can_change_info = can_change_info;
    }

    pub fn set_can_invite_users(&mut self, can_invite_users: Option<bool>) {
        self.can_invite_users = can_invite_users;
    }

    pub fn set_can_pin_messages(&mut self, can_pin_messages: Option<bool>) {
        self.can_pin_messages = can_pin_messages;
    }

    pub fn set_is_member(&mut self, is_member: Option<bool>) {
        self.is_member = is_member;
    }

    pub fn set_can_send_messages(&mut self, can_send_messages: Option<bool>) {
        self.can_send_messages = can_send_messages;
    }

    pub fn set_can_send_media_messages(&mut self, can_send_media_messages: Option<bool>) {
        self.can_send_media_messages = can_send_media_messages;
    }

    pub fn set_can_send_polls(&mut self, can_send_polls: Option<bool>) {
        self.can_send_polls = can_send_polls;
    }

    pub fn set_can_send_other_messages(&mut self, can_send_other_messages: Option<bool>) {
        self.can_send_other_messages = can_send_other_messages;
    }

    pub fn set_can_add_web_page_previews(&mut self, can_add_web_page_previews: Option<bool>) {
        self.can_add_web_page_previews = can_add_web_page_previews;
    }

    pub fn set_until_date(&mut self, until_date: Option<isize>) {
        self.until_date = until_date;
    }

    pub fn user(&self) -> User {
        self.user.clone()
    }

    pub fn status(&self) -> String {
        self.status.clone()
    }

    pub fn custom_title(&self) -> Option<String> {
        self.custom_title.clone()
    }

    pub fn is_anonymous(&self) -> Option<bool> {
        self.is_anonymous
    }

    pub fn can_be_edited(&self) -> Option<bool> {
        self.can_be_edited
    }

    pub fn can_manage_chat(&self) -> Option<bool> {
        self.can_manage_chat
    }

    pub fn can_post_messages(&self) -> Option<bool> {
        self.can_post_messages
    }

    pub fn can_edit_messages(&self) -> Option<bool> {
        self.can_edit_messages
    }

    pub fn can_delete_messages(&self) -> Option<bool> {
        self.can_delete_messages
    }

    pub fn can_manage_voice_chats(&self) -> Option<bool> {
        self.can_manage_voice_chats
    }

    pub fn can_restrict_members(&self) -> Option<bool> {
        self.can_restrict_members
    }

    pub fn can_promote_members(&self) -> Option<bool> {
        self.can_promote_members
    }

    pub fn can_change_info(&self) -> Option<bool> {
        self.can_change_info
    }

    pub fn can_invite_users(&self) -> Option<bool> {
        self.can_invite_users
    }

    pub fn can_pin_messages(&self) -> Option<bool> {
        self.can_pin_messages
    }

    pub fn is_member(&self) -> Option<bool> {
        self.is_member
    }

    pub fn can_send_messages(&self) -> Option<bool> {
        self.can_send_messages
    }

    pub fn can_send_media_messages(&self) -> Option<bool> {
        self.can_send_media_messages
    }

    pub fn can_send_polls(&self) -> Option<bool> {
        self.can_send_polls
    }

    pub fn can_send_other_messages(&self) -> Option<bool> {
        self.can_send_other_messages
    }

    pub fn can_add_web_page_previews(&self) -> Option<bool> {
        self.can_add_web_page_previews
    }

    pub fn until_date(&self) -> Option<isize> {
        self.until_date
    }
}

impl ChatMemberUpdated {
    pub fn new(
        chat: Chat,
        from: User,
        date: isize,
        old_chat_member: ChatMember,
        new_chat_member: ChatMember,
    ) -> Self {
        Self {
            chat,
            from,
            date,
            old_chat_member,
            new_chat_member,
            invite_link: None,
        }
    }

    pub fn set_chat(&mut self, chat: Chat) {
        self.chat = chat;
    }

    pub fn set_from(&mut self, from: User) {
        self.from = from;
    }

    pub fn set_date(&mut self, date: isize) {
        self.date = date;
    }

    pub fn set_old_chat_member(&mut self, old_chat_member: ChatMember) {
        self.old_chat_member = old_chat_member;
    }

    pub fn set_new_chat_member(&mut self, new_chat_member: ChatMember) {
        self.new_chat_member = new_chat_member;
    }

    pub fn set_invite_link(&mut self, invite_link: Option<ChatInviteLink>) {
        self.invite_link = invite_link;
    }

    pub fn chat(&self) -> Chat {
        self.chat.clone()
    }

    pub fn from(&self) -> User {
        self.from.clone()
    }

    pub fn date(&self) -> isize {
        self.date
    }

    pub fn old_chat_member(&self) -> ChatMember {
        self.old_chat_member.clone()
    }

    pub fn new_chat_member(&self) -> ChatMember {
        self.new_chat_member.clone()
    }

    pub fn invite_link(&self) -> Option<ChatInviteLink> {
        self.invite_link.clone()
    }
}

impl ChatPermissions {
    pub fn new() -> Self {
        Self {
            can_send_messages: None,
            can_send_media_messages: None,
            can_send_polls: None,
            can_send_other_messages: None,
            can_add_web_page_previews: None,
            can_change_info: None,
            can_invite_users: None,
            can_pin_messages: None,
        }
    }

    pub fn set_can_send_messages(&mut self, can_send_messages: Option<bool>) {
        self.can_send_messages = can_send_messages;
    }

    pub fn set_can_send_media_messages(&mut self, can_send_media_messages: Option<bool>) {
        self.can_send_media_messages = can_send_media_messages;
    }

    pub fn set_can_send_polls(&mut self, can_send_polls: Option<bool>) {
        self.can_send_polls = can_send_polls;
    }

    pub fn set_can_send_other_messages(&mut self, can_send_other_messages: Option<bool>) {
        self.can_send_other_messages = can_send_other_messages;
    }

    pub fn set_can_add_web_page_previews(&mut self, can_add_web_page_previews: Option<bool>) {
        self.can_add_web_page_previews = can_add_web_page_previews;
    }

    pub fn set_can_change_info(&mut self, can_change_info: Option<bool>) {
        self.can_change_info = can_change_info;
    }

    pub fn set_can_invite_users(&mut self, can_invite_users: Option<bool>) {
        self.can_invite_users = can_invite_users;
    }

    pub fn set_can_pin_messages(&mut self, can_pin_messages: Option<bool>) {
        self.can_pin_messages = can_pin_messages;
    }

    pub fn can_send_messages(&self) -> Option<bool> {
        self.can_send_messages
    }

    pub fn can_send_media_messages(&self) -> Option<bool> {
        self.can_send_media_messages
    }

    pub fn can_send_polls(&self) -> Option<bool> {
        self.can_send_polls
    }

    pub fn can_send_other_messages(&self) -> Option<bool> {
        self.can_send_other_messages
    }

    pub fn can_add_web_page_previews(&self) -> Option<bool> {
        self.can_add_web_page_previews
    }

    pub fn can_change_info(&self) -> Option<bool> {
        self.can_change_info
    }

    pub fn can_invite_users(&self) -> Option<bool> {
        self.can_invite_users
    }

    pub fn can_pin_messages(&self) -> Option<bool> {
        self.can_pin_messages
    }
}

impl ChatLocation {
    pub fn new(location: Location, address: String) -> Self {
        Self { location, address }
    }

    pub fn set_location(&mut self, location: Location) {
        self.location = location;
    }

    pub fn set_address(&mut self, address: String) {
        self.address = address;
    }

    pub fn location(&self) -> Location {
        self.location.clone()
    }

    pub fn address(&self) -> String {
        self.address.clone()
    }
}

impl BotCommand {
    pub fn new(command: String, description: String) -> Self {
        Self {
            command,
            description,
        }
    }

    pub fn set_command(&mut self, command: String) {
        self.command = command;
    }

    pub fn set_description(&mut self, description: String) {
        self.description = description;
    }

    pub fn command(&self) -> String {
        self.command.clone()
    }

    pub fn description(&self) -> String {
        self.description.clone()
    }
}

impl ResponseParameters {
    pub fn new() -> Self {
        Self {
            migrate_to_chat_id: None,
            retry_after: None,
        }
    }

    pub fn set_migrate_to_chat_id(&mut self, migrate_to_chat_id: Option<isize>) {
        self.migrate_to_chat_id = migrate_to_chat_id;
    }

    pub fn set_retry_after(&mut self, retry_after: Option<isize>) {
        self.retry_after = retry_after;
    }

    pub fn migrate_to_chat_id(&self) -> Option<isize> {
        self.migrate_to_chat_id
    }

    pub fn retry_after(&self) -> Option<isize> {
        self.retry_after
    }
}

impl Sticker {
    pub fn new(
        file_id: String,
        file_unique_id: String,
        width: isize,
        height: isize,
        is_animated: bool,
    ) -> Self {
        Self {
            file_id,
            file_unique_id,
            width,
            height,
            is_animated,
            thumb: None,
            emoji: None,
            set_name: None,
            mask_position: None,
            file_size: None,
        }
    }

    pub fn set_file_id(&mut self, file_id: String) {
        self.file_id = file_id;
    }

    pub fn set_file_unique_id(&mut self, file_unique_id: String) {
        self.file_unique_id = file_unique_id;
    }

    pub fn set_width(&mut self, width: isize) {
        self.width = width;
    }

    pub fn set_height(&mut self, height: isize) {
        self.height = height;
    }

    pub fn set_is_animated(&mut self, is_animated: bool) {
        self.is_animated = is_animated;
    }

    pub fn set_thumb(&mut self, thumb: Option<PhotoSize>) {
        self.thumb = thumb;
    }

    pub fn set_emoji(&mut self, emoji: Option<String>) {
        self.emoji = emoji;
    }

    pub fn set_set_name(&mut self, set_name: Option<String>) {
        self.set_name = set_name;
    }

    pub fn set_mask_position(&mut self, mask_position: Option<MaskPosition>) {
        self.mask_position = mask_position;
    }

    pub fn set_file_size(&mut self, file_size: Option<isize>) {
        self.file_size = file_size;
    }

    pub fn file_id(&self) -> String {
        self.file_id.clone()
    }

    pub fn file_unique_id(&self) -> String {
        self.file_unique_id.clone()
    }

    pub fn width(&self) -> isize {
        self.width
    }

    pub fn height(&self) -> isize {
        self.height
    }

    pub fn is_animated(&self) -> bool {
        self.is_animated
    }

    pub fn thumb(&self) -> Option<PhotoSize> {
        self.thumb.clone()
    }

    pub fn emoji(&self) -> Option<String> {
        self.emoji.clone()
    }

    pub fn set_name(&self) -> Option<String> {
        self.set_name.clone()
    }

    pub fn mask_position(&self) -> Option<MaskPosition> {
        self.mask_position.clone()
    }

    pub fn file_size(&self) -> Option<isize> {
        self.file_size
    }
}

impl StickerSet {
    pub fn new(
        name: String,
        title: String,
        is_animated: bool,
        contains_masks: bool,
        stickers: Vec<Sticker>,
    ) -> Self {
        Self {
            name,
            title,
            is_animated,
            contains_masks,
            stickers,
            thumb: None,
        }
    }

    pub fn set_name(&mut self, name: String) {
        self.name = name;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_is_animated(&mut self, is_animated: bool) {
        self.is_animated = is_animated;
    }

    pub fn set_contains_masks(&mut self, contains_masks: bool) {
        self.contains_masks = contains_masks;
    }

    pub fn set_stickers(&mut self, stickers: Vec<Sticker>) {
        self.stickers = stickers;
    }

    pub fn set_thumb(&mut self, thumb: Option<PhotoSize>) {
        self.thumb = thumb;
    }

    pub fn name(&self) -> String {
        self.name.clone()
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn is_animated(&self) -> bool {
        self.is_animated
    }

    pub fn contains_masks(&self) -> bool {
        self.contains_masks
    }

    pub fn stickers(&self) -> Vec<Sticker> {
        self.stickers.clone()
    }

    pub fn thumb(&self) -> Option<PhotoSize> {
        self.thumb.clone()
    }
}

impl MaskPosition {
    pub fn new(point: String, x_shift: f64, y_shift: f64, scale: f64) -> Self {
        Self {
            point,
            x_shift,
            y_shift,
            scale,
        }
    }

    pub fn set_point(&mut self, point: String) {
        self.point = point;
    }

    pub fn set_x_shift(&mut self, x_shift: f64) {
        self.x_shift = x_shift;
    }

    pub fn set_y_shift(&mut self, y_shift: f64) {
        self.y_shift = y_shift;
    }

    pub fn set_scale(&mut self, scale: f64) {
        self.scale = scale;
    }

    pub fn point(&self) -> String {
        self.point.clone()
    }

    pub fn x_shift(&self) -> f64 {
        self.x_shift
    }

    pub fn y_shift(&self) -> f64 {
        self.y_shift
    }

    pub fn scale(&self) -> f64 {
        self.scale
    }
}

impl InlineQuery {
    pub fn new(id: String, from: User, query: String, offset: String) -> Self {
        Self {
            id,
            from,
            query,
            offset,
            location: None,
            chat_type: None,
        }
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_from(&mut self, from: User) {
        self.from = from;
    }

    pub fn set_query(&mut self, query: String) {
        self.query = query;
    }

    pub fn set_offset(&mut self, offset: String) {
        self.offset = offset;
    }

    pub fn set_location(&mut self, location: Option<Location>) {
        self.location = location;
    }

    pub fn set_chat_type(&mut self, chat_type: Option<String>) {
        self.chat_type = chat_type;
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn from(&self) -> User {
        self.from.clone()
    }

    pub fn query(&self) -> String {
        self.query.clone()
    }

    pub fn offset(&self) -> String {
        self.offset.clone()
    }

    pub fn location(&self) -> Option<Location> {
        self.location.clone()
    }

    pub fn chat_type(&self) -> Option<String> {
        self.chat_type.clone()
    }
}

impl InlineQueryResultArticle {
    pub fn new(id: String, title: String, input_message_content: InputMessageContent) -> Self {
        Self {
            id,
            title,
            input_message_content,
            type_field: "article".to_string(),
            reply_markup: None,
            url: None,
            hide_url: None,
            description: None,
            thumb_url: None,
            thumb_width: None,
            thumb_height: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_input_message_content(&mut self, input_message_content: InputMessageContent) {
        self.input_message_content = input_message_content;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_url(&mut self, url: Option<String>) {
        self.url = url;
    }

    pub fn set_hide_url(&mut self, hide_url: Option<bool>) {
        self.hide_url = hide_url;
    }

    pub fn set_description(&mut self, description: Option<String>) {
        self.description = description;
    }

    pub fn set_thumb_url(&mut self, thumb_url: Option<String>) {
        self.thumb_url = thumb_url;
    }

    pub fn set_thumb_width(&mut self, thumb_width: Option<isize>) {
        self.thumb_width = thumb_width;
    }

    pub fn set_thumb_height(&mut self, thumb_height: Option<isize>) {
        self.thumb_height = thumb_height;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn input_message_content(&self) -> InputMessageContent {
        self.input_message_content.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn url(&self) -> Option<String> {
        self.url.clone()
    }

    pub fn hide_url(&self) -> Option<bool> {
        self.hide_url
    }

    pub fn description(&self) -> Option<String> {
        self.description.clone()
    }

    pub fn thumb_url(&self) -> Option<String> {
        self.thumb_url.clone()
    }

    pub fn thumb_width(&self) -> Option<isize> {
        self.thumb_width
    }

    pub fn thumb_height(&self) -> Option<isize> {
        self.thumb_height
    }
}

impl InlineQueryResultPhoto {
    pub fn new(id: String, photo_url: String, thumb_url: String) -> Self {
        Self {
            id,
            photo_url,
            thumb_url,
            type_field: "photo".to_string(),
            photo_width: None,
            photo_height: None,
            title: None,
            description: None,
            caption: None,
            parse_mode: None,
            caption_entities: None,
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_photo_url(&mut self, photo_url: String) {
        self.photo_url = photo_url;
    }

    pub fn set_thumb_url(&mut self, thumb_url: String) {
        self.thumb_url = thumb_url;
    }

    pub fn set_photo_width(&mut self, photo_width: Option<isize>) {
        self.photo_width = photo_width;
    }

    pub fn set_photo_height(&mut self, photo_height: Option<isize>) {
        self.photo_height = photo_height;
    }

    pub fn set_title(&mut self, title: Option<String>) {
        self.title = title;
    }

    pub fn set_description(&mut self, description: Option<String>) {
        self.description = description;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn photo_url(&self) -> String {
        self.photo_url.clone()
    }

    pub fn thumb_url(&self) -> String {
        self.thumb_url.clone()
    }

    pub fn photo_width(&self) -> Option<isize> {
        self.photo_width
    }

    pub fn photo_height(&self) -> Option<isize> {
        self.photo_height
    }

    pub fn title(&self) -> Option<String> {
        self.title.clone()
    }

    pub fn description(&self) -> Option<String> {
        self.description.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InlineQueryResultGif {
    pub fn new(id: String, gif_url: String, thumb_url: String) -> Self {
        Self {
            id,
            gif_url,
            thumb_url,
            type_field: "gif".to_string(),
            gif_width: None,
            gif_height: None,
            gif_duration: None,
            thumb_mime_type: None,
            title: None,
            caption: None,
            parse_mode: None,
            caption_entities: None,
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_gif_url(&mut self, gif_url: String) {
        self.gif_url = gif_url;
    }

    pub fn set_thumb_url(&mut self, thumb_url: String) {
        self.thumb_url = thumb_url;
    }

    pub fn set_gif_width(&mut self, gif_width: Option<isize>) {
        self.gif_width = gif_width;
    }

    pub fn set_gif_height(&mut self, gif_height: Option<isize>) {
        self.gif_height = gif_height;
    }

    pub fn set_gif_duration(&mut self, gif_duration: Option<isize>) {
        self.gif_duration = gif_duration;
    }

    pub fn set_thumb_mime_type(&mut self, thumb_mime_type: Option<String>) {
        self.thumb_mime_type = thumb_mime_type;
    }

    pub fn set_title(&mut self, title: Option<String>) {
        self.title = title;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn gif_url(&self) -> String {
        self.gif_url.clone()
    }

    pub fn thumb_url(&self) -> String {
        self.thumb_url.clone()
    }

    pub fn gif_width(&self) -> Option<isize> {
        self.gif_width
    }

    pub fn gif_height(&self) -> Option<isize> {
        self.gif_height
    }

    pub fn gif_duration(&self) -> Option<isize> {
        self.gif_duration
    }

    pub fn thumb_mime_type(&self) -> Option<String> {
        self.thumb_mime_type.clone()
    }

    pub fn title(&self) -> Option<String> {
        self.title.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InlineQueryResultMpeg4Gif {
    pub fn new(id: String, mpeg4_url: String, thumb_url: String) -> Self {
        Self {
            id,
            mpeg4_url,
            thumb_url,
            type_field: "mpeg4_gif".to_string(),
            mpeg4_width: None,
            mpeg4_height: None,
            mpeg4_duration: None,
            thumb_mime_type: None,
            title: None,
            caption: None,
            parse_mode: None,
            caption_entities: None,
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_mpeg4_url(&mut self, mpeg4_url: String) {
        self.mpeg4_url = mpeg4_url;
    }

    pub fn set_thumb_url(&mut self, thumb_url: String) {
        self.thumb_url = thumb_url;
    }

    pub fn set_mpeg4_width(&mut self, mpeg4_width: Option<isize>) {
        self.mpeg4_width = mpeg4_width;
    }

    pub fn set_mpeg4_height(&mut self, mpeg4_height: Option<isize>) {
        self.mpeg4_height = mpeg4_height;
    }

    pub fn set_mpeg4_duration(&mut self, mpeg4_duration: Option<isize>) {
        self.mpeg4_duration = mpeg4_duration;
    }

    pub fn set_thumb_mime_type(&mut self, thumb_mime_type: Option<String>) {
        self.thumb_mime_type = thumb_mime_type;
    }

    pub fn set_title(&mut self, title: Option<String>) {
        self.title = title;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn mpeg4_url(&self) -> String {
        self.mpeg4_url.clone()
    }

    pub fn thumb_url(&self) -> String {
        self.thumb_url.clone()
    }

    pub fn mpeg4_width(&self) -> Option<isize> {
        self.mpeg4_width
    }

    pub fn mpeg4_height(&self) -> Option<isize> {
        self.mpeg4_height
    }

    pub fn mpeg4_duration(&self) -> Option<isize> {
        self.mpeg4_duration
    }

    pub fn thumb_mime_type(&self) -> Option<String> {
        self.thumb_mime_type.clone()
    }

    pub fn title(&self) -> Option<String> {
        self.title.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InlineQueryResultVideo {
    pub fn new(
        id: String,
        video_url: String,
        mime_type: String,
        thumb_url: String,
        title: String,
    ) -> Self {
        Self {
            id,
            video_url,
            mime_type,
            thumb_url,
            title,
            type_field: "video".to_string(),
            caption: None,
            parse_mode: None,
            caption_entities: None,
            video_width: None,
            video_height: None,
            video_duration: None,
            description: None,
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_video_url(&mut self, video_url: String) {
        self.video_url = video_url;
    }

    pub fn set_mime_type(&mut self, mime_type: String) {
        self.mime_type = mime_type;
    }

    pub fn set_thumb_url(&mut self, thumb_url: String) {
        self.thumb_url = thumb_url;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_video_width(&mut self, video_width: Option<isize>) {
        self.video_width = video_width;
    }

    pub fn set_video_height(&mut self, video_height: Option<isize>) {
        self.video_height = video_height;
    }

    pub fn set_video_duration(&mut self, video_duration: Option<isize>) {
        self.video_duration = video_duration;
    }

    pub fn set_description(&mut self, description: Option<String>) {
        self.description = description;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn video_url(&self) -> String {
        self.video_url.clone()
    }

    pub fn mime_type(&self) -> String {
        self.mime_type.clone()
    }

    pub fn thumb_url(&self) -> String {
        self.thumb_url.clone()
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn video_width(&self) -> Option<isize> {
        self.video_width
    }

    pub fn video_height(&self) -> Option<isize> {
        self.video_height
    }

    pub fn video_duration(&self) -> Option<isize> {
        self.video_duration
    }

    pub fn description(&self) -> Option<String> {
        self.description.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InlineQueryResultAudio {
    pub fn new(id: String, audio_url: String, title: String) -> Self {
        Self {
            id,
            audio_url,
            title,
            type_field: "audio".to_string(),
            caption: None,
            parse_mode: None,
            caption_entities: None,
            performer: None,
            audio_duration: None,
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_audio_url(&mut self, audio_url: String) {
        self.audio_url = audio_url;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_performer(&mut self, performer: Option<String>) {
        self.performer = performer;
    }

    pub fn set_audio_duration(&mut self, audio_duration: Option<isize>) {
        self.audio_duration = audio_duration;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn audio_url(&self) -> String {
        self.audio_url.clone()
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn performer(&self) -> Option<String> {
        self.performer.clone()
    }

    pub fn audio_duration(&self) -> Option<isize> {
        self.audio_duration
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InlineQueryResultVoice {
    pub fn new(id: String, voice_url: String, title: String) -> Self {
        Self {
            id,
            voice_url,
            title,
            type_field: "voice".to_string(),
            caption: None,
            parse_mode: None,
            caption_entities: None,
            voice_duration: None,
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_voice_url(&mut self, voice_url: String) {
        self.voice_url = voice_url;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_voice_duration(&mut self, voice_duration: Option<isize>) {
        self.voice_duration = voice_duration;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn voice_url(&self) -> String {
        self.voice_url.clone()
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn voice_duration(&self) -> Option<isize> {
        self.voice_duration
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InlineQueryResultDocument {
    pub fn new(id: String, title: String, document_url: String, mime_type: String) -> Self {
        Self {
            id,
            title,
            document_url,
            mime_type,
            type_field: "document".to_string(),
            caption: None,
            parse_mode: None,
            caption_entities: None,
            description: None,
            reply_markup: None,
            input_message_content: None,
            thumb_url: None,
            thumb_width: None,
            thumb_height: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_document_url(&mut self, document_url: String) {
        self.document_url = document_url;
    }

    pub fn set_mime_type(&mut self, mime_type: String) {
        self.mime_type = mime_type;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_description(&mut self, description: Option<String>) {
        self.description = description;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn set_thumb_url(&mut self, thumb_url: Option<String>) {
        self.thumb_url = thumb_url;
    }

    pub fn set_thumb_width(&mut self, thumb_width: Option<isize>) {
        self.thumb_width = thumb_width;
    }

    pub fn set_thumb_height(&mut self, thumb_height: Option<isize>) {
        self.thumb_height = thumb_height;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn document_url(&self) -> String {
        self.document_url.clone()
    }

    pub fn mime_type(&self) -> String {
        self.mime_type.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn description(&self) -> Option<String> {
        self.description.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }

    pub fn thumb_url(&self) -> Option<String> {
        self.thumb_url.clone()
    }

    pub fn thumb_width(&self) -> Option<isize> {
        self.thumb_width
    }

    pub fn thumb_height(&self) -> Option<isize> {
        self.thumb_height
    }
}

impl InlineQueryResultLocation {
    pub fn new(id: String, latitude: f64, longitude: f64, title: String) -> Self {
        Self {
            id,
            latitude,
            longitude,
            title,
            type_field: "location".to_string(),
            horizontal_accuracy: None,
            live_period: None,
            heading: None,
            proximity_alert_radius: None,
            reply_markup: None,
            input_message_content: None,
            thumb_url: None,
            thumb_width: None,
            thumb_height: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_latitude(&mut self, latitude: f64) {
        self.latitude = latitude;
    }

    pub fn set_longitude(&mut self, longitude: f64) {
        self.longitude = longitude;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_horizontal_accuracy(&mut self, horizontal_accuracy: Option<f64>) {
        self.horizontal_accuracy = horizontal_accuracy;
    }

    pub fn set_live_period(&mut self, live_period: Option<isize>) {
        self.live_period = live_period;
    }

    pub fn set_heading(&mut self, heading: Option<isize>) {
        self.heading = heading;
    }

    pub fn set_proximity_alert_radius(&mut self, proximity_alert_radius: Option<isize>) {
        self.proximity_alert_radius = proximity_alert_radius;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn set_thumb_url(&mut self, thumb_url: Option<String>) {
        self.thumb_url = thumb_url;
    }

    pub fn set_thumb_width(&mut self, thumb_width: Option<isize>) {
        self.thumb_width = thumb_width;
    }

    pub fn set_thumb_height(&mut self, thumb_height: Option<isize>) {
        self.thumb_height = thumb_height;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn latitude(&self) -> f64 {
        self.latitude
    }

    pub fn longitude(&self) -> f64 {
        self.longitude
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn horizontal_accuracy(&self) -> Option<f64> {
        self.horizontal_accuracy
    }

    pub fn live_period(&self) -> Option<isize> {
        self.live_period
    }

    pub fn heading(&self) -> Option<isize> {
        self.heading
    }

    pub fn proximity_alert_radius(&self) -> Option<isize> {
        self.proximity_alert_radius
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }

    pub fn thumb_url(&self) -> Option<String> {
        self.thumb_url.clone()
    }

    pub fn thumb_width(&self) -> Option<isize> {
        self.thumb_width
    }

    pub fn thumb_height(&self) -> Option<isize> {
        self.thumb_height
    }
}

impl InlineQueryResultVenue {
    pub fn new(id: String, latitude: f64, longitude: f64, title: String, address: String) -> Self {
        Self {
            id,
            latitude,
            longitude,
            title,
            address,
            type_field: "venue".to_string(),
            foursquare_id: None,
            foursquare_type: None,
            google_place_id: None,
            google_place_type: None,
            reply_markup: None,
            input_message_content: None,
            thumb_url: None,
            thumb_width: None,
            thumb_height: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_latitude(&mut self, latitude: f64) {
        self.latitude = latitude;
    }

    pub fn set_longitude(&mut self, longitude: f64) {
        self.longitude = longitude;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_address(&mut self, address: String) {
        self.address = address;
    }

    pub fn set_foursquare_id(&mut self, foursquare_id: Option<String>) {
        self.foursquare_id = foursquare_id;
    }

    pub fn set_foursquare_type(&mut self, foursquare_type: Option<String>) {
        self.foursquare_type = foursquare_type;
    }

    pub fn set_google_place_id(&mut self, google_place_id: Option<String>) {
        self.google_place_id = google_place_id;
    }

    pub fn set_google_place_type(&mut self, google_place_type: Option<String>) {
        self.google_place_type = google_place_type;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn set_thumb_url(&mut self, thumb_url: Option<String>) {
        self.thumb_url = thumb_url;
    }

    pub fn set_thumb_width(&mut self, thumb_width: Option<isize>) {
        self.thumb_width = thumb_width;
    }

    pub fn set_thumb_height(&mut self, thumb_height: Option<isize>) {
        self.thumb_height = thumb_height;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn latitude(&self) -> f64 {
        self.latitude
    }

    pub fn longitude(&self) -> f64 {
        self.longitude
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn address(&self) -> String {
        self.address.clone()
    }

    pub fn foursquare_id(&self) -> Option<String> {
        self.foursquare_id.clone()
    }

    pub fn foursquare_type(&self) -> Option<String> {
        self.foursquare_type.clone()
    }

    pub fn google_place_id(&self) -> Option<String> {
        self.google_place_id.clone()
    }

    pub fn google_place_type(&self) -> Option<String> {
        self.google_place_type.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }

    pub fn thumb_url(&self) -> Option<String> {
        self.thumb_url.clone()
    }

    pub fn thumb_width(&self) -> Option<isize> {
        self.thumb_width
    }

    pub fn thumb_height(&self) -> Option<isize> {
        self.thumb_height
    }
}

impl InlineQueryResultContact {
    pub fn new(id: String, phone_number: String, first_name: String) -> Self {
        Self {
            id,
            phone_number,
            first_name,
            type_field: "contract".to_string(),
            last_name: None,
            vcard: None,
            reply_markup: None,
            input_message_content: None,
            thumb_url: None,
            thumb_width: None,
            thumb_height: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_phone_number(&mut self, phone_number: String) {
        self.phone_number = phone_number;
    }

    pub fn set_first_name(&mut self, first_name: String) {
        self.first_name = first_name;
    }

    pub fn set_last_name(&mut self, last_name: Option<String>) {
        self.last_name = last_name;
    }

    pub fn set_vcard(&mut self, vcard: Option<String>) {
        self.vcard = vcard;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn set_thumb_url(&mut self, thumb_url: Option<String>) {
        self.thumb_url = thumb_url;
    }

    pub fn set_thumb_width(&mut self, thumb_width: Option<isize>) {
        self.thumb_width = thumb_width;
    }

    pub fn set_thumb_height(&mut self, thumb_height: Option<isize>) {
        self.thumb_height = thumb_height;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn phone_number(&self) -> String {
        self.phone_number.clone()
    }

    pub fn first_name(&self) -> String {
        self.first_name.clone()
    }

    pub fn last_name(&self) -> Option<String> {
        self.last_name.clone()
    }

    pub fn vcard(&self) -> Option<String> {
        self.vcard.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }

    pub fn thumb_url(&self) -> Option<String> {
        self.thumb_url.clone()
    }

    pub fn thumb_width(&self) -> Option<isize> {
        self.thumb_width
    }

    pub fn thumb_height(&self) -> Option<isize> {
        self.thumb_height
    }
}

impl InlineQueryResultGame {
    pub fn new(id: String, game_short_name: String) -> Self {
        Self {
            id,
            game_short_name,
            type_field: "game".to_string(),
            reply_markup: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_game_short_name(&mut self, game_short_name: String) {
        self.game_short_name = game_short_name;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn game_short_name(&self) -> String {
        self.game_short_name.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }
}

impl InlineQueryResultCachedPhoto {
    pub fn new(id: String, photo_file_id: String) -> Self {
        Self {
            id,
            photo_file_id,
            type_field: "photo".to_string(),
            title: None,
            description: None,
            caption: None,
            parse_mode: None,
            caption_entities: None,
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_photo_file_id(&mut self, photo_file_id: String) {
        self.photo_file_id = photo_file_id;
    }

    pub fn set_title(&mut self, title: Option<String>) {
        self.title = title;
    }

    pub fn set_description(&mut self, description: Option<String>) {
        self.description = description;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn photo_file_id(&self) -> String {
        self.photo_file_id.clone()
    }

    pub fn title(&self) -> Option<String> {
        self.title.clone()
    }

    pub fn description(&self) -> Option<String> {
        self.description.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InlineQueryResultCachedGif {
    pub fn new(id: String, gif_file_id: String) -> Self {
        Self {
            id,
            gif_file_id,
            type_field: "gif".to_string(),
            title: None,
            caption: None,
            parse_mode: None,
            caption_entities: None,
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_gif_file_id(&mut self, gif_file_id: String) {
        self.gif_file_id = gif_file_id;
    }

    pub fn set_title(&mut self, title: Option<String>) {
        self.title = title;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn gif_file_id(&self) -> String {
        self.gif_file_id.clone()
    }

    pub fn title(&self) -> Option<String> {
        self.title.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InlineQueryResultCachedMpeg4Gif {
    pub fn new(id: String, mpeg4_file_id: String) -> Self {
        Self {
            id,
            mpeg4_file_id,
            type_field: "mpeg4_gif".to_string(),
            title: None,
            caption: None,
            parse_mode: None,
            caption_entities: None,
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_mpeg4_file_id(&mut self, mpeg4_file_id: String) {
        self.mpeg4_file_id = mpeg4_file_id;
    }

    pub fn set_title(&mut self, title: Option<String>) {
        self.title = title;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn mpeg4_file_id(&self) -> String {
        self.mpeg4_file_id.clone()
    }

    pub fn title(&self) -> Option<String> {
        self.title.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InlineQueryResultCachedSticker {
    pub fn new(id: String, sticker_file_id: String) -> Self {
        Self {
            id,
            sticker_file_id,
            type_field: "sticker".to_string(),
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_sticker_file_id(&mut self, sticker_file_id: String) {
        self.sticker_file_id = sticker_file_id;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn sticker_file_id(&self) -> String {
        self.sticker_file_id.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InlineQueryResultCachedDocument {
    pub fn new(id: String, title: String, document_file_id: String) -> Self {
        Self {
            id,
            title,
            document_file_id,
            type_field: "document".to_string(),
            description: None,
            caption: None,
            parse_mode: None,
            caption_entities: None,
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_document_file_id(&mut self, document_file_id: String) {
        self.document_file_id = document_file_id;
    }

    pub fn set_description(&mut self, description: Option<String>) {
        self.description = description;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn document_file_id(&self) -> String {
        self.document_file_id.clone()
    }

    pub fn description(&self) -> Option<String> {
        self.description.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InlineQueryResultCachedVideo {
    pub fn new(id: String, video_file_id: String, title: String) -> Self {
        Self {
            id,
            video_file_id,
            title,
            type_field: "video".to_string(),
            description: None,
            caption: None,
            parse_mode: None,
            caption_entities: None,
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_video_file_id(&mut self, video_file_id: String) {
        self.video_file_id = video_file_id;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_description(&mut self, description: Option<String>) {
        self.description = description;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn video_file_id(&self) -> String {
        self.video_file_id.clone()
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn description(&self) -> Option<String> {
        self.description.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InlineQueryResultCachedVoice {
    pub fn new(id: String, voice_file_id: String, title: String) -> Self {
        Self {
            id,
            voice_file_id,
            title,
            type_field: "voice".to_string(),
            caption: None,
            parse_mode: None,
            caption_entities: None,
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_voice_file_id(&mut self, voice_file_id: String) {
        self.voice_file_id = voice_file_id;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn voice_file_id(&self) -> String {
        self.voice_file_id.clone()
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InlineQueryResultCachedAudio {
    pub fn new(id: String, audio_file_id: String) -> Self {
        Self {
            id,
            audio_file_id,
            type_field: "audio".to_string(),
            caption: None,
            parse_mode: None,
            caption_entities: None,
            reply_markup: None,
            input_message_content: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_audio_file_id(&mut self, audio_file_id: String) {
        self.audio_file_id = audio_file_id;
    }

    pub fn set_caption(&mut self, caption: Option<String>) {
        self.caption = caption;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_caption_entities(&mut self, caption_entities: Option<Vec<MessageEntity>>) {
        self.caption_entities = caption_entities;
    }

    pub fn set_reply_markup(&mut self, reply_markup: Option<InlineKeyboardMarkup>) {
        self.reply_markup = reply_markup;
    }

    pub fn set_input_message_content(
        &mut self,
        input_message_content: Option<InputMessageContent>,
    ) {
        self.input_message_content = input_message_content;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn audio_file_id(&self) -> String {
        self.audio_file_id.clone()
    }

    pub fn caption(&self) -> Option<String> {
        self.caption.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn caption_entities(&self) -> Option<Vec<MessageEntity>> {
        self.caption_entities.clone()
    }

    pub fn reply_markup(&self) -> Option<InlineKeyboardMarkup> {
        self.reply_markup.clone()
    }

    pub fn input_message_content(&self) -> Option<InputMessageContent> {
        self.input_message_content.clone()
    }
}

impl InputTextMessageContent {
    pub fn new(message_text: String) -> Self {
        Self {
            message_text,
            parse_mode: None,
            entities: None,
            disable_web_page_preview: None,
        }
    }

    pub fn set_message_text(&mut self, message_text: String) {
        self.message_text = message_text;
    }

    pub fn set_parse_mode(&mut self, parse_mode: Option<String>) {
        self.parse_mode = parse_mode;
    }

    pub fn set_entities(&mut self, entities: Option<Vec<MessageEntity>>) {
        self.entities = entities;
    }

    pub fn set_disable_web_page_preview(&mut self, disable_web_page_preview: Option<bool>) {
        self.disable_web_page_preview = disable_web_page_preview;
    }

    pub fn message_text(&self) -> String {
        self.message_text.clone()
    }

    pub fn parse_mode(&self) -> Option<String> {
        self.parse_mode.clone()
    }

    pub fn entities(&self) -> Option<Vec<MessageEntity>> {
        self.entities.clone()
    }

    pub fn disable_web_page_preview(&self) -> Option<bool> {
        self.disable_web_page_preview
    }
}

impl InputLocationMessageContent {
    pub fn new(latitude: f64, longitude: f64) -> Self {
        Self {
            latitude,
            longitude,
            horizontal_accuracy: None,
            live_period: None,
            heading: None,
            proximity_alert_radius: None,
        }
    }

    pub fn set_latitude(&mut self, latitude: f64) {
        self.latitude = latitude;
    }

    pub fn set_longitude(&mut self, longitude: f64) {
        self.longitude = longitude;
    }

    pub fn set_horizontal_accuracy(&mut self, horizontal_accuracy: Option<f64>) {
        self.horizontal_accuracy = horizontal_accuracy;
    }

    pub fn set_live_period(&mut self, live_period: Option<isize>) {
        self.live_period = live_period;
    }

    pub fn set_heading(&mut self, heading: Option<isize>) {
        self.heading = heading;
    }

    pub fn set_proximity_alert_radius(&mut self, proximity_alert_radius: Option<isize>) {
        self.proximity_alert_radius = proximity_alert_radius;
    }

    pub fn latitude(&self) -> f64 {
        self.latitude
    }

    pub fn longitude(&self) -> f64 {
        self.longitude
    }

    pub fn horizontal_accuracy(&self) -> Option<f64> {
        self.horizontal_accuracy
    }

    pub fn live_period(&self) -> Option<isize> {
        self.live_period
    }

    pub fn heading(&self) -> Option<isize> {
        self.heading
    }

    pub fn proximity_alert_radius(&self) -> Option<isize> {
        self.proximity_alert_radius
    }
}

impl InputVenueMessageContent {
    pub fn new(latitude: f64, longitude: f64, title: String, address: String) -> Self {
        Self {
            latitude,
            longitude,
            title,
            address,
            foursquare_id: None,
            foursquare_type: None,
            google_place_id: None,
            google_place_type: None,
        }
    }

    pub fn set_latitude(&mut self, latitude: f64) {
        self.latitude = latitude;
    }

    pub fn set_longitude(&mut self, longitude: f64) {
        self.longitude = longitude;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_address(&mut self, address: String) {
        self.address = address;
    }

    pub fn set_foursquare_id(&mut self, foursquare_id: Option<String>) {
        self.foursquare_id = foursquare_id;
    }

    pub fn set_foursquare_type(&mut self, foursquare_type: Option<String>) {
        self.foursquare_type = foursquare_type;
    }

    pub fn set_google_place_id(&mut self, google_place_id: Option<String>) {
        self.google_place_id = google_place_id;
    }

    pub fn set_google_place_type(&mut self, google_place_type: Option<String>) {
        self.google_place_type = google_place_type;
    }

    pub fn latitude(&self) -> f64 {
        self.latitude
    }

    pub fn longitude(&self) -> f64 {
        self.longitude
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn address(&self) -> String {
        self.address.clone()
    }

    pub fn foursquare_id(&self) -> Option<String> {
        self.foursquare_id.clone()
    }

    pub fn foursquare_type(&self) -> Option<String> {
        self.foursquare_type.clone()
    }

    pub fn google_place_id(&self) -> Option<String> {
        self.google_place_id.clone()
    }

    pub fn google_place_type(&self) -> Option<String> {
        self.google_place_type.clone()
    }
}

impl InputContactMessageContent {
    pub fn new(phone_number: String, first_name: String) -> Self {
        Self {
            phone_number,
            first_name,
            last_name: None,
            vcard: None,
        }
    }

    pub fn set_phone_number(&mut self, phone_number: String) {
        self.phone_number = phone_number;
    }

    pub fn set_first_name(&mut self, first_name: String) {
        self.first_name = first_name;
    }

    pub fn set_last_name(&mut self, last_name: Option<String>) {
        self.last_name = last_name;
    }

    pub fn set_vcard(&mut self, vcard: Option<String>) {
        self.vcard = vcard;
    }

    pub fn phone_number(&self) -> String {
        self.phone_number.clone()
    }

    pub fn first_name(&self) -> String {
        self.first_name.clone()
    }

    pub fn last_name(&self) -> Option<String> {
        self.last_name.clone()
    }

    pub fn vcard(&self) -> Option<String> {
        self.vcard.clone()
    }
}

impl InputInvoiceMessageContent {
    pub fn new(
        title: String,
        description: String,
        payload: String,
        provider_token: String,
        currency: String,
        prices: Vec<LabeledPrice>,
    ) -> Self {
        Self {
            title,
            description,
            payload,
            provider_token,
            currency,
            prices,
            max_tip_amount: None,
            suggested_tip_amounts: None,
            provider_data: None,
            photo_url: None,
            photo_size: None,
            photo_width: None,
            photo_height: None,
            need_name: None,
            need_phone_number: None,
            need_email: None,
            need_shipping_address: None,
            send_phone_number_to_provider: None,
            send_email_to_provider: None,
            is_flexible: None,
        }
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_description(&mut self, description: String) {
        self.description = description;
    }

    pub fn set_payload(&mut self, payload: String) {
        self.payload = payload;
    }

    pub fn set_provider_token(&mut self, provider_token: String) {
        self.provider_token = provider_token;
    }

    pub fn set_currency(&mut self, currency: String) {
        self.currency = currency;
    }

    pub fn set_prices(&mut self, prices: Vec<LabeledPrice>) {
        self.prices = prices;
    }

    pub fn set_max_tip_amount(&mut self, max_tip_amount: Option<isize>) {
        self.max_tip_amount = max_tip_amount;
    }

    pub fn set_suggested_tip_amounts(&mut self, suggested_tip_amounts: Option<Vec<isize>>) {
        self.suggested_tip_amounts = suggested_tip_amounts;
    }

    pub fn set_provider_data(&mut self, provider_data: Option<String>) {
        self.provider_data = provider_data;
    }

    pub fn set_photo_url(&mut self, photo_url: Option<String>) {
        self.photo_url = photo_url;
    }

    pub fn set_photo_size(&mut self, photo_size: Option<isize>) {
        self.photo_size = photo_size;
    }

    pub fn set_photo_width(&mut self, photo_width: Option<isize>) {
        self.photo_width = photo_width;
    }

    pub fn set_photo_height(&mut self, photo_height: Option<isize>) {
        self.photo_height = photo_height;
    }

    pub fn set_need_name(&mut self, need_name: Option<bool>) {
        self.need_name = need_name;
    }

    pub fn set_need_phone_number(&mut self, need_phone_number: Option<bool>) {
        self.need_phone_number = need_phone_number;
    }

    pub fn set_need_email(&mut self, need_email: Option<bool>) {
        self.need_email = need_email;
    }

    pub fn set_need_shipping_address(&mut self, need_shipping_address: Option<bool>) {
        self.need_shipping_address = need_shipping_address;
    }

    pub fn set_send_phone_number_to_provider(
        &mut self,
        send_phone_number_to_provider: Option<bool>,
    ) {
        self.send_phone_number_to_provider = send_phone_number_to_provider;
    }

    pub fn set_send_email_to_provider(&mut self, send_email_to_provider: Option<bool>) {
        self.send_email_to_provider = send_email_to_provider;
    }

    pub fn set_is_flexible(&mut self, is_flexible: Option<bool>) {
        self.is_flexible = is_flexible;
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn description(&self) -> String {
        self.description.clone()
    }

    pub fn payload(&self) -> String {
        self.payload.clone()
    }

    pub fn provider_token(&self) -> String {
        self.provider_token.clone()
    }

    pub fn currency(&self) -> String {
        self.currency.clone()
    }

    pub fn prices(&self) -> Vec<LabeledPrice> {
        self.prices.clone()
    }

    pub fn max_tip_amount(&self) -> Option<isize> {
        self.max_tip_amount
    }

    pub fn suggested_tip_amounts(&self) -> Option<Vec<isize>> {
        self.suggested_tip_amounts.clone()
    }

    pub fn provider_data(&self) -> Option<String> {
        self.provider_data.clone()
    }

    pub fn photo_url(&self) -> Option<String> {
        self.photo_url.clone()
    }

    pub fn photo_size(&self) -> Option<isize> {
        self.photo_size
    }

    pub fn photo_width(&self) -> Option<isize> {
        self.photo_width
    }

    pub fn photo_height(&self) -> Option<isize> {
        self.photo_height
    }

    pub fn need_name(&self) -> Option<bool> {
        self.need_name
    }

    pub fn need_phone_number(&self) -> Option<bool> {
        self.need_phone_number
    }

    pub fn need_email(&self) -> Option<bool> {
        self.need_email
    }

    pub fn need_shipping_address(&self) -> Option<bool> {
        self.need_shipping_address
    }

    pub fn send_phone_number_to_provider(&self) -> Option<bool> {
        self.send_phone_number_to_provider
    }

    pub fn send_email_to_provider(&self) -> Option<bool> {
        self.send_email_to_provider
    }

    pub fn is_flexible(&self) -> Option<bool> {
        self.is_flexible
    }
}

impl ChosenInlineResult {
    pub fn new(result_id: String, from: User, query: String) -> Self {
        Self {
            result_id,
            from,
            query,
            location: None,
            inline_message_id: None,
        }
    }

    pub fn set_result_id(&mut self, result_id: String) {
        self.result_id = result_id;
    }

    pub fn set_from(&mut self, from: User) {
        self.from = from;
    }

    pub fn set_query(&mut self, query: String) {
        self.query = query;
    }

    pub fn set_location(&mut self, location: Option<Location>) {
        self.location = location;
    }

    pub fn set_inline_message_id(&mut self, inline_message_id: Option<String>) {
        self.inline_message_id = inline_message_id;
    }

    pub fn result_id(&self) -> String {
        self.result_id.clone()
    }

    pub fn from(&self) -> User {
        self.from.clone()
    }

    pub fn query(&self) -> String {
        self.query.clone()
    }

    pub fn location(&self) -> Option<Location> {
        self.location.clone()
    }

    pub fn inline_message_id(&self) -> Option<String> {
        self.inline_message_id.clone()
    }
}

impl LabeledPrice {
    pub fn new(label: String, amount: isize) -> Self {
        Self { label, amount }
    }

    pub fn set_label(&mut self, label: String) {
        self.label = label;
    }

    pub fn set_amount(&mut self, amount: isize) {
        self.amount = amount;
    }

    pub fn label(&self) -> String {
        self.label.clone()
    }

    pub fn amount(&self) -> isize {
        self.amount
    }
}

impl Invoice {
    pub fn new(
        title: String,
        description: String,
        start_parameter: String,
        currency: String,
        total_amount: isize,
    ) -> Self {
        Self {
            title,
            description,
            start_parameter,
            currency,
            total_amount,
        }
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_description(&mut self, description: String) {
        self.description = description;
    }

    pub fn set_start_parameter(&mut self, start_parameter: String) {
        self.start_parameter = start_parameter;
    }

    pub fn set_currency(&mut self, currency: String) {
        self.currency = currency;
    }

    pub fn set_total_amount(&mut self, total_amount: isize) {
        self.total_amount = total_amount;
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn description(&self) -> String {
        self.description.clone()
    }

    pub fn start_parameter(&self) -> String {
        self.start_parameter.clone()
    }

    pub fn currency(&self) -> String {
        self.currency.clone()
    }

    pub fn total_amount(&self) -> isize {
        self.total_amount
    }
}

impl ShippingAddress {
    pub fn new(
        country_code: String,
        state: String,
        city: String,
        street_line1: String,
        street_line2: String,
        post_code: String,
    ) -> Self {
        Self {
            country_code,
            state,
            city,
            street_line1,
            street_line2,
            post_code,
        }
    }

    pub fn set_country_code(&mut self, country_code: String) {
        self.country_code = country_code;
    }

    pub fn set_state(&mut self, state: String) {
        self.state = state;
    }

    pub fn set_city(&mut self, city: String) {
        self.city = city;
    }

    pub fn set_street_line1(&mut self, street_line1: String) {
        self.street_line1 = street_line1;
    }

    pub fn set_street_line2(&mut self, street_line2: String) {
        self.street_line2 = street_line2;
    }

    pub fn set_post_code(&mut self, post_code: String) {
        self.post_code = post_code;
    }

    pub fn country_code(&self) -> String {
        self.country_code.clone()
    }

    pub fn state(&self) -> String {
        self.state.clone()
    }

    pub fn city(&self) -> String {
        self.city.clone()
    }

    pub fn street_line1(&self) -> String {
        self.street_line1.clone()
    }

    pub fn street_line2(&self) -> String {
        self.street_line2.clone()
    }

    pub fn post_code(&self) -> String {
        self.post_code.clone()
    }
}

impl OrderInfo {
    pub fn new() -> Self {
        Self {
            name: None,
            phone_number: None,
            email: None,
            shipping_address: None,
        }
    }

    pub fn set_name(&mut self, name: Option<String>) {
        self.name = name;
    }

    pub fn set_phone_number(&mut self, phone_number: Option<String>) {
        self.phone_number = phone_number;
    }

    pub fn set_email(&mut self, email: Option<String>) {
        self.email = email;
    }

    pub fn set_shipping_address(&mut self, shipping_address: Option<ShippingAddress>) {
        self.shipping_address = shipping_address;
    }

    pub fn name(&self) -> Option<String> {
        self.name.clone()
    }

    pub fn phone_number(&self) -> Option<String> {
        self.phone_number.clone()
    }

    pub fn email(&self) -> Option<String> {
        self.email.clone()
    }

    pub fn shipping_address(&self) -> Option<ShippingAddress> {
        self.shipping_address.clone()
    }
}

impl ShippingOption {
    pub fn new(id: String, title: String, prices: Vec<LabeledPrice>) -> Self {
        Self { id, title, prices }
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_prices(&mut self, prices: Vec<LabeledPrice>) {
        self.prices = prices;
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn prices(&self) -> Vec<LabeledPrice> {
        self.prices.clone()
    }
}

impl SuccessfulPayment {
    pub fn new(
        currency: String,
        total_amount: isize,
        invoice_payload: String,
        telegram_payment_charge_id: String,
        provider_payment_charge_id: String,
    ) -> Self {
        Self {
            currency,
            total_amount,
            invoice_payload,
            telegram_payment_charge_id,
            provider_payment_charge_id,
            shipping_option_id: None,
            order_info: None,
        }
    }

    pub fn set_currency(&mut self, currency: String) {
        self.currency = currency;
    }

    pub fn set_total_amount(&mut self, total_amount: isize) {
        self.total_amount = total_amount;
    }

    pub fn set_invoice_payload(&mut self, invoice_payload: String) {
        self.invoice_payload = invoice_payload;
    }

    pub fn set_telegram_payment_charge_id(&mut self, telegram_payment_charge_id: String) {
        self.telegram_payment_charge_id = telegram_payment_charge_id;
    }

    pub fn set_provider_payment_charge_id(&mut self, provider_payment_charge_id: String) {
        self.provider_payment_charge_id = provider_payment_charge_id;
    }

    pub fn set_shipping_option_id(&mut self, shipping_option_id: Option<String>) {
        self.shipping_option_id = shipping_option_id;
    }

    pub fn set_order_info(&mut self, order_info: Option<OrderInfo>) {
        self.order_info = order_info;
    }

    pub fn currency(&self) -> String {
        self.currency.clone()
    }

    pub fn total_amount(&self) -> isize {
        self.total_amount
    }

    pub fn invoice_payload(&self) -> String {
        self.invoice_payload.clone()
    }

    pub fn telegram_payment_charge_id(&self) -> String {
        self.telegram_payment_charge_id.clone()
    }

    pub fn provider_payment_charge_id(&self) -> String {
        self.provider_payment_charge_id.clone()
    }

    pub fn shipping_option_id(&self) -> Option<String> {
        self.shipping_option_id.clone()
    }

    pub fn order_info(&self) -> Option<OrderInfo> {
        self.order_info.clone()
    }
}

impl ShippingQuery {
    pub fn new(
        id: String,
        from: User,
        invoice_payload: String,
        shipping_address: ShippingAddress,
    ) -> Self {
        Self {
            id,
            from,
            invoice_payload,
            shipping_address,
        }
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_from(&mut self, from: User) {
        self.from = from;
    }

    pub fn set_invoice_payload(&mut self, invoice_payload: String) {
        self.invoice_payload = invoice_payload;
    }

    pub fn set_shipping_address(&mut self, shipping_address: ShippingAddress) {
        self.shipping_address = shipping_address;
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn from(&self) -> User {
        self.from.clone()
    }

    pub fn invoice_payload(&self) -> String {
        self.invoice_payload.clone()
    }

    pub fn shipping_address(&self) -> ShippingAddress {
        self.shipping_address.clone()
    }
}

impl PreCheckoutQuery {
    pub fn new(
        id: String,
        from: User,
        currency: String,
        total_amount: isize,
        invoice_payload: String,
    ) -> Self {
        Self {
            id,
            from,
            currency,
            total_amount,
            invoice_payload,
            shipping_option_id: None,
            order_info: None,
        }
    }

    pub fn set_id(&mut self, id: String) {
        self.id = id;
    }

    pub fn set_from(&mut self, from: User) {
        self.from = from;
    }

    pub fn set_currency(&mut self, currency: String) {
        self.currency = currency;
    }

    pub fn set_total_amount(&mut self, total_amount: isize) {
        self.total_amount = total_amount;
    }

    pub fn set_invoice_payload(&mut self, invoice_payload: String) {
        self.invoice_payload = invoice_payload;
    }

    pub fn set_shipping_option_id(&mut self, shipping_option_id: Option<String>) {
        self.shipping_option_id = shipping_option_id;
    }

    pub fn set_order_info(&mut self, order_info: Option<OrderInfo>) {
        self.order_info = order_info;
    }

    pub fn id(&self) -> String {
        self.id.clone()
    }

    pub fn from(&self) -> User {
        self.from.clone()
    }

    pub fn currency(&self) -> String {
        self.currency.clone()
    }

    pub fn total_amount(&self) -> isize {
        self.total_amount
    }

    pub fn invoice_payload(&self) -> String {
        self.invoice_payload.clone()
    }

    pub fn shipping_option_id(&self) -> Option<String> {
        self.shipping_option_id.clone()
    }

    pub fn order_info(&self) -> Option<OrderInfo> {
        self.order_info.clone()
    }
}

impl PassportData {
    pub fn new(data: Vec<EncryptedPassportElement>, credentials: EncryptedCredentials) -> Self {
        Self { data, credentials }
    }

    pub fn set_data(&mut self, data: Vec<EncryptedPassportElement>) {
        self.data = data;
    }

    pub fn set_credentials(&mut self, credentials: EncryptedCredentials) {
        self.credentials = credentials;
    }

    pub fn data(&self) -> Vec<EncryptedPassportElement> {
        self.data.clone()
    }

    pub fn credentials(&self) -> EncryptedCredentials {
        self.credentials.clone()
    }
}

impl PassportFile {
    pub fn new(
        file_id: String,
        file_unique_id: String,
        file_size: isize,
        file_date: isize,
    ) -> Self {
        Self {
            file_id,
            file_unique_id,
            file_size,
            file_date,
        }
    }

    pub fn set_file_id(&mut self, file_id: String) {
        self.file_id = file_id;
    }

    pub fn set_file_unique_id(&mut self, file_unique_id: String) {
        self.file_unique_id = file_unique_id;
    }

    pub fn set_file_size(&mut self, file_size: isize) {
        self.file_size = file_size;
    }

    pub fn set_file_date(&mut self, file_date: isize) {
        self.file_date = file_date;
    }

    pub fn file_id(&self) -> String {
        self.file_id.clone()
    }

    pub fn file_unique_id(&self) -> String {
        self.file_unique_id.clone()
    }

    pub fn file_size(&self) -> isize {
        self.file_size
    }

    pub fn file_date(&self) -> isize {
        self.file_date
    }
}

impl EncryptedPassportElement {
    pub fn new(type_field: String, hash: String) -> Self {
        Self {
            type_field,
            hash,
            data: None,
            phone_number: None,
            email: None,
            files: None,
            front_side: None,
            reverse_side: None,
            selfie: None,
            translation: None,
        }
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_hash(&mut self, hash: String) {
        self.hash = hash;
    }

    pub fn set_data(&mut self, data: Option<String>) {
        self.data = data;
    }

    pub fn set_phone_number(&mut self, phone_number: Option<String>) {
        self.phone_number = phone_number;
    }

    pub fn set_email(&mut self, email: Option<String>) {
        self.email = email;
    }

    pub fn set_files(&mut self, files: Option<Vec<PassportFile>>) {
        self.files = files;
    }

    pub fn set_front_side(&mut self, front_side: Option<PassportFile>) {
        self.front_side = front_side;
    }

    pub fn set_reverse_side(&mut self, reverse_side: Option<PassportFile>) {
        self.reverse_side = reverse_side;
    }

    pub fn set_selfie(&mut self, selfie: Option<PassportFile>) {
        self.selfie = selfie;
    }

    pub fn set_translation(&mut self, translation: Option<Vec<PassportFile>>) {
        self.translation = translation;
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn hash(&self) -> String {
        self.hash.clone()
    }

    pub fn data(&self) -> Option<String> {
        self.data.clone()
    }

    pub fn phone_number(&self) -> Option<String> {
        self.phone_number.clone()
    }

    pub fn email(&self) -> Option<String> {
        self.email.clone()
    }

    pub fn files(&self) -> Option<Vec<PassportFile>> {
        self.files.clone()
    }

    pub fn front_side(&self) -> Option<PassportFile> {
        self.front_side.clone()
    }

    pub fn reverse_side(&self) -> Option<PassportFile> {
        self.reverse_side.clone()
    }

    pub fn selfie(&self) -> Option<PassportFile> {
        self.selfie.clone()
    }

    pub fn translation(&self) -> Option<Vec<PassportFile>> {
        self.translation.clone()
    }
}

impl EncryptedCredentials {
    pub fn new(data: String, hash: String, secret: String) -> Self {
        Self { data, hash, secret }
    }

    pub fn set_data(&mut self, data: String) {
        self.data = data;
    }

    pub fn set_hash(&mut self, hash: String) {
        self.hash = hash;
    }

    pub fn set_secret(&mut self, secret: String) {
        self.secret = secret;
    }

    pub fn data(&self) -> String {
        self.data.clone()
    }

    pub fn hash(&self) -> String {
        self.hash.clone()
    }

    pub fn secret(&self) -> String {
        self.secret.clone()
    }
}

impl PassportElementErrorDataField {
    pub fn new(
        source: String,
        type_field: String,
        field_name: String,
        data_hash: String,
        message: String,
    ) -> Self {
        Self {
            source,
            type_field,
            field_name,
            data_hash,
            message,
        }
    }

    pub fn set_source(&mut self, source: String) {
        self.source = source;
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_field_name(&mut self, field_name: String) {
        self.field_name = field_name;
    }

    pub fn set_data_hash(&mut self, data_hash: String) {
        self.data_hash = data_hash;
    }

    pub fn set_message(&mut self, message: String) {
        self.message = message;
    }

    pub fn source(&self) -> String {
        self.source.clone()
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn field_name(&self) -> String {
        self.field_name.clone()
    }

    pub fn data_hash(&self) -> String {
        self.data_hash.clone()
    }

    pub fn message(&self) -> String {
        self.message.clone()
    }
}

impl PassportElementErrorFrontSide {
    pub fn new(source: String, type_field: String, file_hash: String, message: String) -> Self {
        Self {
            source,
            type_field,
            file_hash,
            message,
        }
    }

    pub fn set_source(&mut self, source: String) {
        self.source = source;
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_file_hash(&mut self, file_hash: String) {
        self.file_hash = file_hash;
    }

    pub fn set_message(&mut self, message: String) {
        self.message = message;
    }

    pub fn source(&self) -> String {
        self.source.clone()
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn file_hash(&self) -> String {
        self.file_hash.clone()
    }

    pub fn message(&self) -> String {
        self.message.clone()
    }
}

impl PassportElementErrorReverseSide {
    pub fn new(source: String, type_field: String, file_hash: String, message: String) -> Self {
        Self {
            source,
            type_field,
            file_hash,
            message,
        }
    }

    pub fn set_source(&mut self, source: String) {
        self.source = source;
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_file_hash(&mut self, file_hash: String) {
        self.file_hash = file_hash;
    }

    pub fn set_message(&mut self, message: String) {
        self.message = message;
    }

    pub fn source(&self) -> String {
        self.source.clone()
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn file_hash(&self) -> String {
        self.file_hash.clone()
    }

    pub fn message(&self) -> String {
        self.message.clone()
    }
}

impl PassportElementErrorSelfie {
    pub fn new(source: String, type_field: String, file_hash: String, message: String) -> Self {
        Self {
            source,
            type_field,
            file_hash,
            message,
        }
    }

    pub fn set_source(&mut self, source: String) {
        self.source = source;
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_file_hash(&mut self, file_hash: String) {
        self.file_hash = file_hash;
    }

    pub fn set_message(&mut self, message: String) {
        self.message = message;
    }

    pub fn source(&self) -> String {
        self.source.clone()
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn file_hash(&self) -> String {
        self.file_hash.clone()
    }

    pub fn message(&self) -> String {
        self.message.clone()
    }
}

impl PassportElementErrorFile {
    pub fn new(source: String, type_field: String, file_hash: String, message: String) -> Self {
        Self {
            source,
            type_field,
            file_hash,
            message,
        }
    }

    pub fn set_source(&mut self, source: String) {
        self.source = source;
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_file_hash(&mut self, file_hash: String) {
        self.file_hash = file_hash;
    }

    pub fn set_message(&mut self, message: String) {
        self.message = message;
    }

    pub fn source(&self) -> String {
        self.source.clone()
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn file_hash(&self) -> String {
        self.file_hash.clone()
    }

    pub fn message(&self) -> String {
        self.message.clone()
    }
}

impl PassportElementErrorFiles {
    pub fn new(
        source: String,
        type_field: String,
        file_hashes: Vec<String>,
        message: String,
    ) -> Self {
        Self {
            source,
            type_field,
            file_hashes,
            message,
        }
    }

    pub fn set_source(&mut self, source: String) {
        self.source = source;
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_file_hashes(&mut self, file_hashes: Vec<String>) {
        self.file_hashes = file_hashes;
    }

    pub fn set_message(&mut self, message: String) {
        self.message = message;
    }

    pub fn source(&self) -> String {
        self.source.clone()
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn file_hashes(&self) -> Vec<String> {
        self.file_hashes.clone()
    }

    pub fn message(&self) -> String {
        self.message.clone()
    }
}

impl PassportElementErrorTranslationFile {
    pub fn new(source: String, type_field: String, file_hash: String, message: String) -> Self {
        Self {
            source,
            type_field,
            file_hash,
            message,
        }
    }

    pub fn set_source(&mut self, source: String) {
        self.source = source;
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_file_hash(&mut self, file_hash: String) {
        self.file_hash = file_hash;
    }

    pub fn set_message(&mut self, message: String) {
        self.message = message;
    }

    pub fn source(&self) -> String {
        self.source.clone()
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn file_hash(&self) -> String {
        self.file_hash.clone()
    }

    pub fn message(&self) -> String {
        self.message.clone()
    }
}

impl PassportElementErrorTranslationFiles {
    pub fn new(
        source: String,
        type_field: String,
        file_hashes: Vec<String>,
        message: String,
    ) -> Self {
        Self {
            source,
            type_field,
            file_hashes,
            message,
        }
    }

    pub fn set_source(&mut self, source: String) {
        self.source = source;
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_file_hashes(&mut self, file_hashes: Vec<String>) {
        self.file_hashes = file_hashes;
    }

    pub fn set_message(&mut self, message: String) {
        self.message = message;
    }

    pub fn source(&self) -> String {
        self.source.clone()
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn file_hashes(&self) -> Vec<String> {
        self.file_hashes.clone()
    }

    pub fn message(&self) -> String {
        self.message.clone()
    }
}

impl PassportElementErrorUnspecified {
    pub fn new(source: String, type_field: String, element_hash: String, message: String) -> Self {
        Self {
            source,
            type_field,
            element_hash,
            message,
        }
    }

    pub fn set_source(&mut self, source: String) {
        self.source = source;
    }

    pub fn set_type_field(&mut self, type_field: String) {
        self.type_field = type_field;
    }

    pub fn set_element_hash(&mut self, element_hash: String) {
        self.element_hash = element_hash;
    }

    pub fn set_message(&mut self, message: String) {
        self.message = message;
    }

    pub fn source(&self) -> String {
        self.source.clone()
    }

    pub fn type_field(&self) -> String {
        self.type_field.clone()
    }

    pub fn element_hash(&self) -> String {
        self.element_hash.clone()
    }

    pub fn message(&self) -> String {
        self.message.clone()
    }
}

impl Game {
    pub fn new(title: String, description: String, photo: Vec<PhotoSize>) -> Self {
        Self {
            title,
            description,
            photo,
            text: None,
            text_entities: None,
            animation: None,
        }
    }

    pub fn set_title(&mut self, title: String) {
        self.title = title;
    }

    pub fn set_description(&mut self, description: String) {
        self.description = description;
    }

    pub fn set_photo(&mut self, photo: Vec<PhotoSize>) {
        self.photo = photo;
    }

    pub fn set_text(&mut self, text: Option<String>) {
        self.text = text;
    }

    pub fn set_text_entities(&mut self, text_entities: Option<Vec<MessageEntity>>) {
        self.text_entities = text_entities;
    }

    pub fn set_animation(&mut self, animation: Option<Animation>) {
        self.animation = animation;
    }

    pub fn title(&self) -> String {
        self.title.clone()
    }

    pub fn description(&self) -> String {
        self.description.clone()
    }

    pub fn photo(&self) -> Vec<PhotoSize> {
        self.photo.clone()
    }

    pub fn text(&self) -> Option<String> {
        self.text.clone()
    }

    pub fn text_entities(&self) -> Option<Vec<MessageEntity>> {
        self.text_entities.clone()
    }

    pub fn animation(&self) -> Option<Animation> {
        self.animation.clone()
    }
}

impl GameHighScore {
    pub fn new(position: isize, user: User, score: isize) -> Self {
        Self {
            position,
            user,
            score,
        }
    }

    pub fn set_position(&mut self, position: isize) {
        self.position = position;
    }

    pub fn set_user(&mut self, user: User) {
        self.user = user;
    }

    pub fn set_score(&mut self, score: isize) {
        self.score = score;
    }

    pub fn position(&self) -> isize {
        self.position
    }

    pub fn user(&self) -> User {
        self.user.clone()
    }

    pub fn score(&self) -> isize {
        self.score
    }
}

impl InputFile {
    pub fn new(path: PathBuf) -> Self {
        Self { path }
    }

    pub fn set_path(&mut self, path: PathBuf) {
        self.path = path;
    }

    pub fn path(&self) -> PathBuf {
        self.path.clone()
    }
}
