use crate::api_params::AddStickerToSetParams;
use crate::api_params::AnswerCallbackQueryParams;
use crate::api_params::AnswerInlineQueryParams;
use crate::api_params::AnswerPreCheckoutQueryParams;
use crate::api_params::AnswerShippingQueryParams;
use crate::api_params::CopyMessageParams;
use crate::api_params::CreateChatInviteLinkParams;
use crate::api_params::CreateNewStickerSetParams;
use crate::api_params::DeleteChatPhotoParams;
use crate::api_params::DeleteChatStickerSetParams;
use crate::api_params::DeleteMessageParams;
use crate::api_params::DeleteStickerFromSetParams;
use crate::api_params::DeleteWebhookParams;
use crate::api_params::EditChatInviteLinkParams;
use crate::api_params::EditMessageCaptionParams;
use crate::api_params::EditMessageLiveLocationParams;
use crate::api_params::EditMessageMediaParams;
use crate::api_params::EditMessageReplyMarkupParams;
use crate::api_params::EditMessageTextParams;
use crate::api_params::ExportChatInviteLinkParams;
use crate::api_params::ForwardMessageParams;
use crate::api_params::GetChatAdministratorsParams;
use crate::api_params::GetChatMemberParams;
use crate::api_params::GetChatMembersCountParams;
use crate::api_params::GetChatParams;
use crate::api_params::GetFileParams;
use crate::api_params::GetGameHighScoresParams;
use crate::api_params::GetStickerSetParams;
use crate::api_params::GetUpdatesParams;
use crate::api_params::GetUserProfilePhotosParams;
use crate::api_params::InputMedia;
use crate::api_params::KickChatMemberParams;
use crate::api_params::LeaveChatParams;
use crate::api_params::MediaEnum;
use crate::api_params::PinChatMessageParams;
use crate::api_params::PromoteChatMemberParams;
use crate::api_params::RestrictChatMemberParams;
use crate::api_params::RevokeChatInviteLinkParams;
use crate::api_params::SendAnimationParams;
use crate::api_params::SendAudioParams;
use crate::api_params::SendChatActionParams;
use crate::api_params::SendContactParams;
use crate::api_params::SendDiceParams;
use crate::api_params::SendDocumentParams;
use crate::api_params::SendGameParams;
use crate::api_params::SendInvoiceParams;
use crate::api_params::SendLocationParams;
use crate::api_params::SendMediaGroupParams;
use crate::api_params::SendMessageParams;
use crate::api_params::SendPhotoParams;
use crate::api_params::SendPollParams;
use crate::api_params::SendStickerParams;
use crate::api_params::SendVenueParams;
use crate::api_params::SendVideoNoteParams;
use crate::api_params::SendVideoParams;
use crate::api_params::SendVoiceParams;
use crate::api_params::SetChatAdministratorCustomTitleParams;
use crate::api_params::SetChatDescriptionParams;
use crate::api_params::SetChatPermissionsParams;
use crate::api_params::SetChatPhotoParams;
use crate::api_params::SetChatStickerSetParams;
use crate::api_params::SetChatTitleParams;
use crate::api_params::SetGameScoreParams;
use crate::api_params::SetMyCommandsParams;
use crate::api_params::SetStickerPositionInSetParams;
use crate::api_params::SetStickerSetThumbParams;
use crate::api_params::SetWebhookParams;
use crate::api_params::StopMessageLiveLocationParams;
use crate::api_params::StopPollParams;
use crate::api_params::UnbanChatMemberParams;
use crate::api_params::UnpinChatMessageParams;
use crate::api_params::UploadStickerFileParams;
use crate::objects::BotCommand;
use crate::objects::Chat;
use crate::objects::ChatInviteLink;
use crate::objects::ChatMember;
use crate::objects::File;
use crate::objects::FileEnum;
use crate::objects::GameHighScore;
use crate::objects::Message;
use crate::objects::MessageId;
use crate::objects::Poll;
use crate::objects::StickerSet;
use crate::objects::Update;
use crate::objects::User;
use crate::objects::UserProfilePhotos;
use crate::objects::WebhookInfo;
use multipart::client::lazy::Multipart;
use serde::{Deserialize, Serialize};
use serde_json::Value;
use std::path::PathBuf;

static BASE_API_URL: &str = "https://api.telegram.org/bot";

#[derive(PartialEq, Debug)]
pub struct Api {
    api_url: String,
}

#[derive(PartialEq, Debug, Serialize, Deserialize)]
pub struct MethodResponse<T> {
    pub ok: bool,
    pub result: T,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub description: Option<String>,
}

#[derive(PartialEq, Debug, Serialize, Deserialize)]
pub struct ErrorResponse {
    pub ok: bool,
    pub description: String,
    pub error_code: u64,
}

#[derive(PartialEq, Debug, Serialize, Deserialize)]
#[serde(untagged)]
pub enum EditMessageResponse {
    Message(MethodResponse<Message>),
    Bool(MethodResponse<bool>),
}

#[derive(PartialEq, Debug, Serialize, Deserialize)]
#[serde(untagged)]
pub enum Error {
    HttpError(HttpError),
    ApiError(ErrorResponse),
}

#[derive(PartialEq, Debug, Serialize, Deserialize)]
pub struct HttpError {
    pub code: u16,
    pub message: String,
}

impl From<ureq::Error> for Error {
    fn from(error: ureq::Error) -> Self {
        match error {
            ureq::Error::Status(status, response) => match response.into_string() {
                Ok(string) => {
                    let json_result: Result<ErrorResponse, serde_json::Error> =
                        serde_json::from_str(&string);

                    match json_result {
                        Ok(result) => Error::ApiError(result),
                        Err(_) => {
                            let error = HttpError {
                                code: status,
                                message: string,
                            };

                            Error::HttpError(error)
                        }
                    }
                }
                Err(_) => {
                    let message = "Failed to decode response".to_string();

                    let error = HttpError {
                        code: status,
                        message: message,
                    };

                    Error::HttpError(error)
                }
            },
            ureq::Error::Transport(transport_error) => {
                let message = format!("{:?}", transport_error);

                let error = HttpError {
                    code: 500,
                    message: message,
                };

                Error::HttpError(error)
            }
        }
    }
}

impl Api {
    pub fn new(api_key: String) -> Self {
        let api_url = format!("{}{}", BASE_API_URL, api_key);

        Self { api_url }
    }

    pub fn new_url(api_url: String) -> Self {
        Self { api_url }
    }

    pub fn get_updates(
        &self,
        params: &GetUpdatesParams,
    ) -> Result<MethodResponse<Vec<Update>>, Error> {
        self.request("getUpdates", Some(params))
    }

    pub fn send_message(
        &self,
        params: &SendMessageParams,
    ) -> Result<MethodResponse<Message>, Error> {
        self.request("sendMessage", Some(params))
    }

    pub fn set_webhook(&self, params: &SetWebhookParams) -> Result<MethodResponse<bool>, Error> {
        self.request("setWebhook", Some(params))
    }

    pub fn delete_webhook(
        &self,
        params: &DeleteWebhookParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("deleteWebhook", Some(params))
    }

    pub fn get_webhook_info(&self) -> Result<MethodResponse<WebhookInfo>, Error> {
        self.request_without_body("getWebhookInfo")
    }

    pub fn get_me(&self) -> Result<MethodResponse<User>, Error> {
        self.request_without_body("getMe")
    }

    pub fn log_out(&self) -> Result<MethodResponse<bool>, Error> {
        self.request_without_body("logOut")
    }

    pub fn close(&self) -> Result<MethodResponse<bool>, Error> {
        self.request_without_body("close")
    }

    pub fn forward_message(
        &self,
        params: &ForwardMessageParams,
    ) -> Result<MethodResponse<Message>, Error> {
        self.request("forwardMessage", Some(params))
    }

    pub fn copy_message(
        &self,
        params: &CopyMessageParams,
    ) -> Result<MethodResponse<MessageId>, Error> {
        self.request("copyMessage", Some(params))
    }

    pub fn send_photo(&self, params: &SendPhotoParams) -> Result<MethodResponse<Message>, Error> {
        let method_name = "sendPhoto";
        let mut files: Vec<(&str, PathBuf)> = vec![];

        if let FileEnum::InputFileVariant(input_file) = params.photo() {
            files.push(("photo", input_file.path()));
        }

        self.request_with_possible_form_data(method_name, params, files)
    }

    pub fn send_audio(&self, params: &SendAudioParams) -> Result<MethodResponse<Message>, Error> {
        let method_name = "sendAudio";
        let mut files: Vec<(&str, PathBuf)> = vec![];

        if let FileEnum::InputFileVariant(input_file) = params.audio() {
            files.push(("audio", input_file.path()));
        }

        if let Some(FileEnum::InputFileVariant(input_file)) = params.thumb() {
            files.push(("thumb", input_file.path()));
        }

        self.request_with_possible_form_data(method_name, params, files)
    }

    pub fn send_media_group(
        &self,
        params: &SendMediaGroupParams,
    ) -> Result<MethodResponse<Vec<Message>>, Error> {
        let method_name = "sendMediaGroup";
        let mut files: Vec<(String, PathBuf)> = vec![];
        let mut new_medias: Vec<MediaEnum> = vec![];
        let mut file_idx = 0;

        for media in params.media() {
            match media {
                MediaEnum::InputMediaAudioVariant(audio) => {
                    let mut new_audio = audio.clone();

                    if let FileEnum::InputFileVariant(input_file) = audio.media() {
                        let name = format!("file{}", file_idx);
                        let attach_name = format!("attach://{}", name);
                        file_idx += 1;

                        new_audio.set_media(FileEnum::StringVariant(attach_name));

                        files.push((name, input_file.path()));
                    };

                    if let Some(FileEnum::InputFileVariant(input_file)) = audio.thumb() {
                        let name = format!("file{}", file_idx);
                        let attach_name = format!("attach://{}", name);
                        file_idx += 1;

                        new_audio.set_thumb(Some(FileEnum::StringVariant(attach_name)));

                        files.push((name, input_file.path()));
                    };

                    new_medias.push(MediaEnum::InputMediaAudioVariant(new_audio));
                }

                MediaEnum::InputMediaDocumentVariant(document) => {
                    let mut new_document = document.clone();

                    if let FileEnum::InputFileVariant(input_file) = document.media() {
                        let name = format!("file{}", file_idx);
                        let attach_name = format!("attach://{}", name);
                        file_idx += 1;

                        new_document.set_media(FileEnum::StringVariant(attach_name));

                        files.push((name, input_file.path()));
                    };

                    new_medias.push(MediaEnum::InputMediaDocumentVariant(new_document));
                }
                MediaEnum::InputMediaPhotoVariant(photo) => {
                    let mut new_photo = photo.clone();

                    if let FileEnum::InputFileVariant(input_file) = photo.media() {
                        let name = format!("file{}", file_idx);
                        let attach_name = format!("attach://{}", name);
                        file_idx += 1;

                        new_photo.set_media(FileEnum::StringVariant(attach_name));

                        files.push((name, input_file.path()));
                    };

                    new_medias.push(MediaEnum::InputMediaPhotoVariant(new_photo));
                }

                MediaEnum::InputMediaVideoVariant(video) => {
                    let mut new_video = video.clone();

                    if let FileEnum::InputFileVariant(input_file) = video.media() {
                        let name = format!("file{}", file_idx);
                        let attach_name = format!("attach://{}", name);
                        file_idx += 1;

                        new_video.set_media(FileEnum::StringVariant(attach_name));

                        files.push((name, input_file.path()));
                    };

                    if let Some(FileEnum::InputFileVariant(input_file)) = video.thumb() {
                        let name = format!("file{}", file_idx);
                        let attach_name = format!("attach://{}", name);
                        file_idx += 1;

                        new_video.set_thumb(Some(FileEnum::StringVariant(attach_name)));

                        files.push((name, input_file.path()));
                    };

                    new_medias.push(MediaEnum::InputMediaVideoVariant(new_video));
                }
            };
        }

        let mut new_params = params.clone();
        new_params.set_media(new_medias);

        let files_with_str_names: Vec<(&str, PathBuf)> = files
            .iter()
            .map(|(key, path)| (key.as_str(), path.clone()))
            .collect();

        self.request_with_possible_form_data(method_name, &new_params, files_with_str_names)
    }

    pub fn send_document(
        &self,
        params: &SendDocumentParams,
    ) -> Result<MethodResponse<Message>, Error> {
        let method_name = "sendDocument";
        let mut files: Vec<(&str, PathBuf)> = vec![];

        if let FileEnum::InputFileVariant(input_file) = params.document() {
            files.push(("document", input_file.path()));
        }

        if let Some(FileEnum::InputFileVariant(input_file)) = params.thumb() {
            files.push(("thumb", input_file.path()));
        }

        self.request_with_possible_form_data(method_name, params, files)
    }

    pub fn send_video(&self, params: &SendVideoParams) -> Result<MethodResponse<Message>, Error> {
        let method_name = "sendVideo";
        let mut files: Vec<(&str, PathBuf)> = vec![];

        if let FileEnum::InputFileVariant(input_file) = params.video() {
            files.push(("video", input_file.path()));
        }

        if let Some(FileEnum::InputFileVariant(input_file)) = params.thumb() {
            files.push(("thumb", input_file.path()));
        }

        self.request_with_possible_form_data(method_name, params, files)
    }

    pub fn send_animation(
        &self,
        params: &SendAnimationParams,
    ) -> Result<MethodResponse<Message>, Error> {
        let method_name = "sendAnimation";
        let mut files: Vec<(&str, PathBuf)> = vec![];

        if let FileEnum::InputFileVariant(input_file) = params.animation() {
            files.push(("animation", input_file.path()));
        }

        if let Some(FileEnum::InputFileVariant(input_file)) = params.thumb() {
            files.push(("thumb", input_file.path()));
        }

        self.request_with_possible_form_data(method_name, params, files)
    }

    pub fn send_voice(&self, params: &SendVoiceParams) -> Result<MethodResponse<Message>, Error> {
        let method_name = "sendVoice";
        let mut files: Vec<(&str, PathBuf)> = vec![];

        if let FileEnum::InputFileVariant(input_file) = params.voice() {
            files.push(("voice", input_file.path()));
        }

        self.request_with_possible_form_data(method_name, params, files)
    }

    pub fn send_video_note(
        &self,
        params: &SendVideoNoteParams,
    ) -> Result<MethodResponse<Message>, Error> {
        let method_name = "sendVideoNote";
        let mut files: Vec<(&str, PathBuf)> = vec![];

        if let FileEnum::InputFileVariant(input_file) = params.video_note() {
            files.push(("video_note", input_file.path()));
        }

        if let Some(FileEnum::InputFileVariant(input_file)) = params.thumb() {
            files.push(("thumb", input_file.path()));
        }

        self.request_with_possible_form_data(method_name, params, files)
    }

    pub fn send_location(
        &self,
        params: &SendLocationParams,
    ) -> Result<MethodResponse<Message>, Error> {
        self.request("sendLocation", Some(params))
    }

    pub fn edit_message_live_location(
        &self,
        params: &EditMessageLiveLocationParams,
    ) -> Result<EditMessageResponse, Error> {
        self.request("editMessageLiveLocation", Some(params))
    }

    pub fn stop_message_live_location(
        &self,
        params: &StopMessageLiveLocationParams,
    ) -> Result<EditMessageResponse, Error> {
        self.request("stopMessageLiveLocation", Some(params))
    }

    pub fn send_venue(&self, params: &SendVenueParams) -> Result<MethodResponse<Message>, Error> {
        self.request("sendVenue", Some(params))
    }

    pub fn send_contact(
        &self,
        params: &SendContactParams,
    ) -> Result<MethodResponse<Message>, Error> {
        self.request("sendContact", Some(params))
    }

    pub fn send_poll(&self, params: &SendPollParams) -> Result<MethodResponse<Message>, Error> {
        self.request("sendPoll", Some(params))
    }

    pub fn send_dice(&self, params: &SendDiceParams) -> Result<MethodResponse<Message>, Error> {
        self.request("sendDice", Some(params))
    }

    pub fn send_chat_action(
        &self,
        params: &SendChatActionParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("sendChatAction", Some(params))
    }

    pub fn get_user_profile_photos(
        &self,
        params: &GetUserProfilePhotosParams,
    ) -> Result<MethodResponse<UserProfilePhotos>, Error> {
        self.request("getUserProfilePhotos", Some(params))
    }

    pub fn get_file(&self, params: &GetFileParams) -> Result<MethodResponse<File>, Error> {
        self.request("getFile", Some(params))
    }

    pub fn kick_chat_member(
        &self,
        params: &KickChatMemberParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("kickChatMember", Some(params))
    }

    pub fn unban_chat_member(
        &self,
        params: &UnbanChatMemberParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("unbanChatMember", Some(params))
    }

    pub fn restrict_chat_member(
        &self,
        params: &RestrictChatMemberParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("restrictChatMember", Some(params))
    }

    pub fn promote_chat_member(
        &self,
        params: &PromoteChatMemberParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("promoteChatMember", Some(params))
    }

    pub fn set_chat_administrator_custom_title(
        &self,
        params: &SetChatAdministratorCustomTitleParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("setChatAdministratorCustomTitle", Some(params))
    }

    pub fn set_chat_permissions(
        &self,
        params: &SetChatPermissionsParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("setChatPermissions", Some(params))
    }

    pub fn export_chat_invite_link(
        &self,
        params: &ExportChatInviteLinkParams,
    ) -> Result<MethodResponse<String>, Error> {
        self.request("exportChatInviteLink", Some(params))
    }

    pub fn create_chat_invite_link(
        &self,
        params: &CreateChatInviteLinkParams,
    ) -> Result<MethodResponse<ChatInviteLink>, Error> {
        self.request("createChatInviteLink", Some(params))
    }

    pub fn edit_chat_invite_link(
        &self,
        params: &EditChatInviteLinkParams,
    ) -> Result<MethodResponse<ChatInviteLink>, Error> {
        self.request("editChatInviteLink", Some(params))
    }

    pub fn revoke_chat_invite_link(
        &self,
        params: &RevokeChatInviteLinkParams,
    ) -> Result<MethodResponse<ChatInviteLink>, Error> {
        self.request("revokeChatInviteLink", Some(params))
    }

    pub fn set_chat_photo(
        &self,
        params: &SetChatPhotoParams,
    ) -> Result<MethodResponse<bool>, Error> {
        let photo = params.photo();

        self.request_with_form_data("setChatPhoto", params, vec![("photo", photo.path())])
    }

    pub fn delete_chat_photo(
        &self,
        params: &DeleteChatPhotoParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("deleteChatPhoto", Some(params))
    }

    pub fn set_chat_title(
        &self,
        params: &SetChatTitleParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("setChatTitle", Some(params))
    }

    pub fn set_chat_description(
        &self,
        params: &SetChatDescriptionParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("setChatDescription", Some(params))
    }

    pub fn pin_chat_message(
        &self,
        params: &PinChatMessageParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("pinChatMessage", Some(params))
    }

    pub fn unpin_chat_message(
        &self,
        params: &UnpinChatMessageParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("unpinChatMessage", Some(params))
    }

    pub fn leave_chat(&self, params: &LeaveChatParams) -> Result<MethodResponse<bool>, Error> {
        self.request("leaveChat", Some(params))
    }

    pub fn get_chat(&self, params: &GetChatParams) -> Result<MethodResponse<Chat>, Error> {
        self.request("getChat", Some(params))
    }

    pub fn get_chat_administrators(
        &self,
        params: &GetChatAdministratorsParams,
    ) -> Result<MethodResponse<Vec<ChatMember>>, Error> {
        self.request("getChatAdministrators", Some(params))
    }

    pub fn get_chat_members_count(
        &self,
        params: &GetChatMembersCountParams,
    ) -> Result<MethodResponse<usize>, Error> {
        self.request("getChatMembersCount", Some(params))
    }

    pub fn get_chat_member(
        &self,
        params: &GetChatMemberParams,
    ) -> Result<MethodResponse<ChatMember>, Error> {
        self.request("getChatMember", Some(params))
    }

    pub fn set_chat_sticker_set(
        &self,
        params: &SetChatStickerSetParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("setChatStickerSet", Some(params))
    }

    pub fn delete_chat_sticker_set(
        &self,
        params: &DeleteChatStickerSetParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("deleteChatStickerSet", Some(params))
    }

    pub fn answer_callback_query(
        &self,
        params: &AnswerCallbackQueryParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("answerCallbackQuery", Some(params))
    }

    pub fn set_my_commands(
        &self,
        params: &SetMyCommandsParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("setMyCommands", Some(params))
    }

    pub fn get_my_commands(&self) -> Result<MethodResponse<Vec<BotCommand>>, Error> {
        self.request_without_body("getMyCommands")
    }

    pub fn answer_inline_query(
        &self,
        params: &AnswerInlineQueryParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("answerInlineQuery", Some(params))
    }

    pub fn edit_message_text(
        &self,
        params: &EditMessageTextParams,
    ) -> Result<EditMessageResponse, Error> {
        self.request("editMessageText", Some(params))
    }

    pub fn edit_message_caption(
        &self,
        params: &EditMessageCaptionParams,
    ) -> Result<EditMessageResponse, Error> {
        self.request("editMessageCaption", Some(params))
    }

    pub fn edit_message_media(
        &self,
        params: &EditMessageMediaParams,
    ) -> Result<EditMessageResponse, Error> {
        let method_name = "editMessageMedia";
        let mut files: Vec<(String, PathBuf)> = vec![];

        let new_media = match params.media() {
            InputMedia::InputMediaAnimationVariant(animation) => {
                let mut new_animation = animation.clone();

                if let FileEnum::InputFileVariant(input_file) = animation.media() {
                    let name = "animation".to_string();
                    let attach_name = format!("attach://{}", name);

                    new_animation.set_media(FileEnum::StringVariant(attach_name));

                    files.push((name, input_file.path()));
                };

                if let Some(FileEnum::InputFileVariant(input_file)) = animation.thumb() {
                    let name = "animation_thumb".to_string();
                    let attach_name = format!("attach://{}", name);

                    new_animation.set_thumb(Some(FileEnum::StringVariant(attach_name)));

                    files.push((name, input_file.path()));
                };

                InputMedia::InputMediaAnimationVariant(new_animation)
            }
            InputMedia::InputMediaDocumentVariant(document) => {
                let mut new_document = document.clone();

                if let FileEnum::InputFileVariant(input_file) = document.media() {
                    let name = "document".to_string();
                    let attach_name = format!("attach://{}", name);

                    new_document.set_media(FileEnum::StringVariant(attach_name));

                    files.push((name, input_file.path()));
                };

                if let Some(FileEnum::InputFileVariant(input_file)) = document.thumb() {
                    let name = "document_thumb".to_string();
                    let attach_name = format!("attach://{}", name);

                    new_document.set_thumb(Some(FileEnum::StringVariant(attach_name)));

                    files.push((name, input_file.path()));
                };

                InputMedia::InputMediaDocumentVariant(new_document)
            }
            InputMedia::InputMediaAudioVariant(audio) => {
                let mut new_audio = audio.clone();

                if let FileEnum::InputFileVariant(input_file) = audio.media() {
                    let name = "audio".to_string();
                    let attach_name = format!("attach://{}", name);

                    new_audio.set_media(FileEnum::StringVariant(attach_name));

                    files.push((name, input_file.path()));
                };

                if let Some(FileEnum::InputFileVariant(input_file)) = audio.thumb() {
                    let name = "audio_thumb".to_string();
                    let attach_name = format!("attach://{}", name);

                    new_audio.set_thumb(Some(FileEnum::StringVariant(attach_name)));

                    files.push((name, input_file.path()));
                };

                InputMedia::InputMediaAudioVariant(new_audio)
            }
            InputMedia::InputMediaPhotoVariant(photo) => {
                let mut new_photo = photo.clone();

                if let FileEnum::InputFileVariant(input_file) = photo.media() {
                    let name = "photo".to_string();
                    let attach_name = format!("attach://{}", name);

                    new_photo.set_media(FileEnum::StringVariant(attach_name));

                    files.push((name, input_file.path()));
                };

                InputMedia::InputMediaPhotoVariant(new_photo)
            }
            InputMedia::InputMediaVideoVariant(video) => {
                let mut new_video = video.clone();

                if let FileEnum::InputFileVariant(input_file) = video.media() {
                    let name = "video".to_string();
                    let attach_name = format!("attach://{}", name);

                    new_video.set_media(FileEnum::StringVariant(attach_name));

                    files.push((name, input_file.path()));
                };

                if let Some(FileEnum::InputFileVariant(input_file)) = video.thumb() {
                    let name = "video_thumb".to_string();
                    let attach_name = format!("attach://{}", name);

                    new_video.set_thumb(Some(FileEnum::StringVariant(attach_name)));

                    files.push((name, input_file.path()));
                };

                InputMedia::InputMediaVideoVariant(new_video)
            }
        };

        let mut new_params = params.clone();
        new_params.set_media(new_media);

        let files_with_str_names: Vec<(&str, PathBuf)> = files
            .iter()
            .map(|(key, path)| (key.as_str(), path.clone()))
            .collect();

        self.request_with_possible_form_data(method_name, &new_params, files_with_str_names)
    }

    pub fn edit_message_reply_markup(
        &self,
        params: &EditMessageReplyMarkupParams,
    ) -> Result<EditMessageResponse, Error> {
        self.request("editMessageReplyMarkup", Some(params))
    }

    pub fn stop_poll(&self, params: &StopPollParams) -> Result<MethodResponse<Poll>, Error> {
        self.request("stopPoll", Some(params))
    }

    pub fn delete_message(
        &self,
        params: &DeleteMessageParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("deleteMessage", Some(params))
    }

    pub fn send_sticker(
        &self,
        params: &SendStickerParams,
    ) -> Result<MethodResponse<Message>, Error> {
        let method_name = "sendSticker";
        let mut files: Vec<(&str, PathBuf)> = vec![];

        if let FileEnum::InputFileVariant(input_file) = params.sticker() {
            files.push(("sticker", input_file.path()));
        }

        self.request_with_possible_form_data(method_name, params, files)
    }

    pub fn get_sticker_set(
        &self,
        params: &GetStickerSetParams,
    ) -> Result<MethodResponse<StickerSet>, Error> {
        self.request("getStickerSet", Some(params))
    }

    pub fn upload_sticker_file(
        &self,
        params: &UploadStickerFileParams,
    ) -> Result<MethodResponse<File>, Error> {
        let sticker = params.png_sticker();

        self.request_with_form_data(
            "uploadStickerFile",
            params,
            vec![("png_sticker", sticker.path())],
        )
    }

    pub fn create_new_sticker_set(
        &self,
        params: &CreateNewStickerSetParams,
    ) -> Result<MethodResponse<bool>, Error> {
        let method_name = "createNewStickerSet";
        let mut files: Vec<(&str, PathBuf)> = vec![];

        if let Some(FileEnum::InputFileVariant(input_file)) = params.png_sticker() {
            files.push(("png_sticker", input_file.path()));
        }

        if let Some(input_file) = params.tgs_sticker() {
            files.push(("tgs_sticker", input_file.path()));
        }

        self.request_with_possible_form_data(method_name, params, files)
    }

    pub fn add_sticker_to_set(
        &self,
        params: &AddStickerToSetParams,
    ) -> Result<MethodResponse<bool>, Error> {
        let method_name = "addStickerToSet";
        let mut files: Vec<(&str, PathBuf)> = vec![];

        if let Some(FileEnum::InputFileVariant(input_file)) = params.png_sticker() {
            files.push(("png_sticker", input_file.path()));
        }

        if let Some(input_file) = params.tgs_sticker() {
            files.push(("tgs_sticker", input_file.path()));
        }

        self.request_with_possible_form_data(method_name, params, files)
    }

    pub fn set_sticker_position_in_set(
        &self,
        params: &SetStickerPositionInSetParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("setStickerPositionInSet", Some(params))
    }

    pub fn delete_sticker_from_set(
        &self,
        params: &DeleteStickerFromSetParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("deleteStickerFromSet", Some(params))
    }

    pub fn set_sticker_set_thumb(
        &self,
        params: &SetStickerSetThumbParams,
    ) -> Result<MethodResponse<bool>, Error> {
        let method_name = "setStickerSetThumb";
        let mut files: Vec<(&str, PathBuf)> = vec![];

        if let Some(FileEnum::InputFileVariant(input_file)) = params.thumb() {
            files.push(("thumb", input_file.path()));
        }

        self.request_with_possible_form_data(method_name, params, files)
    }

    pub fn send_invoice(
        &self,
        params: &SendInvoiceParams,
    ) -> Result<MethodResponse<Message>, Error> {
        self.request("sendInvoice", Some(params))
    }

    pub fn answer_shipping_query(
        &self,
        params: &AnswerShippingQueryParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("answerShippingQuery", Some(params))
    }

    pub fn answer_pre_checkout_query(
        &self,
        params: &AnswerPreCheckoutQueryParams,
    ) -> Result<MethodResponse<bool>, Error> {
        self.request("answerPreCheckoutQuery", Some(params))
    }

    pub fn send_game(&self, params: &SendGameParams) -> Result<MethodResponse<Message>, Error> {
        self.request("sendGame", Some(params))
    }

    pub fn set_game_score(
        &self,
        params: &SetGameScoreParams,
    ) -> Result<EditMessageResponse, Error> {
        self.request("setGameScore", Some(params))
    }

    pub fn get_game_high_scores(
        &self,
        params: &GetGameHighScoresParams,
    ) -> Result<MethodResponse<Vec<GameHighScore>>, Error> {
        self.request("getGameHighScores", Some(params))
    }

    fn request_without_body<T: serde::de::DeserializeOwned>(
        &self,
        method: &str,
    ) -> Result<T, Error> {
        let params: Option<()> = None;

        self.request(method, params)
    }

    fn request<T1: serde::ser::Serialize, T2: serde::de::DeserializeOwned>(
        &self,
        method: &str,
        params: Option<T1>,
    ) -> Result<T2, Error> {
        let url = format!("{}/{}", self.api_url, method);

        let prepared_request = ureq::post(&url).set("Content-Type", "application/json");

        let response = match params {
            None => prepared_request.call()?,
            Some(data) => {
                let json = serde_json::to_string(&data).unwrap();

                prepared_request.send_string(&json)?
            }
        };

        let parsed_response: T2 = serde_json::from_reader(response.into_reader()).unwrap();

        Ok(parsed_response)
    }

    fn request_with_possible_form_data<
        T1: serde::ser::Serialize,
        T2: serde::de::DeserializeOwned,
    >(
        &self,
        method_name: &str,
        params: &T1,
        files: Vec<(&str, PathBuf)>,
    ) -> Result<T2, Error> {
        if files.is_empty() {
            self.request(method_name, Some(params))
        } else {
            self.request_with_form_data(method_name, params, files)
        }
    }

    fn request_with_form_data<T1: serde::ser::Serialize, T2: serde::de::DeserializeOwned>(
        &self,
        method: &str,
        params: T1,
        files: Vec<(&str, PathBuf)>,
    ) -> Result<T2, Error> {
        let json_string = serde_json::to_string(&params).unwrap();
        let json_struct: Value = serde_json::from_str(&json_string).unwrap();
        let file_keys: Vec<&str> = files.iter().map(|(key, _)| *key).collect();
        let files_with_names: Vec<(&str, Option<&str>, PathBuf)> = files
            .iter()
            .map(|(key, path)| (*key, path.file_name().unwrap().to_str(), path.clone()))
            .collect();

        let mut form = Multipart::new();
        for (key, val) in json_struct.as_object().unwrap().iter() {
            if !file_keys.contains(&key.as_str()) {
                let val = match val {
                    &Value::String(ref val) => val.to_string(),
                    other => other.to_string(),
                };

                form.add_text(key, val);
            }
        }

        for (parameter_name, file_name, file_path) in files_with_names {
            let file = std::fs::File::open(&file_path).unwrap();
            let file_extension = file_path.extension().and_then(|s| s.to_str()).unwrap_or("");
            let mime = mime_guess::from_ext(&file_extension).first_or_octet_stream();

            form.add_stream(parameter_name, file, file_name, Some(mime));
        }

        let url = format!("{}/{}", self.api_url, method);
        let form_data = form.prepare().unwrap();
        let response = ureq::post(&url)
            .set(
                "Content-Type",
                &format!("multipart/form-data; boundary={}", form_data.boundary()),
            )
            .send(form_data)?;

        let parsed_response: T2 = serde_json::from_reader(response.into_reader()).unwrap();

        Ok(parsed_response)
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::api_params::*;
    use crate::objects::*;

    #[test]
    fn new_sets_correct_url() {
        let api = Api::new("hey".to_string());

        assert_eq!(
            Api {
                api_url: "https://api.telegram.org/bothey".to_string()
            },
            api
        );
    }

    #[test]
    fn get_updates_success() {
        let response_string = "{\"ok\":true,\"result\":[{\"update_id\":379656753,\"message\":{\"message_id\":2741,\"from\":{\"id\":275808073,\"is_bot\":false,\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\",\"username\":\"Ayrat555\",\"language_code\":\"en\"},\"date\":1618149703,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"text\":\"dsaf\"}}]}";
        let mut params = GetUpdatesParams::new();
        params.set_allowed_updates(Some(vec!["message".to_string()]));

        let _m = mockito::mock("POST", "/getUpdates")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.get_updates(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();

        assert_eq!(response_string, json);
        assert_eq!(1, response.result.len());

        let update = &response.result[0];

        assert_eq!(379656753, update.update_id())
    }

    #[test]
    fn send_message_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2746,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618207352,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"text\":\"Hello!\"}}";
        let params =
            SendMessageParams::new(ChatIdEnum::IsizeVariant(275808073), "Hello!".to_string());
        let _m = mockito::mock("POST", "/sendMessage")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_message(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_message_failure() {
        let response_string =
            "{\"ok\":false,\"description\":\"Bad Request: chat not found\",\"error_code\":400}";
        let params = SendMessageParams::new(ChatIdEnum::IsizeVariant(1), "Hello!".to_string());
        let _m = mockito::mock("POST", "/sendMessage")
            .with_status(400)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        if let Err(Error::ApiError(ErrorResponse {
            ok: false,
            description,
            error_code: 400,
        })) = api.send_message(&params)
        {
            assert_eq!("Bad Request: chat not found".to_string(), description);
        } else {
            assert!(false)
        }
    }

    #[test]
    fn set_webhook_success() {
        let response_string =
            "{\"ok\":true,\"result\":true,\"description\":\"Webhook is already deleted\"}";
        let params = SetWebhookParams::new("".to_string());
        let _m = mockito::mock("POST", "/setWebhook")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.set_webhook(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn delete_webhook_success() {
        let response_string =
            "{\"ok\":true,\"result\":true,\"description\":\"Webhook is already deleted\"}";
        let params = DeleteWebhookParams::new();
        let _m = mockito::mock("POST", "/deleteWebhook")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.delete_webhook(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn get_webhook_info_success() {
        let response_string = "{\"ok\":true,\"result\":{\"url\":\"\",\"has_custom_certificate\":false,\"pending_update_count\":0,\"allowed_updates\":[\"message\"]}}";
        let _m = mockito::mock("POST", "/getWebhookInfo")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.get_webhook_info().unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn get_me_success() {
        let response_string = "{\"ok\":true,\"result\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\",\"can_join_groups\":true,\"can_read_all_group_messages\":false,\"supports_inline_queries\":false}}";
        let _m = mockito::mock("POST", "/getMe")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.get_me().unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn log_out_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let _m = mockito::mock("POST", "/logOut")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.log_out().unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn close_failure() {
        let response_string = "{\"ok\":false,\"description\":\"Unauthorized\",\"error_code\":401}";

        let _m = mockito::mock("POST", "/close")
            .with_status(400)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.close().err();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn close_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let _m = mockito::mock("POST", "/close")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.close().unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn forward_message_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2747,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618294971,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"text\":\"Hello\"}}";
        let params = ForwardMessageParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            ChatIdEnum::IsizeVariant(275808073),
            2747,
        );

        let _m = mockito::mock("POST", "/forwardMessage")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.forward_message(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn copy_message_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2749}}";
        let params = CopyMessageParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            ChatIdEnum::IsizeVariant(275808073),
            2747,
        );

        let _m = mockito::mock("POST", "/copyMessage")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.copy_message(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_location_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2750,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618382060,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"location\":{\"longitude\":6.949981,\"latitude\":49.700002}}}";
        let params = SendLocationParams::new(ChatIdEnum::IsizeVariant(275808073), 49.7, 6.95);

        let _m = mockito::mock("POST", "/sendLocation")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_location(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn edit_message_live_location_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2752,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618382998,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"edit_date\":1618383189,\"location\":{\"longitude\":6.96,\"latitude\":49.800009,\"live_period\":300}}}";
        let mut params = EditMessageLiveLocationParams::new(49.8, 6.96);
        params.set_message_id(Some(2752));
        params.set_chat_id(Some(ChatIdEnum::IsizeVariant(275808073)));

        let _m = mockito::mock("POST", "/editMessageLiveLocation")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.edit_message_live_location(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn stop_message_live_location_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2752,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618382998,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"edit_date\":1618383189,\"location\":{\"longitude\":6.96,\"latitude\":49.800009,\"live_period\":300}}}";
        let mut params = StopMessageLiveLocationParams::new();
        params.set_message_id(Some(2752));
        params.set_chat_id(Some(ChatIdEnum::IsizeVariant(275808073)));

        let _m = mockito::mock("POST", "/stopMessageLiveLocation")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.stop_message_live_location(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_venue_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2754,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618410490,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"venue\":{\"location\":{\"longitude\":6.949981,\"latitude\":49.700002},\"title\":\"Meow\",\"address\":\"Hoof\"},\"location\":{\"longitude\":6.949981,\"latitude\":49.700002}}}";
        let params = SendVenueParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            49.7,
            6.95,
            "Meow".to_string(),
            "Hoof".to_string(),
        );

        let _m = mockito::mock("POST", "/sendVenue")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_venue(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_contact_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2755,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618465934,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"contact\":{\"phone_number\":\"49\",\"first_name\":\"Meow\"}}}";
        let params = SendContactParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            "49".to_string(),
            "Meow".to_string(),
        );

        let _m = mockito::mock("POST", "/sendContact")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_contact(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_poll_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2756,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618466302,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"poll\":{\"id\":\"5458519832506925078\",\"question\":\"are you?\",\"options\":[{\"text\":\"1\",\"voter_count\":0},{\"text\":\"2\",\"voter_count\":0}],\"total_voter_count\":0,\"is_closed\":false,\"is_anonymous\":true,\"type\":\"regular\",\"allows_multiple_answers\":false}}}";
        let params = SendPollParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            "are you?".to_string(),
            vec!["1".to_string(), "2".to_string()],
        );

        let _m = mockito::mock("POST", "/sendPoll")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_poll(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_dice_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2757,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618467133,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"dice\":{\"emoji\":\"🎲\",\"value\":5}}}";
        let params = SendDiceParams::new(ChatIdEnum::IsizeVariant(275808073));

        let _m = mockito::mock("POST", "/sendDice")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_dice(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_chat_action_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let params =
            SendChatActionParams::new(ChatIdEnum::IsizeVariant(275808073), "typing".to_string());

        let _m = mockito::mock("POST", "/sendChatAction")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_chat_action(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn get_user_profile_photos_success() {
        let response_string = "{\"ok\":true,\"result\":{\"total_count\":3,\"photos\":[[{\"file_id\":\"AgACAgIAAxUAAWB332IlzabFGWzaMrOdQ4ODVLyaAAKypzEbSX9wEEzMxT7F-grc3UA5DwAEAQADAgADYQADg0kCAAEfBA\",\"file_unique_id\":\"AQAD3UA5DwAEg0kCAAE\",\"width\":160,\"height\":160,\"file_size\":8068},{\"file_id\":\"AgACAgIAAxUAAWB332IlzabFGWzaMrOdQ4ODVLyaAAKypzEbSX9wEEzMxT7F-grc3UA5DwAEAQADAgADYgADhEkCAAEfBA\",\"file_unique_id\":\"AQAD3UA5DwAEhEkCAAE\",\"width\":320,\"height\":320,\"file_size\":22765},{\"file_id\":\"AgACAgIAAxUAAWB332IlzabFGWzaMrOdQ4ODVLyaAAKypzEbSX9wEEzMxT7F-grc3UA5DwAEAQADAgADYwADhUkCAAEfBA\",\"file_unique_id\":\"AQAD3UA5DwAEhUkCAAE\",\"width\":640,\"height\":640,\"file_size\":65663}],[{\"file_id\":\"AgACAgIAAxUAAWB332JpnZNv9ZNeZeIt1FFCdOroAAKwpzEbSX9wEOb5okUMX3tVSRdLDQAEAQADAgADYQADZj0KAAEfBA\",\"file_unique_id\":\"AQADSRdLDQAEZj0KAAE\",\"width\":160,\"height\":160,\"file_size\":13459},{\"file_id\":\"AgACAgIAAxUAAWB332JpnZNv9ZNeZeIt1FFCdOroAAKwpzEbSX9wEOb5okUMX3tVSRdLDQAEAQADAgADYgADZz0KAAEfBA\",\"file_unique_id\":\"AQADSRdLDQAEZz0KAAE\",\"width\":320,\"height\":320,\"file_size\":41243},{\"file_id\":\"AgACAgIAAxUAAWB332JpnZNv9ZNeZeIt1FFCdOroAAKwpzEbSX9wEOb5okUMX3tVSRdLDQAEAQADAgADYwADaD0KAAEfBA\",\"file_unique_id\":\"AQADSRdLDQAEaD0KAAE\",\"width\":640,\"height\":640,\"file_size\":114427}],[{\"file_id\":\"AgACAgIAAxUAAWB332ISVowq4pXLx3y1o-7WQteeAAKvpzEbSX9wEBlOkdDjqlYW1Du3DQAEAQADAgADYQADdkwAAh8E\",\"file_unique_id\":\"AQAD1Du3DQAEdkwAAg\",\"width\":160,\"height\":160,\"file_size\":6631},{\"file_id\":\"AgACAgIAAxUAAWB332ISVowq4pXLx3y1o-7WQteeAAKvpzEbSX9wEBlOkdDjqlYW1Du3DQAEAQADAgADYgADd0wAAh8E\",\"file_unique_id\":\"AQAD1Du3DQAEd0wAAg\",\"width\":320,\"height\":320,\"file_size\":20495},{\"file_id\":\"AgACAgIAAxUAAWB332ISVowq4pXLx3y1o-7WQteeAAKvpzEbSX9wEBlOkdDjqlYW1Du3DQAEAQADAgADYwADeEwAAh8E\",\"file_unique_id\":\"AQAD1Du3DQAEeEwAAg\",\"width\":640,\"height\":640,\"file_size\":54395}]]}}";
        let params = GetUserProfilePhotosParams::new(275808073);

        let _m = mockito::mock("POST", "/getUserProfilePhotos")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.get_user_profile_photos(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn get_file_success() {
        let response_string = "{\"ok\":true,\"result\":{\"file_id\":\"AgACAgIAAxUAAWB332IlzabFGWzaMrOdQ4ODVLyaAAKypzEbSX9wEEzMxT7F-grc3UA5DwAEAQADAgADYQADg0kCAAEfBA\",\"file_unique_id\":\"AQAD3UA5DwAEg0kCAAE\",\"file_size\":8068,\"file_path\":\"photos/file_0.jpg\"}}";
        let  params = GetFileParams::new(
        "AgACAgIAAxUAAWB332IlzabFGWzaMrOdQ4ODVLyaAAKypzEbSX9wEEzMxT7F-grc3UA5DwAEAQADAgADYQADg0kCAAEfBA".to_string()
    );

        let _m = mockito::mock("POST", "/getFile")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.get_file(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn kick_chat_member_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let params = KickChatMemberParams::new(ChatIdEnum::IsizeVariant(-1001368460856), 275808073);

        let _m = mockito::mock("POST", "/kickChatMember")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.kick_chat_member(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn unban_chat_member_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let params =
            UnbanChatMemberParams::new(ChatIdEnum::IsizeVariant(-1001368460856), 275808072);

        let _m = mockito::mock("POST", "/unbanChatMember")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.unban_chat_member(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn restrict_chat_member_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let mut perm = ChatPermissions::new();
        perm.set_can_add_web_page_previews(Some(true));
        let params = RestrictChatMemberParams::new(
            ChatIdEnum::IsizeVariant(-1001368460856),
            275808073,
            perm,
        );

        let _m = mockito::mock("POST", "/restrictChatMember")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.restrict_chat_member(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn promote_chat_member_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let mut params =
            PromoteChatMemberParams::new(ChatIdEnum::IsizeVariant(-1001368460856), 275808073);
        params.set_can_change_info(Some(true));

        let _m = mockito::mock("POST", "/promoteChatMember")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.promote_chat_member(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn set_chat_administrator_custom_title_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let params = SetChatAdministratorCustomTitleParams::new(
            ChatIdEnum::IsizeVariant(-1001368460856),
            275808073,
            "King".to_string(),
        );

        let _m = mockito::mock("POST", "/setChatAdministratorCustomTitle")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.set_chat_administrator_custom_title(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn set_chat_permissions_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let mut perm = ChatPermissions::new();
        perm.set_can_add_web_page_previews(Some(true));

        let params = SetChatPermissionsParams::new(ChatIdEnum::IsizeVariant(-1001368460856), perm);

        let _m = mockito::mock("POST", "/setChatPermissions")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.set_chat_permissions(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn export_chat_invite_link_success() {
        let response_string = "{\"ok\":true,\"result\":\"https://t.me/joinchat/txIUDwjfk7M2ODEy\"}";

        let params = ExportChatInviteLinkParams::new(ChatIdEnum::IsizeVariant(-1001368460856));

        let _m = mockito::mock("POST", "/exportChatInviteLink")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.export_chat_invite_link(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn create_chat_invite_link_success() {
        let response_string = "{\"ok\":true,\"result\":{\"invite_link\":\"https://t.me/joinchat/yFEnSaeiQm83ZTNi\",\"creator\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"is_primary\":false,\"is_revoked\":false}}";

        let params = CreateChatInviteLinkParams::new(ChatIdEnum::IsizeVariant(-1001368460856));

        let _m = mockito::mock("POST", "/createChatInviteLink")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.create_chat_invite_link(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn edit_chat_invite_link_success() {
        let response_string = "{\"ok\":true,\"result\":{\"invite_link\":\"https://t.me/joinchat/O458bA8hQ0MzNmQy\",\"creator\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"is_primary\":false,\"is_revoked\":false}}";

        let params = EditChatInviteLinkParams::new(
            ChatIdEnum::IsizeVariant(-1001368460856),
            "https://t.me/joinchat/O458bA8hQ0MzNmQy".to_string(),
        );

        let _m = mockito::mock("POST", "/editChatInviteLink")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.edit_chat_invite_link(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn revoke_chat_invite_link_success() {
        let response_string = "{\"ok\":true,\"result\":{\"invite_link\":\"https://t.me/joinchat/O458bA8hQ0MzNmQy\",\"creator\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"is_primary\":false,\"is_revoked\":true}}";

        let params = RevokeChatInviteLinkParams::new(
            ChatIdEnum::IsizeVariant(-1001368460856),
            "https://t.me/joinchat/O458bA8hQ0MzNmQy".to_string(),
        );

        let _m = mockito::mock("POST", "/revokeChatInviteLink")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.revoke_chat_invite_link(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn delete_chat_photo_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let params = DeleteChatPhotoParams::new(ChatIdEnum::IsizeVariant(-1001368460856));

        let _m = mockito::mock("POST", "/deleteChatPhoto")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.delete_chat_photo(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_photo_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2763,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618730180,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"photo\":[{\"file_id\":\"AgACAgIAAxkDAAIKy2B73MQXIhoDDmLXjcUjgqGf-m8bAALjsDEbORLgS-s4BkBzcC5DYvIBny4AAwEAAwIAA20AA0U3AwABHwQ\",\"file_unique_id\":\"AQADYvIBny4AA0U3AwAB\",\"width\":320,\"height\":320,\"file_size\":19968},{\"file_id\":\"AgACAgIAAxkDAAIKy2B73MQXIhoDDmLXjcUjgqGf-m8bAALjsDEbORLgS-s4BkBzcC5DYvIBny4AAwEAAwIAA3gAA0Y3AwABHwQ\",\"file_unique_id\":\"AQADYvIBny4AA0Y3AwAB\",\"width\":799,\"height\":800,\"file_size\":63581},{\"file_id\":\"AgACAgIAAxkDAAIKy2B73MQXIhoDDmLXjcUjgqGf-m8bAALjsDEbORLgS-s4BkBzcC5DYvIBny4AAwEAAwIAA3kAA0M3AwABHwQ\",\"file_unique_id\":\"AQADYvIBny4AA0M3AwAB\",\"width\":847,\"height\":848,\"file_size\":63763}]}}";
        let params = SendPhotoParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            FileEnum::InputFileVariant(InputFile::new(std::path::PathBuf::from(
                "./frankenstein_logo.png",
            ))),
        );

        let _m = mockito::mock("POST", "/sendPhoto")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_photo(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_audio_file_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2766,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618735176,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"audio\":{\"file_id\":\"CQACAgIAAxkDAAIKzmB78EjK-iOHo-HKC-M6p4r0jGdmAALkDAACORLgS5co1z0uFAKgHwQ\",\"file_unique_id\":\"AgAD5AwAAjkS4Es\",\"duration\":123,\"title\":\"Way Back Home\",\"file_name\":\"audio.mp3\",\"mime_type\":\"audio/mpeg\",\"file_size\":2957092}}}";
        let params = SendAudioParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            FileEnum::InputFileVariant(InputFile::new(std::path::PathBuf::from(
                "./frankenstein_logo.png",
            ))),
        );
        let _m = mockito::mock("POST", "/sendAudio")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_audio(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_audio_file_with_thumb_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2766,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618735176,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"audio\":{\"file_id\":\"CQACAgIAAxkDAAIKzmB78EjK-iOHo-HKC-M6p4r0jGdmAALkDAACORLgS5co1z0uFAKgHwQ\",\"file_unique_id\":\"AgAD5AwAAjkS4Es\",\"duration\":123,\"title\":\"Way Back Home\",\"file_name\":\"audio.mp3\",\"mime_type\":\"audio/mpeg\",\"file_size\":2957092}}}";
        let mut params = SendAudioParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            FileEnum::InputFileVariant(InputFile::new(std::path::PathBuf::from(
                "./frankenstein_logo.png",
            ))),
        );
        params.set_thumb(Some(FileEnum::InputFileVariant(InputFile::new(
            std::path::PathBuf::from("./frankenstein_logo.png"),
        ))));
        let _m = mockito::mock("POST", "/sendAudio")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_audio(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_audio_file_id_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2769,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618735333,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"audio\":{\"file_id\":\"CQACAgIAAxkDAAIK0WB78OUFavWx6fjzCQ_d5qnu_R7mAALkDAACORLgS5co1z0uFAKgHwQ\",\"file_unique_id\":\"AgAD5AwAAjkS4Es\",\"duration\":123,\"title\":\"Way Back Home\",\"file_name\":\"audio.mp3\",\"mime_type\":\"audio/mpeg\",\"file_size\":2957092}}}";
        let params = SendAudioParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            FileEnum::StringVariant(
                "CQACAgIAAxkDAAIKzmB78EjK-iOHo-HKC-M6p4r0jGdmAALkDAACORLgS5co1z0uFAKgHwQ"
                    .to_string(),
            ),
        );
        let _m = mockito::mock("POST", "/sendAudio")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_audio(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_document_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2770,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618737593,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"audio\":{\"file_id\":\"CQACAgIAAxkDAAIK0mB7-bnnewABfdaFKK4NzVLQ7BvgCwAC6gwAAjkS4Et_njaNR8IUMB8E\",\"file_unique_id\":\"AgAD6gwAAjkS4Es\",\"duration\":123,\"title\":\"Way Back Home\",\"file_name\":\"audio.mp3\",\"mime_type\":\"audio/mpeg\",\"file_size\":2957092}}}";
        let params = SendDocumentParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            FileEnum::InputFileVariant(InputFile::new(std::path::PathBuf::from(
                "./frankenstein_logo.png",
            ))),
        );
        let _m = mockito::mock("POST", "/sendDocument")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_document(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_video_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2770,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618737593,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"audio\":{\"file_id\":\"CQACAgIAAxkDAAIK0mB7-bnnewABfdaFKK4NzVLQ7BvgCwAC6gwAAjkS4Et_njaNR8IUMB8E\",\"file_unique_id\":\"AgAD6gwAAjkS4Es\",\"duration\":123,\"title\":\"Way Back Home\",\"file_name\":\"audio.mp3\",\"mime_type\":\"audio/mpeg\",\"file_size\":2957092}}}";
        let params = SendVideoParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            FileEnum::InputFileVariant(InputFile::new(std::path::PathBuf::from(
                "./frankenstein_logo.png",
            ))),
        );
        let _m = mockito::mock("POST", "/sendVideo")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_video(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_animation_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2770,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618737593,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"audio\":{\"file_id\":\"CQACAgIAAxkDAAIK0mB7-bnnewABfdaFKK4NzVLQ7BvgCwAC6gwAAjkS4Et_njaNR8IUMB8E\",\"file_unique_id\":\"AgAD6gwAAjkS4Es\",\"duration\":123,\"title\":\"Way Back Home\",\"file_name\":\"audio.mp3\",\"mime_type\":\"audio/mpeg\",\"file_size\":2957092}}}";
        let params = SendAnimationParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            FileEnum::InputFileVariant(InputFile::new(std::path::PathBuf::from(
                "./frankenstein_logo.png",
            ))),
        );
        let _m = mockito::mock("POST", "/sendAnimation")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_animation(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_voice_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2770,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618737593,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"audio\":{\"file_id\":\"CQACAgIAAxkDAAIK0mB7-bnnewABfdaFKK4NzVLQ7BvgCwAC6gwAAjkS4Et_njaNR8IUMB8E\",\"file_unique_id\":\"AgAD6gwAAjkS4Es\",\"duration\":123,\"title\":\"Way Back Home\",\"file_name\":\"audio.mp3\",\"mime_type\":\"audio/mpeg\",\"file_size\":2957092}}}";
        let params = SendVoiceParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            FileEnum::InputFileVariant(InputFile::new(std::path::PathBuf::from(
                "./frankenstein_logo.png",
            ))),
        );
        let _m = mockito::mock("POST", "/sendVoice")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_voice(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_video_note_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2770,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1618737593,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"audio\":{\"file_id\":\"CQACAgIAAxkDAAIK0mB7-bnnewABfdaFKK4NzVLQ7BvgCwAC6gwAAjkS4Et_njaNR8IUMB8E\",\"file_unique_id\":\"AgAD6gwAAjkS4Es\",\"duration\":123,\"title\":\"Way Back Home\",\"file_name\":\"audio.mp3\",\"mime_type\":\"audio/mpeg\",\"file_size\":2957092}}}";
        let params = SendVideoNoteParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            FileEnum::InputFileVariant(InputFile::new(std::path::PathBuf::from(
                "./frankenstein_logo.png",
            ))),
        );
        let _m = mockito::mock("POST", "/sendVideoNote")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_video_note(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn set_chat_photo_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let params = SetChatPhotoParams::new(
            ChatIdEnum::IsizeVariant(-1001368460856),
            InputFile::new(std::path::PathBuf::from("./frankenstein_logo.png")),
        );

        let _m = mockito::mock("POST", "/setChatPhoto")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.set_chat_photo(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn set_chat_title_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let params = SetChatTitleParams::new(
            ChatIdEnum::IsizeVariant(-1001368460856),
            "Frankenstein".to_string(),
        );

        let _m = mockito::mock("POST", "/setChatTitle")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.set_chat_title(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn set_chat_description_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let mut params = SetChatDescriptionParams::new(ChatIdEnum::IsizeVariant(-1001368460856));
        params.set_description(Some("Frankenstein group".to_string()));

        let _m = mockito::mock("POST", "/setChatDescription")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.set_chat_description(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn pin_chat_message_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let params = PinChatMessageParams::new(ChatIdEnum::IsizeVariant(275808073), 2766);

        let _m = mockito::mock("POST", "/pinChatMessage")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.pin_chat_message(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn unpin_chat_message_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let params = UnpinChatMessageParams::new(ChatIdEnum::IsizeVariant(275808073));

        let _m = mockito::mock("POST", "/unpinChatMessage")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.unpin_chat_message(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn leave_chat_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let params = LeaveChatParams::new(ChatIdEnum::IsizeVariant(-1001368460856));

        let _m = mockito::mock("POST", "/leaveChat")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.leave_chat(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn get_chat_success() {
        let response_string = "{\"ok\":true,\"result\":{\"id\":-1001368460856,\"type\":\"supergroup\",\"title\":\"Frankenstein\",\"photo\":{\"small_file_id\":\"AQADAgAT-kgrmy4AAwIAA8jhydkW____s1Cm6Dc_w8Ge7QUAAR8E\",\"small_file_unique_id\":\"AQAD-kgrmy4AA57tBQAB\",\"big_file_id\":\"AQADAgAT-kgrmy4AAwMAA8jhydkW____s1Cm6Dc_w8Gg7QUAAR8E\",\"big_file_unique_id\":\"AQAD-kgrmy4AA6DtBQAB\"},\"description\":\"Frankenstein group\",\"invite_link\":\"https://t.me/joinchat/smSXMzNKTwA0ZjFi\",\"permissions\":{\"can_send_messages\":true,\"can_send_media_messages\":true,\"can_send_polls\":true,\"can_send_other_messages\":true,\"can_add_web_page_previews\":true,\"can_change_info\":true,\"can_invite_users\":true,\"can_pin_messages\":true}}}";
        let params = GetChatParams::new(ChatIdEnum::IsizeVariant(-1001368460856));

        let _m = mockito::mock("POST", "/getChat")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.get_chat(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn get_chat_administrators_success() {
        let response_string = "{\"ok\":true,\"result\":[{\"user\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"status\":\"administrator\",\"is_anonymous\":false,\"can_be_edited\":false,\"can_manage_chat\":true,\"can_delete_messages\":true,\"can_manage_voice_chats\":true,\"can_restrict_members\":true,\"can_promote_members\":true,\"can_change_info\":true,\"can_invite_users\":true,\"can_pin_messages\":true},{\"user\":{\"id\":275808073,\"is_bot\":false,\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\",\"username\":\"Ayrat555\",\"language_code\":\"en\"},\"status\":\"creator\",\"is_anonymous\":false}]}";
        let params = GetChatAdministratorsParams::new(ChatIdEnum::IsizeVariant(-1001368460856));

        let _m = mockito::mock("POST", "/getChatAdministrators")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.get_chat_administrators(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn get_chat_members_count_success() {
        let response_string = "{\"ok\":true,\"result\":4}";
        let params = GetChatMembersCountParams::new(ChatIdEnum::IsizeVariant(-1001368460856));

        let _m = mockito::mock("POST", "/getChatMembersCount")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.get_chat_members_count(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn get_chat_member_success() {
        let response_string = "{\"ok\":true,\"result\":{\"user\":{\"id\":275808073,\"is_bot\":false,\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\",\"username\":\"Ayrat555\",\"language_code\":\"en\"},\"status\":\"creator\",\"is_anonymous\":false}}";
        let params = GetChatMemberParams::new(ChatIdEnum::IsizeVariant(-1001368460856), 275808073);

        let _m = mockito::mock("POST", "/getChatMember")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.get_chat_member(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn set_chat_sticker_set_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let params = SetChatStickerSetParams::new(
            ChatIdEnum::IsizeVariant(-1001368460856),
            "GBTDPack".to_string(),
        );

        let _m = mockito::mock("POST", "/setChatStickerSet")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.set_chat_sticker_set(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn delete_chat_sticker_set_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let params = DeleteChatStickerSetParams::new(ChatIdEnum::IsizeVariant(-1001368460856));

        let _m = mockito::mock("POST", "/deleteChatStickerSet")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.delete_chat_sticker_set(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn answer_callback_query_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let mut params = AnswerCallbackQueryParams::new("id".to_string());
        params.set_text(Some("text".to_string()));

        let _m = mockito::mock("POST", "/answerCallbackQuery")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.answer_callback_query(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn set_my_commands_success() {
        let response_string = "{\"ok\":true,\"result\":true}";
        let params = SetMyCommandsParams::new(vec![
            BotCommand::new("meow".to_string(), "mewo".to_string()),
            BotCommand::new("meow1".to_string(), "mewo1".to_string()),
        ]);

        let _m = mockito::mock("POST", "/setMyCommands")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.set_my_commands(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn get_my_commands_success() {
        let response_string = "{\"ok\":true,\"result\":[{\"command\":\"meow\",\"description\":\"mewo\"},{\"command\":\"meow1\",\"description\":\"mewo1\"}]}";

        let _m = mockito::mock("POST", "/getMyCommands")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.get_my_commands().unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn answer_inline_query_success() {
        let response_string = "{\"ok\":true,\"result\":true}";

        let venue_result = InlineQueryResultVenue::new(
            "hey9sdfasdflasdfadsfasd".to_string(),
            40.0,
            40.0,
            "title".to_string(),
            "address".to_string(),
        );

        let params = AnswerInlineQueryParams::new(
            "inline_query.id()".to_string(),
            vec![InlineQueryResult::InlineQueryResultVenueVariant(
                venue_result,
            )],
        );

        let _m = mockito::mock("POST", "/answerInlineQuery")
            .with_status(200)
            .with_body(response_string)
            .create();

        let api = Api::new_url(mockito::server_url());

        let response = api.answer_inline_query(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn edit_message_text_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2782,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1619158127,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"edit_date\":1619158446,\"text\":\"Hi!!\"}}";

        let mut params = EditMessageTextParams::new("Hi!!".to_string());

        params.set_chat_id(Some(ChatIdEnum::IsizeVariant(275808073)));
        params.set_message_id(Some(2782));

        let _m = mockito::mock("POST", "/editMessageText")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.edit_message_text(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn edit_message_caption_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2784,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1619159414,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"edit_date\":1619159461,\"photo\":[{\"file_id\":\"AgACAgIAAxkDAAIK4GCCaaWDayYgzQ-BykVy8LYkW0wzAAL-rzEbRx8RSCnCkWjXtdN9ZNLmny4AAwEAAwIAA20AA1hCAAIfBA\",\"file_unique_id\":\"AQADZNLmny4AA1hCAAI\",\"width\":320,\"height\":320,\"file_size\":19162},{\"file_id\":\"AgACAgIAAxkDAAIK4GCCaaWDayYgzQ-BykVy8LYkW0wzAAL-rzEbRx8RSCnCkWjXtdN9ZNLmny4AAwEAAwIAA3gAA1lCAAIfBA\",\"file_unique_id\":\"AQADZNLmny4AA1lCAAI\",\"width\":800,\"height\":800,\"file_size\":65697},{\"file_id\":\"AgACAgIAAxkDAAIK4GCCaaWDayYgzQ-BykVy8LYkW0wzAAL-rzEbRx8RSCnCkWjXtdN9ZNLmny4AAwEAAwIAA3kAA1pCAAIfBA\",\"file_unique_id\":\"AQADZNLmny4AA1pCAAI\",\"width\":1146,\"height\":1146,\"file_size\":101324}],\"caption\":\"Caption\"}}";

        let mut params = EditMessageCaptionParams::new();

        params.set_chat_id(Some(ChatIdEnum::IsizeVariant(275808073)));
        params.set_message_id(Some(2784));
        params.set_caption(Some("Caption".to_string()));

        let _m = mockito::mock("POST", "/editMessageCaption")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.edit_message_caption(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn stop_poll_success() {
        let response_string = "{\"ok\":true,\"result\":{\"id\":\"5195109123171024927\",\"question\":\"are you?\",\"options\":[{\"text\":\"1\",\"voter_count\":1},{\"text\":\"2\",\"voter_count\":0}],\"total_voter_count\":1,\"is_closed\":true,\"is_anonymous\":true,\"type\":\"regular\",\"allows_multiple_answers\":false}}";

        let params = StopPollParams::new(ChatIdEnum::IsizeVariant(-1001368460856), 495);

        let _m = mockito::mock("POST", "/stopPoll")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.stop_poll(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn delete_message_success() {
        let response_string = "{\"ok\":true,\"result\":true}";

        let params = DeleteMessageParams::new(ChatIdEnum::IsizeVariant(275808073), 2784);

        let _m = mockito::mock("POST", "/deleteMessage")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.delete_message(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_sticker_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":2788,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1619245784,\"chat\":{\"id\":275808073,\"type\":\"private\",\"username\":\"Ayrat555\",\"first_name\":\"Ayrat\",\"last_name\":\"Badykov\"},\"sticker\":{\"file_id\":\"CAACAgIAAxkDAAIK5GCDutgNxc07rqqtjkGWrGskbHfQAAIMEAACRx8ZSKJ6Z5GkdVHcHwQ\",\"file_unique_id\":\"AgADDBAAAkcfGUg\",\"width\":512,\"height\":512,\"is_animated\":false,\"thumb\":{\"file_id\":\"AAMCAgADGQMAAgrkYIO62A3FzTuuqq2OQZasayRsd9AAAgwQAAJHHxlIonpnkaR1Udz29bujLgADAQAHbQADzR4AAh8E\",\"file_unique_id\":\"AQAD9vW7oy4AA80eAAI\",\"width\":320,\"height\":320,\"file_size\":19264},\"file_size\":36596}}}";
        let params = SendStickerParams::new(
            ChatIdEnum::IsizeVariant(275808073),
            FileEnum::InputFileVariant(InputFile::new(std::path::PathBuf::from(
                "./frankenstein_logo.png",
            ))),
        );

        let _m = mockito::mock("POST", "/sendSticker")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_sticker(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn get_sticker_set_success() {
        let response_string = "{\"ok\":true,\"result\":{\"name\":\"unocards\",\"title\":\"UNO Bot\",\"is_animated\":false,\"contains_masks\":false,\"stickers\":[{\"file_id\":\"CAACAgQAAxUAAWCDxAQVJ6X7FGiBD5NyjN5DDvgfAALZAQACX1eZAAEqnpNt3SpG_x8E\",\"file_unique_id\":\"AgAD2QEAAl9XmQAB\",\"width\":342,\"height\":512,\"is_animated\":false,\"thumb\":{\"file_id\":\"AAMCBAADFQABYIPEBBUnpfsUaIEPk3KM3kMO-B8AAtkBAAJfV5kAASqek23dKkb_P75BGQAEAQAHbQADBBEAAh8E\",\"file_unique_id\":\"AQADP75BGQAEBBEAAg\",\"width\":85,\"height\":128,\"file_size\":2452},\"emoji\":\"dd\",\"set_name\":\"unocards\",\"file_size\":8898}]}}";
        let params = GetStickerSetParams::new("unocards".to_string());
        let _m = mockito::mock("POST", "/getStickerSet")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.get_sticker_set(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn send_media_group_success() {
        let response_string = "{\"ok\":true,\"result\":[{\"message_id\":510,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1619267462,\"chat\":{\"id\":-1001368460856,\"type\":\"supergroup\",\"title\":\"Frankenstein\"},\"media_group_id\":\"12954139699368426\",\"photo\":[{\"file_id\":\"AgACAgIAAx0EUZEOOAACAf5ghA-GtOaBIP2NOmtXdze-Un7PGgAC_q8xG0cfEUgpwpFo17XTfWTS5p8uAAMBAAMCAANtAANYQgACHwQ\",\"file_unique_id\":\"AQADZNLmny4AA1hCAAI\",\"width\":320,\"height\":320,\"file_size\":19162},{\"file_id\":\"AgACAgIAAx0EUZEOOAACAf5ghA-GtOaBIP2NOmtXdze-Un7PGgAC_q8xG0cfEUgpwpFo17XTfWTS5p8uAAMBAAMCAAN4AANZQgACHwQ\",\"file_unique_id\":\"AQADZNLmny4AA1lCAAI\",\"width\":800,\"height\":800,\"file_size\":65697},{\"file_id\":\"AgACAgIAAx0EUZEOOAACAf5ghA-GtOaBIP2NOmtXdze-Un7PGgAC_q8xG0cfEUgpwpFo17XTfWTS5p8uAAMBAAMCAAN5AANaQgACHwQ\",\"file_unique_id\":\"AQADZNLmny4AA1pCAAI\",\"width\":1146,\"height\":1146,\"file_size\":101324}]},{\"message_id\":511,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1619267462,\"chat\":{\"id\":-1001368460856,\"type\":\"supergroup\",\"title\":\"Frankenstein\"},\"media_group_id\":\"12954139699368426\",\"photo\":[{\"file_id\":\"AgACAgIAAx0EUZEOOAACAf9ghA-GeFo0B7v78UyXoOD9drjEGgAC_q8xG0cfEUgpwpFo17XTfWTS5p8uAAMBAAMCAANtAANYQgACHwQ\",\"file_unique_id\":\"AQADZNLmny4AA1hCAAI\",\"width\":320,\"height\":320,\"file_size\":19162},{\"file_id\":\"AgACAgIAAx0EUZEOOAACAf9ghA-GeFo0B7v78UyXoOD9drjEGgAC_q8xG0cfEUgpwpFo17XTfWTS5p8uAAMBAAMCAAN4AANZQgACHwQ\",\"file_unique_id\":\"AQADZNLmny4AA1lCAAI\",\"width\":800,\"height\":800,\"file_size\":65697},{\"file_id\":\"AgACAgIAAx0EUZEOOAACAf9ghA-GeFo0B7v78UyXoOD9drjEGgAC_q8xG0cfEUgpwpFo17XTfWTS5p8uAAMBAAMCAAN5AANaQgACHwQ\",\"file_unique_id\":\"AQADZNLmny4AA1pCAAI\",\"width\":1146,\"height\":1146,\"file_size\":101324}]}]}";

        let file = FileEnum::InputFileVariant(InputFile::new(std::path::PathBuf::from(
            "./frankenstein_logo.png",
        )));

        let photo = InputMediaPhoto::new(file);

        let medias = vec![
            MediaEnum::InputMediaPhotoVariant(photo.clone()),
            MediaEnum::InputMediaPhotoVariant(photo.clone()),
        ];

        let params = SendMediaGroupParams::new(ChatIdEnum::IsizeVariant(-1001368460856), medias);

        let _m = mockito::mock("POST", "/sendMediaGroup")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.send_media_group(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }

    #[test]
    fn edit_message_media_success() {
        let response_string = "{\"ok\":true,\"result\":{\"message_id\":513,\"from\":{\"id\":1276618370,\"is_bot\":true,\"first_name\":\"test_el_bot\",\"username\":\"el_mon_test_bot\"},\"date\":1619336672,\"chat\":{\"id\":-1001368460856,\"type\":\"supergroup\",\"title\":\"Frankenstein\"},\"edit_date\":1619336788,\"photo\":[{\"file_id\":\"AgACAgIAAx0EUZEOOAACAgFghR5URaBN41jx7VNgLPi29xmfQgAC_q8xG0cfEUgpwpFo17XTfWTS5p8uAAMBAAMCAANtAANYQgACHwQ\",\"file_unique_id\":\"AQADZNLmny4AA1hCAAI\",\"width\":320,\"height\":320,\"file_size\":19162},{\"file_id\":\"AgACAgIAAx0EUZEOOAACAgFghR5URaBN41jx7VNgLPi29xmfQgAC_q8xG0cfEUgpwpFo17XTfWTS5p8uAAMBAAMCAAN4AANZQgACHwQ\",\"file_unique_id\":\"AQADZNLmny4AA1lCAAI\",\"width\":800,\"height\":800,\"file_size\":65697},{\"file_id\":\"AgACAgIAAx0EUZEOOAACAgFghR5URaBN41jx7VNgLPi29xmfQgAC_q8xG0cfEUgpwpFo17XTfWTS5p8uAAMBAAMCAAN5AANaQgACHwQ\",\"file_unique_id\":\"AQADZNLmny4AA1pCAAI\",\"width\":1146,\"height\":1146,\"file_size\":101324}]}}";

        let file = FileEnum::InputFileVariant(InputFile::new(std::path::PathBuf::from(
            "./frankenstein_logo.png",
        )));

        let mut params = EditMessageMediaParams::new(InputMedia::InputMediaPhotoVariant(
            InputMediaPhoto::new(file),
        ));

        params.set_chat_id(Some(ChatIdEnum::IsizeVariant(-1001368460856)));
        params.set_message_id(Some(513));
        let _m = mockito::mock("POST", "/editMessageMedia")
            .with_status(200)
            .with_body(response_string)
            .create();
        let api = Api::new_url(mockito::server_url());

        let response = api.edit_message_media(&params).unwrap();

        let json = serde_json::to_string(&response).unwrap();
        assert_eq!(response_string, json);
    }
}
