// Copyright Fauna, Inc.
// SPDX-License-Identifier: MIT-0

use super::values::*;
use super::verbs::*;

extern crate pest;

use pest::error::Error;
use pest::Parser;

#[derive(Parser)]
#[grammar = "fql/fql.pest"]
struct FQLParser;

pub fn parse_expr(expr: &str) -> Result<FQLValue, Error<Rule>> {
  let fql = FQLParser::parse(Rule::fql, expr)?.next().unwrap();

  use pest::iterators::Pair;

  fn parse_value(pair: Pair<Rule>) -> FQLValue {
    match pair.as_rule() {
      Rule::object => FQLValue::Object(parse_object(pair).unwrap()),
      Rule::array => FQLValue::Array(pair.into_inner().map(parse_value).collect()),
      Rule::string => FQLValue::String(pair.into_inner().next().unwrap().as_str()),
      Rule::number => FQLValue::Number(pair.as_str().parse().unwrap()),
      Rule::boolean => FQLValue::Boolean(pair.as_str().parse().unwrap()),
      Rule::niladicExpr => FQLValue::NiladicVerb(parse_niladic_verb(pair).unwrap()),
      Rule::monadicExpr => FQLValue::MonadicVerb(parse_monadic_verb(pair).unwrap()),
      Rule::dyadicExpr => FQLValue::DyadicVerb(parse_dyadic_verb(pair).unwrap()),
      Rule::triadicExpr => FQLValue::TriadicVerb(parse_triadic_verb(pair).unwrap()),
      Rule::quadraticExpr => FQLValue::QuadraticVerb(parse_quadratic_verb(pair).unwrap()),
      Rule::variadicExpr => FQLValue::VariadicVerb(parse_variadic_verb(pair).unwrap()),
      Rule::null => FQLValue::Null,
      Rule::EOI
      | Rule::fql
      | Rule::pair
      | Rule::value
      | Rule::niladicVerb
      | Rule::monadicVerb
      | Rule::dyadicVerb
      | Rule::triadicVerb
      | Rule::quadraticVerb
      | Rule::variadicVerb
      | Rule::expr
      | Rule::inner
      | Rule::char
      | Rule::WHITESPACE => unreachable!(),
    }
  }

  fn parse_object(pair: Pair<Rule>) -> Result<Vec<(&str, FQLValue)>, Error<Rule>> {
    Ok(
      pair
        .into_inner()
        .map(|pair| {
          let mut inner_rules = pair.into_inner();
          let name = inner_rules
            .next()
            .unwrap()
            .into_inner()
            .next()
            .unwrap()
            .as_str();
          let value = parse_value(inner_rules.next().unwrap());
          (name, value)
        })
        .collect(),
    )
  }

  fn parse_niladic_verb(pair: Pair<Rule>) -> Result<NiladicVerb, Error<Rule>> {
    let mut inner_rules = pair.into_inner();
    let verb = inner_rules.next().unwrap().as_str();

    Ok(NiladicVerb { verb })
  }

  fn parse_monadic_verb(pair: Pair<Rule>) -> Result<MonadicVerb, Error<Rule>> {
    let mut inner_rules = pair.into_inner();
    let verb = inner_rules.next().unwrap().as_str();
    let args: Vec<FQLValue> = inner_rules.map(parse_value).collect();

    Ok(MonadicVerb { verb, args })
  }

  fn parse_dyadic_verb(pair: Pair<Rule>) -> Result<DyadicVerb, Error<Rule>> {
    let mut inner_rules = pair.into_inner();
    let verb = inner_rules.next().unwrap().as_str();
    let args: Vec<FQLValue> = inner_rules.map(parse_value).collect();

    Ok(DyadicVerb { verb, args })
  }

  fn parse_triadic_verb(pair: Pair<Rule>) -> Result<TriadicVerb, Error<Rule>> {
    let mut inner_rules = pair.into_inner();
    let verb = inner_rules.next().unwrap().as_str();
    let args: Vec<FQLValue> = inner_rules.map(parse_value).collect();

    Ok(TriadicVerb { verb, args })
  }

  fn parse_quadratic_verb(pair: Pair<Rule>) -> Result<QuadraticVerb, Error<Rule>> {
    let mut inner_rules = pair.into_inner();
    let verb = inner_rules.next().unwrap().as_str();
    let args: Vec<FQLValue> = inner_rules.map(parse_value).collect();

    Ok(QuadraticVerb { verb, args })
  }

  fn parse_variadic_verb(pair: Pair<Rule>) -> Result<VariadicVerb, Error<Rule>> {
    let mut inner_rules = pair.into_inner();
    let verb = inner_rules.next().unwrap().as_str();
    let args: Vec<FQLValue> = inner_rules.map(parse_value).collect();

    Ok(VariadicVerb { verb, args })
  }

  Ok(parse_value(fql))
}
