// Copyright Fauna, Inc.
// SPDX-License-Identifier: MIT-0

mod fql;

use wasm_bindgen::prelude::*;

use crate::fql::values;
use crate::values::FQLValue;
use crate::fql::verbs;

extern crate pest;

#[macro_use]
extern crate pest_derive;

use pest::Parser;

#[derive(Parser)]
#[grammar = "fql.pest"]
struct FQLParser;

use pest::error::Error;

fn parse_fql_query(query: &str) -> Result<FQLValue, Error<Rule>> {
    let fql = FQLParser::parse(Rule::fql, query)?.next().unwrap();

    use pest::iterators::Pair;

    fn parse_value(pair: Pair<Rule>) -> FQLValue {
        match pair.as_rule() {
            Rule::object => FQLValue::Object(
                pair.into_inner()
                    .map(|pair| {
                        let mut inner_rules = pair.into_inner();
                        let name = inner_rules
                            .next()
                            .unwrap()
                            .into_inner()
                            .next()
                            .unwrap()
                            .as_str();
                        let value = parse_value(inner_rules.next().unwrap());
                        (name, value)
                    })
                    .collect(),
            ),
            Rule::array => FQLValue::Array(pair.into_inner().map(parse_value).collect()),
            Rule::string => FQLValue::String(pair.into_inner().next().unwrap().as_str()),
            Rule::number => FQLValue::Number(pair.as_str().parse().unwrap()),
            Rule::boolean => FQLValue::Boolean(pair.as_str().parse().unwrap()),
            Rule::triadicExpr => FQLValue::TriadicVerb(parse_triadic_verb(pair).unwrap()),
            Rule::variadicExpr => FQLValue::VariadicVerb(parse_variadic_verb(pair).unwrap()),
            Rule::null => FQLValue::Null,
            Rule::EOI
            | Rule::fql
            | Rule::pair
            | Rule::value
            | Rule::triadicVerb
            | Rule::variadicVerb
            | Rule::expr
            | Rule::inner
            | Rule::char
            | Rule::WHITESPACE => unreachable!(),
        }
    }
    
    fn parse_triadic_verb(pair: Pair<Rule>) -> Result<(&str, Vec<FQLValue>), Error<Rule>> {
        let mut inner_rules = pair.into_inner();
        let verb = inner_rules.next().unwrap().as_str();
        let args = inner_rules.map(parse_value).collect();
        Ok((verb, args))
    }

    fn parse_variadic_verb(pair: Pair<Rule>) -> Result<(&str, Vec<FQLValue>), Error<Rule>> {
        let mut inner_rules = pair.into_inner();
        let verb = inner_rules.next().unwrap().as_str();
        let args = inner_rules.map(parse_value).collect();
        Ok((verb, args))
    }

    Ok(parse_value(fql))
}

fn serialize_fqlvalue(val: &FQLValue) -> String {
    use FQLValue::*;

    match val {
        Object(o) => {
            let contents: Vec<_> = o
                .iter()
                .map(|(name, value)|
                    format!("\"{}\":{}", name, serialize_fqlvalue(value)))
                .collect();
            format!("{{{}}}", contents.join(","))
        }
        Array(a) => {
            let contents: Vec<_> = a.iter().map(serialize_fqlvalue).collect();
            format!("[{}]", contents.join(","))
        }
        String(s) => format!("\"{}\"", s),
        Number(n) => format!("{}", n),
        Boolean(b) => format!("{}", b),
        TriadicVerb(v) => format!(r#"{{"{}":[{}]}}"#, verbs::parse_verb(v.0), v.1.iter().map(serialize_fqlvalue).collect::<Vec<_>>().join(",")),
        VariadicVerb(v) => format!(r#"{{"{}":[{}]}}"#, verbs::parse_verb(v.0), v.1.iter().map(serialize_fqlvalue).collect::<Vec<_>>().join(",")),
        Null => format!("null"),
    }
}

#[wasm_bindgen]
pub fn serialize_fql_query(query: &str) -> String {
    let parsed = parse_fql_query(query).unwrap();
    serialize_fqlvalue(&parsed)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn serialize_number() {
        let v: FQLValue = FQLValue::Number(2.0);

        assert_eq!(serialize_fqlvalue(&v), "2");
    }

    #[test]
    fn serialize_boolean() {
        let t: FQLValue = FQLValue::Boolean(true);
        let f: FQLValue = FQLValue::Boolean(false);

        assert_eq!(serialize_fqlvalue(&t), "true");
        assert_eq!(serialize_fqlvalue(&f), "false");
    }

    #[test]
    fn serialize_string() {
        let test_string = "This is a test string with numbers (234.042.023) and punctuation.";
        let compare_string = format!("\"{}\"", test_string);
        let v: FQLValue = FQLValue::String(test_string);

        assert_eq!(serialize_fqlvalue(&v), compare_string);
    }

    #[test]
    fn serialize_null() {
        let n: FQLValue = FQLValue::Null;

        assert_eq!(serialize_fqlvalue(&n), "null");
    }

    #[test]
    fn serialize_array() {
        let a: FQLValue = FQLValue::Array([1.0, 2.0, 3.14159].iter().map(|n| FQLValue::Number(*n)).collect());
        let compare_string = "[1,2,3.14159]";

        assert_eq!(serialize_fqlvalue(&a), compare_string);
    }
}
