use std::thread;
use std::net::IpAddr;
use std::sync::{Arc, Mutex};
use pnet::packet::ethernet::EtherTypes;
use pnet::packet::ip::IpNextHeaderProtocols;
use pnet::packet::icmp::{IcmpTypes, IcmpType};
use crate::packet::{icmp, tcp, udp, ipv4, ethernet};
use crate::base_type::{ProbeType, ProbeSetting};
use crate::packet::{ICMP_PACKET_SIZE, TCP_PACKET_SIZE, UDP_PACKET_SIZE};

fn build_tcp_probe_packet(probe_setting: &ProbeSetting, tmp_packet: &mut [u8], probe_type: ProbeType){
    // Setup Ethernet header
    let mut eth_header = pnet::packet::ethernet::MutableEthernetPacket::new(&mut tmp_packet[..ethernet::ETHERNET_HEADER_LEN]).unwrap();
    ethernet::build_ethernet_packet(&mut eth_header, probe_setting.src_mac, probe_setting.dst_mac, EtherTypes::Ipv4);
    // Setup IP header
    let mut ip_header = pnet::packet::ipv4::MutableIpv4Packet::new(&mut tmp_packet[ethernet::ETHERNET_HEADER_LEN..(ethernet::ETHERNET_HEADER_LEN + ipv4::IPV4_HEADER_LEN)]).unwrap();
    match probe_setting.src_ip {
        IpAddr::V4(src_ip) => {
            match probe_setting.dst_info.ip_addr {
                IpAddr::V4(dst_ip) => {
                    ipv4::build_ipv4_packet(&mut ip_header, src_ip, dst_ip, IpNextHeaderProtocols::Tcp);
                },
                IpAddr::V6(_ip) => {},
            }
        },
        IpAddr::V6(_ip) => {},
    }
    // Setup TCP header
    let mut tcp_header = pnet::packet::tcp::MutableTcpPacket::new(&mut tmp_packet[(ethernet::ETHERNET_HEADER_LEN + ipv4::IPV4_HEADER_LEN)..]).unwrap();
    match probe_type {
        ProbeType::TcpSynAckProbe => {
            let dst_port: u16 = *probe_setting.dst_info.open_tcp_ports.get(0).unwrap_or(&80);
            tcp::build_tcp_packet(&mut tcp_header, probe_setting.src_ip, probe_setting.src_port, probe_setting.dst_info.ip_addr, dst_port, probe_type);
        },
        ProbeType::TcpRstAckProbe => {
            tcp::build_tcp_packet(&mut tcp_header, probe_setting.src_ip, probe_setting.src_port, probe_setting.dst_info.ip_addr, probe_setting.dst_info.closed_tcp_port, probe_type);
        },
        ProbeType::TcpEcnProbe => {
            let dst_port: u16 = match probe_setting.dst_info.open_tcp_ports.get(1) {
                Some(dst_port) => dst_port.clone(),
                None => *probe_setting.dst_info.open_tcp_ports.get(0).unwrap_or(&80)
            };
            tcp::build_tcp_packet(&mut tcp_header, probe_setting.src_ip, probe_setting.src_port, probe_setting.dst_info.ip_addr, dst_port, probe_type);
        },
        _ => {
            let dst_port: u16 = *probe_setting.dst_info.open_tcp_ports.get(0).unwrap_or(&80);
            tcp::build_tcp_packet(&mut tcp_header, probe_setting.src_ip, probe_setting.src_port, probe_setting.dst_info.ip_addr, dst_port, probe_type);
        },
    }
}

fn build_icmp_probe_packet(probe_setting: &ProbeSetting, tmp_packet: &mut [u8], icmp_type: IcmpType) {
    // Setup Ethernet header
    let mut eth_header = pnet::packet::ethernet::MutableEthernetPacket::new(&mut tmp_packet[..ethernet::ETHERNET_HEADER_LEN]).unwrap();
    ethernet::build_ethernet_packet(&mut eth_header, probe_setting.src_mac, probe_setting.dst_mac, EtherTypes::Ipv4);
    // Setup IP header
    let mut ip_header = pnet::packet::ipv4::MutableIpv4Packet::new(&mut tmp_packet[ethernet::ETHERNET_HEADER_LEN..(ethernet::ETHERNET_HEADER_LEN + ipv4::IPV4_HEADER_LEN)]).unwrap();
    match probe_setting.src_ip {
        IpAddr::V4(src_ip) => {
            match probe_setting.dst_info.ip_addr {
                IpAddr::V4(dst_ip) => {
                    ipv4::build_ipv4_packet(&mut ip_header, src_ip, dst_ip, IpNextHeaderProtocols::Icmp);
                },
                IpAddr::V6(_ip) => {},
            }
        },
        IpAddr::V6(_ip) => {},
    }
    // Setup ICMP header
    match icmp_type {
        IcmpTypes::EchoRequest => {
            let mut icmp_packet = pnet::packet::icmp::echo_request::MutableEchoRequestPacket::new(&mut tmp_packet[(ethernet::ETHERNET_HEADER_LEN + ipv4::IPV4_HEADER_LEN)..]).unwrap();
            icmp::build_icmp_echo_packet(&mut icmp_packet, IcmpTypes::EchoRequest);
        },
        _ => {
            let mut icmp_packet = pnet::packet::icmp::MutableIcmpPacket::new(&mut tmp_packet[(ethernet::ETHERNET_HEADER_LEN + ipv4::IPV4_HEADER_LEN)..]).unwrap();
            icmp::build_icmp_packet(&mut icmp_packet, icmp_type);
        },
    }
}

fn build_udp_probe_packet(probe_setting: &ProbeSetting, tmp_packet: &mut [u8]) {
    // Setup Ethernet header
    let mut eth_header = pnet::packet::ethernet::MutableEthernetPacket::new(&mut tmp_packet[..ethernet::ETHERNET_HEADER_LEN]).unwrap();
    ethernet::build_ethernet_packet(&mut eth_header, probe_setting.src_mac, probe_setting.dst_mac, EtherTypes::Ipv4);
    // Setup IP header
    let mut ip_header = pnet::packet::ipv4::MutableIpv4Packet::new(&mut tmp_packet[ethernet::ETHERNET_HEADER_LEN..(ethernet::ETHERNET_HEADER_LEN + ipv4::IPV4_HEADER_LEN)]).unwrap();
    match probe_setting.src_ip {
        IpAddr::V4(src_ip) => {
            match probe_setting.dst_info.ip_addr {
                IpAddr::V4(dst_ip) => {
                    ipv4::build_ipv4_packet(&mut ip_header, src_ip, dst_ip, IpNextHeaderProtocols::Udp);
                },
                IpAddr::V6(_ip) => {},
            }
        },
        IpAddr::V6(_ip) => {},
    }
    // Setup UDP header
    let mut udp_header = pnet::packet::udp::MutableUdpPacket::new(&mut tmp_packet[(ethernet::ETHERNET_HEADER_LEN + ipv4::IPV4_HEADER_LEN)..]).unwrap();
    udp::build_udp_packet(&mut udp_header, probe_setting.src_ip, probe_setting.src_port, probe_setting.dst_info.ip_addr, probe_setting.dst_info.closed_udp_port);
}

pub fn send_packets(tx: &mut Box<dyn pnet::datalink::DataLinkSender>, probe_setting: &ProbeSetting, stop: &Arc<Mutex<bool>>) {
    for probe_type in probe_setting.probe_types.clone() {
        match probe_type {
            ProbeType::IcmpEchoProbe => {
                tx.build_and_send(1, ICMP_PACKET_SIZE, &mut |packet: &mut [u8]| {
                    build_icmp_probe_packet(probe_setting, packet, IcmpTypes::EchoRequest);
                });
            },
            ProbeType::IcmpTimestampProbe => {
                tx.build_and_send(1, ICMP_PACKET_SIZE, &mut |packet: &mut [u8]| {
                    build_icmp_probe_packet(probe_setting, packet, IcmpTypes::Timestamp);
                });
            },
            ProbeType::IcmpAddressMaskProbe => {
                tx.build_and_send(1, ICMP_PACKET_SIZE, &mut |packet: &mut [u8]| {
                    build_icmp_probe_packet(probe_setting, packet, IcmpTypes::AddressMaskRequest);
                });
            },
            ProbeType::IcmpInformationProbe => {
                tx.build_and_send(1, ICMP_PACKET_SIZE, &mut |packet: &mut [u8]| {
                    build_icmp_probe_packet(probe_setting, packet, IcmpTypes::InformationRequest);
                });
            },
            ProbeType::IcmpUnreachableProbe => {
                tx.build_and_send(1, UDP_PACKET_SIZE, &mut |packet: &mut [u8]| {
                    build_udp_probe_packet(probe_setting, packet);
                });
            },
            ProbeType::TcpSynAckProbe => {
                tx.build_and_send(1, TCP_PACKET_SIZE, &mut |packet: &mut [u8]| {
                    build_tcp_probe_packet(probe_setting, packet, probe_type);
                });
            },
            ProbeType::TcpRstAckProbe => {
                tx.build_and_send(1, TCP_PACKET_SIZE, &mut |packet: &mut [u8]| {
                    build_tcp_probe_packet(probe_setting, packet, probe_type);
                });
            },
            ProbeType::TcpEcnProbe => {
                tx.build_and_send(1, TCP_PACKET_SIZE, &mut |packet: &mut [u8]| {
                    build_tcp_probe_packet(probe_setting, packet, probe_type);
                });
            },
        }
    }
    thread::sleep(probe_setting.wait_time);
    *stop.lock().unwrap() = true;
}