use crate::{
    functions::FunctionList,
    types::{CargoDependency, Enum, Field, Struct, Type, TypeIdent, TypeMap},
    RustPluginConfig,
};
use std::{
    collections::{BTreeMap, BTreeSet},
    fs,
};

pub fn generate_bindings(
    import_functions: FunctionList,
    export_functions: FunctionList,
    types: TypeMap,
    config: RustPluginConfig,
    path: &str,
) {
    let src_path = format!("{}/src", path);
    fs::create_dir_all(&src_path).expect("Could not create output directory");

    generate_cargo_file(config, &import_functions, &types, path);

    generate_type_bindings(&types, &src_path, "rust_plugin");
    generate_imported_function_bindings(import_functions, &types, &src_path);
    generate_exported_function_bindings(export_functions, &types, &src_path);

    write_bindings_file(
        format!("{}/lib.rs", src_path),
        "#[rustfmt::skip]
mod export;
#[rustfmt::skip]
mod import;
#[rustfmt::skip]
mod types;

pub use export::*;
pub use import::*;
pub use types::*;

pub use fp_bindgen_support::*;
",
    );
}

fn generate_cargo_file(
    config: RustPluginConfig,
    import_functions: &FunctionList,
    types: &TypeMap,
    path: &str,
) {
    let requires_async = import_functions.iter().any(|function| function.is_async);

    let mut support_features = BTreeSet::from(["guest"]);
    if requires_async {
        support_features.insert("async");
    }

    let mut dependencies = BTreeMap::from([
        (
            "fp-bindgen-support",
            CargoDependency {
                version: Some(env!("CARGO_PKG_VERSION")),
                features: support_features,
                ..CargoDependency::default()
            },
        ),
        ("once_cell", CargoDependency::with_version("1.4")),
        ("rmp-serde", CargoDependency::with_version("1.0")),
        (
            "serde",
            CargoDependency::with_version_and_features("1.0", BTreeSet::from(["derive"])),
        ),
    ]);

    // Inject dependencies from custom types:
    for ty in types.values() {
        if let Type::Custom(custom_type) = ty {
            for (name, dependency) in custom_type.rs_dependencies.iter() {
                let dependency = if let Some(existing_dependency) = dependencies.remove(name) {
                    existing_dependency.merge_or_replace_with(dependency)
                } else {
                    dependency.clone()
                };
                dependencies.insert(name, dependency);
            }
        }
    }

    // Inject dependencies passed through the config:
    for (name, dependency) in config.dependencies {
        let dependency = if let Some(existing_dependency) = dependencies.remove(name) {
            existing_dependency.merge_or_replace_with(&dependency)
        } else {
            dependency.clone()
        };
        dependencies.insert(name, dependency);
    }

    write_bindings_file(
        format!("{}/Cargo.toml", path),
        format!(
            "[package]
name = \"{}\"
version = \"{}\"
authors = {}
edition = \"2018\"

[dependencies]
{}
",
            config.name,
            config.version,
            config.authors,
            dependencies
                .iter()
                .map(|(name, value)| format!("{} = {}", name, value))
                .collect::<Vec<_>>()
                .join("\n")
        ),
    );
}

pub fn generate_type_bindings(types: &TypeMap, path: &str, module_key: &str) {
    let std_types = types
        .values()
        .filter_map(collect_std_types)
        .collect::<BTreeSet<_>>();
    let std_imports = if std_types.is_empty() {
        "".to_owned()
    } else if std_types.len() == 1 {
        format!("use std::{};\n", std_types.iter().next().unwrap())
    } else {
        format!(
            "use std::{{{}}};\n",
            std_types.into_iter().collect::<Vec<_>>().join(", ")
        )
    };

    let type_imports = types
        .values()
        .filter_map(|ty| {
            let (ident, native_modules) = match ty {
                Type::Enum(Enum { ident, options, .. }) => (ident, &options.native_modules),
                Type::Struct(Struct { ident, options, .. }) => (ident, &options.native_modules),
                _ => return None,
            };
            native_modules
                .get(module_key)
                .map(|module| format!("pub use {}::{};", module, ident.name))
        })
        .collect::<Vec<_>>();
    let type_imports = if type_imports.is_empty() {
        "".to_owned()
    } else {
        format!("{}\n\n", type_imports.join("\n"))
    };

    let type_defs = types
        .values()
        .filter_map(|ty| match ty {
            Type::Alias(name, ty) => {
                Some(format!("pub type {} = {};", name, format_ident(ty, types)))
            }
            Type::Enum(ty) => {
                if ty.options.native_modules.contains_key(module_key) || ty.ident.name == "Result" {
                    None
                } else {
                    Some(create_enum_definition(ty, types))
                }
            }
            Type::Struct(ty) => {
                if ty.options.native_modules.contains_key(module_key) {
                    None
                } else {
                    Some(create_struct_definition(ty, types))
                }
            }
            _ => None,
        })
        .collect::<Vec<_>>();

    write_bindings_file(
        format!("{}/types.rs", path),
        format!(
            "use serde::{{Deserialize, Serialize}};\n{}\n{}{}\n",
            std_imports,
            type_imports,
            type_defs.join("\n\n")
        ),
    );
}

fn format_functions(export_functions: FunctionList, types: &TypeMap, macro_path: &str) -> String {
    export_functions
        .iter()
        .map(|func| {
            let name = &func.name;
            let doc = func
                .doc_lines
                .iter()
                .map(|line| format!("///{}\n", line))
                .collect::<Vec<_>>()
                .join("");
            let modifiers = if func.is_async { "async " } else { "" };
            let args_with_types = func
                .args
                .iter()
                .map(|arg| format!("{}: {}", arg.name, format_ident(&arg.ty, types)))
                .collect::<Vec<_>>()
                .join(", ");
            let return_type = match &func.return_type {
                Some(ty) => format!(" -> {}", format_ident(ty, types)),
                None => "".to_owned(),
            };
            format!(
                "#[{}]\n{}pub {}fn {}({}){};",
                macro_path, doc, modifiers, name, args_with_types, return_type,
            )
        })
        .collect::<Vec<_>>()
        .join("\n\n")
}

fn format_ident(ident: &TypeIdent, types: &TypeMap) -> String {
    match types.get(ident) {
        Some(ty) => format_type_with_ident(ty, ident, types),
        None => ident.to_string(), // Must be a generic.
    }
}

fn format_type_with_ident(ty: &Type, ident: &TypeIdent, types: &TypeMap) -> String {
    match ty {
        Type::Alias(name, _) => name.clone(),
        Type::Container(name, _) | Type::List(name, _) => {
            let arg = ident
                .generic_args
                .first()
                .expect("Identifier was expected to contain a generic argument");
            format!("{}<{}>", name, format_ident(arg, types))
        }
        Type::Custom(custom) => custom.rs_ty.clone(),
        Type::Map(name, _, _) => {
            let arg1 = ident
                .generic_args
                .first()
                .expect("Identifier was expected to contain a generic argument");
            let arg2 = ident
                .generic_args
                .get(1)
                .expect("Identifier was expected to contain two arguments");
            format!(
                "{}<{}, {}>",
                name,
                format_ident(arg1, types),
                format_ident(arg2, types)
            )
        }
        Type::Tuple(items) => format!(
            "[{}]",
            items
                .iter()
                .map(|item| format_ident(item, types))
                .collect::<Vec<_>>()
                .join(", ")
        ),
        Type::Unit => "void".to_owned(),
        _ => ident.to_string(),
    }
}

fn generate_imported_function_bindings(
    import_functions: FunctionList,
    types: &TypeMap,
    path: &str,
) {
    write_bindings_file(
        format!("{}/import.rs", path),
        format!(
            "use crate::types::*;\n\n{}\n",
            format_functions(
                import_functions,
                types,
                "fp_bindgen_support::fp_import_signature"
            )
        ),
    );
}

fn generate_exported_function_bindings(
    export_functions: FunctionList,
    types: &TypeMap,
    path: &str,
) {
    write_bindings_file(
        format!("{}/export.rs", path),
        format!(
            "use crate::types::*;\n\n{}\n",
            format_functions(
                export_functions,
                types,
                "fp_bindgen_support::fp_export_signature"
            )
        ),
    );
}

fn collect_std_types(ty: &Type) -> Option<String> {
    match ty {
        Type::Container(name, _) if name == "Rc" => Some("rc::Rc".to_owned()),
        Type::List(name, _) if (name == "BTreeSet" || name == "HashSet") => {
            Some(format!("collections::{}", name))
        }
        Type::Map(name, _, _) if (name == "BTreeMap" || name == "HashMap") => {
            Some(format!("collections::{}", name))
        }
        _ => None,
    }
}

fn create_enum_definition(ty: &Enum, types: &TypeMap) -> String {
    let variants = ty
        .variants
        .iter()
        .flat_map(|variant| {
            let mut serde_attrs = variant.attrs.to_serde_attrs();
            let mut variant_decl = match &variant.ty {
                Type::Unit => format!("{},", variant.name),
                Type::Struct(variant) => {
                    let fields = format_struct_fields(&variant.fields, types);
                    let has_multiple_lines = fields.iter().any(|field| field.contains('\n'));
                    let fields = if has_multiple_lines {
                        format!(
                            "\n{}\n",
                            fields
                                .iter()
                                .flat_map(|field| field.split('\n'))
                                .map(|line| if line.is_empty() {
                                    line.to_owned()
                                } else {
                                    format!("    {}", line)
                                })
                                .collect::<Vec<_>>()
                                .join("\n")
                                .trim_start_matches('\n'),
                        )
                    } else {
                        let fields = fields.join(" ");
                        format!(" {} ", &fields.trim_end_matches(','))
                    };
                    format!("{} {{{}}},", variant.ident.name, fields)
                }
                Type::Tuple(items) => {
                    let items = items
                        .iter()
                        .map(|item| format_ident(item, types))
                        .collect::<Vec<_>>()
                        .join(", ");
                    format!("{}({}),", variant.name, items)
                }
                other => panic!("Unsupported type for enum variant: {:?}", other),
            };

            if !serde_attrs.is_empty() {
                serde_attrs.sort();
                variant_decl = format!("#[serde({})]\n{}", serde_attrs.join(", "), variant_decl);
            }

            let lines = if variant.doc_lines.is_empty() {
                variant_decl
                    .split('\n')
                    .map(str::to_owned)
                    .collect::<Vec<_>>()
            } else {
                let mut lines = format_docs(&variant.doc_lines)
                    .trim_end_matches('\n')
                    .split('\n')
                    .map(str::to_owned)
                    .collect::<Vec<_>>();
                lines.append(
                    &mut variant_decl
                        .split('\n')
                        .map(str::to_owned)
                        .collect::<Vec<_>>(),
                );
                lines
            };

            lines
                .iter()
                .map(|line| {
                    if line.is_empty() {
                        line.clone()
                    } else {
                        format!("    {}", line)
                    }
                })
                .collect::<Vec<_>>()
        })
        .collect::<Vec<_>>()
        .join("\n");

    let serde_annotation = {
        let attrs = ty.options.to_serde_attrs();
        if attrs.is_empty() {
            "".to_owned()
        } else {
            format!("#[serde({})]\n", attrs.join(", "))
        }
    };

    format!(
        "{}#[derive(Clone, Debug, Deserialize, PartialEq, Serialize)]\n{}\
        pub enum {} {{\n\
            {}\n\
        }}",
        format_docs(&ty.doc_lines),
        serde_annotation,
        ty.ident,
        variants
    )
}

fn create_struct_definition(ty: &Struct, types: &TypeMap) -> String {
    let fields = format_struct_fields(&ty.fields, types)
        .iter()
        .flat_map(|field| field.split('\n'))
        .map(|line| {
            if line.is_empty() {
                line.to_owned()
            } else {
                format!(
                    "    {}{}",
                    if line.starts_with('#') || line.starts_with("///") {
                        ""
                    } else {
                        "pub "
                    },
                    line
                )
            }
        })
        .collect::<Vec<_>>()
        .join("\n");

    let serde_annotation = {
        let attrs = ty.options.to_serde_attrs();
        if attrs.is_empty() {
            "".to_owned()
        } else {
            format!("#[serde({})]\n", attrs.join(", "))
        }
    };

    format!(
        "{}#[derive(Clone, Debug, Deserialize, PartialEq, Serialize)]\n{}\
        pub struct {} {{\n\
            {}\n\
        }}",
        format_docs(&ty.doc_lines),
        serde_annotation,
        ty.ident,
        fields.trim_start_matches('\n')
    )
}

fn format_docs(doc_lines: &[String]) -> String {
    doc_lines
        .iter()
        .map(|line| format!("///{}\n", line))
        .collect::<Vec<_>>()
        .join("")
}

fn format_struct_fields(fields: &[Field], types: &TypeMap) -> Vec<String> {
    fields
        .iter()
        .map(|field| {
            let mut serde_attrs = field.attrs.to_serde_attrs();

            match types.get(&field.ty) {
                Some(Type::Container(name, _)) if name == "Option" => {
                    if !serde_attrs
                        .iter()
                        .any(|attr| attr == "default" || attr.starts_with("default = "))
                    {
                        serde_attrs.push("default".to_owned());
                    }
                    if !serde_attrs
                        .iter()
                        .any(|attr| attr.starts_with("skip_serializing_if ="))
                    {
                        serde_attrs.push("skip_serializing_if = \"Option::is_none\"".to_owned());
                    }
                }
                Some(Type::Custom(custom_type)) => {
                    for attr in custom_type.serde_attrs.iter() {
                        serde_attrs.push(attr.clone());
                    }
                }
                _ => {}
            }

            let docs = if field.doc_lines.is_empty() {
                "".to_owned()
            } else {
                format!(
                    "\n{}",
                    field
                        .doc_lines
                        .iter()
                        .map(|line| format!("///{}\n", line))
                        .collect::<Vec<_>>()
                        .join("")
                )
            };

            let annotations = if serde_attrs.is_empty() {
                "".to_owned()
            } else {
                serde_attrs.sort();
                format!("#[serde({})]\n", serde_attrs.join(", "))
            };

            format!(
                "{}{}{}: {},",
                docs,
                annotations,
                field.name,
                format_ident(&field.ty, types)
            )
        })
        .collect()
}

fn write_bindings_file<C>(file_path: String, contents: C)
where
    C: AsRef<[u8]>,
{
    fs::write(&file_path, &contents).expect("Could not write bindings file");
}
