# forsyth

A pure Rust implementation of Tom Forsyth's [Linear-Speed Vertex Cache Optimisation](https://tomforsyth1000.github.io/papers/fast_vert_cache_opt.html).

[![pipeline status](https://gitlab.com/kento_asashima/forsyth/badges/master/pipeline.svg)](https://gitlab.com/kento_asashima/forsyth/-/commits/master)

 Two algorithms are provided in this crate.
 - `order_triangles_inplace` and `order_triangles` to order triangle indices to maximize data locality.
 - `order_vertices` to order vertex data in such an way, that vertex data locality is maximized when iterating sequentially through index data.

 Both algorithms can be run independently, but ordering indices first and then ordering vertices provides most benefits.

 Note that [`Kerbel et al. 2018`] argued that [`GPU caching may not benefit from such ordering`].
 However, there may be use cases that benefit from improved data locality when sequentially processing index and vertex data,
 such as when streaming data from persistent storage or when processing geometry with CPUs.

 Despite the original paper's title, the algorithm is [`not guaranteed to be exactly linear`].
 There are pathological cases where runtime can be worse, especially when  there are many vertices with many connected edges (ie. high valence).
 Meshes mostly containing very fine-grained triangle fans are an example. However, one can still expect a throughput of hundreds of thousand indices per second on contemporary CPUs even for these cases.

 In practice, both algorithms are fast enough to opportunistically apply them to geometry to be processed or read multiple times.
 Apart from data locality, both algorithms may be useful to improve subsequent compression by producing more contiguous data useful for delta compression and other algorithms.

 ```
 use forsyth::{order_vertices,order_triangles};

 let input_vertices = &['a', 'b', 'c', 'd', 'e'];
 let input_indices = &[0_u32, 1, 2, 0, 1, 3, 0, 3, 4, 2, 1, 4];

 // order indices first
 let ordered_indices = order_triangles(input_indices, 32).unwrap();

 // then order vertices and remap indices accordingly
 let (ordered_vertices, ordered_indices) =
     order_vertices(&input_vertices[..], ordered_indices.as_slice()).unwrap();

 assert_eq!(&ordered_vertices, &['a', 'd', 'e', 'b', 'c']);
 assert_eq!(&ordered_indices, &[0, 1, 2, 0, 3, 1, 4, 3, 2, 0, 3, 4]);
 ```

This crate is made using [`gitlab CI`]:
  - it is built and tested with rust versions from [`1.34 to latest`]
  - [`test coverage reporting`] is provided by [`cargo-tarpaulin`]
  - [`fuzzing`] is provided by [`cargo-fuzz`]
  - [`benchmarking`] is provided by [`criterion`]

 [`Linear-speed Vertex Cache Optimisation`]: https://tomforsyth1000.github.io/papers/fast_vert_cache_opt.html
 [`Kerbel et al. 2018`]: https://arbook.icg.tugraz.at/schmalstieg/Schmalstieg_351.pdf
 [`GPU caching may not benefit from such ordering`]: https://www.highperformancegraphics.org/wp-content/uploads/2018/Papers-Session2/HPG2018_RevisitingVertexCache.pdf
 [`not guaranteed to be exactly linear`]: https://kento_asashima.gitlab.io/-/forsyth/-/jobs/1192073834/artifacts/target/criterion/order_triangles/report/index.html
 [`test coverage reporting`]: https://kento_asashima.gitlab.io/-/forsyth/-/jobs/1421605715/artifacts/cov/tarpaulin-report.html#src/lib.rs
 [`gitlab CI`]: https://gitlab.com/kento_asashima/forsyth/-/pipelines/336386578
 [`cargo-tarpaulin`]: https://crates.io/crates/cargo-tarpaulin
 [`1.34 to latest`]: https://gitlab.com/kento_asashima/forsyth/-/jobs/1421605710
 [`fuzzing`]: https://gitlab.com/kento_asashima/forsyth/-/jobs/1421605714
 [`cargo-fuzz`]: https://github.com/rust-fuzz/cargo-fuzz
 [`benchmarking`]: https://kento_asashima.gitlab.io/-/forsyth/-/jobs/1421605712/artifacts/target/criterion/report/index.html
 [`criterion`]: https://crates.io/crates/criterion

# Benchmarking and Test Models

Models for benchmarking can be found in Morgan McGuire's Computer Graphics Archive https://casual-effects.com/data