use std::convert::TryInto;

extern crate tobj;

#[allow(unused)]
pub(crate) fn generate_mesh_indices(obj_file: &str) -> Vec<u32> {
    let (models, _) = tobj::load_obj(&obj_file, true).expect("Failed to load obj");

    let mut num_ignored_indices = 0;
    let mut indices = Vec::with_capacity(4096);

    for m in models.iter() {
        let mesh = &m.mesh;

        indices.reserve(mesh.num_face_indices.len() * 3);

        let mut next_face = 0;
        for f in 0..mesh.num_face_indices.len() {
            let num_verts = mesh.num_face_indices[f];
            let end = next_face + num_verts as usize;
            if num_verts == 3 {
                indices.extend_from_slice(&mesh.indices[next_face..end]);
            } else {
                num_ignored_indices += 1;
            }
            next_face = end;
        }
    }

    println!(
        "Generated {} (+{}) indices from {}",
        indices.len(),
        num_ignored_indices,
        obj_file
    );

    indices
}

#[allow(unused)]
pub(crate) fn synthesize_vertex_buffer<Vertex, Index, Callback>(
    indices: &[Index],
    generator: Callback,
) -> Vec<Vertex>
where
    Callback: Fn(usize) -> Vertex,
    Index: Copy + TryInto<usize>,
{
    if indices.is_empty() {
        return Vec::new();
    }

    let max = indices
        .iter()
        .map(|idx| (*idx).try_into().unwrap_or(0))
        .max()
        .unwrap_or(0);
    if max == 0 {
        return Vec::new();
    }

    let mut vertices = Vec::with_capacity(max + 1);
    for v in 0..max + 1 {
        vertices.push(generator(v));
    }
    vertices
}

#[cfg(test)]
mod tests {

    #[test]
    fn vertex_generation() {
        {
            assert_eq!(
                crate::synthesize_vertex_buffer(&[0, 1, 2, 3], |v| 1000 + v),
                [1000, 1001, 1002, 1003]
            );
        }
    }
}
