use proc_macro2::{Group, Ident, Span, TokenStream, TokenTree};
use quote::{quote, quote_spanned, ToTokens, TokenStreamExt};
use syn::{parse2, punctuated::Punctuated, token::Comma, FnArg, ItemFn};

#[derive(Default)]
struct FnAttrs {
    ensures: Vec<TokenStream>,
    requires: Vec<TokenStream>,
    proof: bool,
    user_attrs: Vec<syn::Attribute>,
}

struct FnMetadata {
    attrs: FnAttrs,
    name: Ident,
    params: Punctuated<FnArg, Comma>,
    ty: syn::ReturnType,
    body: syn::Block,
}

struct MyError {
    span: Span,
    text: String,
}

impl From<syn::Error> for MyError {
    fn from(x: syn::Error) -> Self {
        Self {
            span: x.span(),
            text: x.to_string(),
        }
    }
}

type Result<T> = std::result::Result<T, MyError>;

fn parse_our_special(tokens: TokenStream, ctx: TokenStream) -> Result<TokenStream> {
    let mut r = TokenStream::default();
    let mut iter = tokens.into_iter();
    while let Some(token) = iter.next() {
        match token {
            TokenTree::Group(x) => {
                let mut y = TokenTree::Group(Group::new(
                    x.delimiter(),
                    parse_our_special(x.stream(), ctx.clone())?,
                ));
                y.set_span(x.span());
                r.append(y);
            }
            TokenTree::Ident(_) => r.append(token),
            TokenTree::Punct(x) if x.to_string() == "$" => {
                let fun = iter.next().unwrap();
                match fun.to_string().as_str() {
                    "old" => {}
                    "result" => r.extend(quote_spanned! { fun.span() => __FORMAL_SPEC_result }),
                    "forall" | "exists" => {
                        let abs_token = iter.next().unwrap();
                        if let TokenTree::Group(g) = &abs_token {
                            let mut iter = g.stream().into_iter();
                            match iter.next() {
                                Some(TokenTree::Punct(p)) => {
                                    if p.to_string() != "|" {
                                        return Err(MyError {
                                            span: abs_token.span(),
                                            text: "Expected closure".to_string(),
                                        });
                                    }
                                }
                                _ => {
                                    return Err(MyError {
                                        span: abs_token.span(),
                                        text: "Expected closure".to_string(),
                                    })
                                }
                            }
                            let mut args = quote! { #ctx , };
                            loop {
                                match iter.next() {
                                    Some(TokenTree::Punct(p)) if p.to_string() == "|" => {
                                        break;
                                    }
                                    Some(x) => {
                                        args.append(x);
                                    }
                                    None => {
                                        return Err(MyError {
                                            span: abs_token.span(),
                                            text: "Expected closure".to_string(),
                                        })
                                    }
                                }
                            }
                            let exp = parse_our_special(iter.collect(), args.clone())?;
                            dbg!(exp.clone());
                            r.extend(quote! {
                                {
                                    const fn __FORMAL_SPEC_fn( #args ) -> bool {
                                        return #exp ;
                                    };
                                    true
                                }
                            })
                        } else {
                            return Err(MyError {
                                span: abs_token.span(),
                                text: "Expected paranthesis".to_string(),
                            });
                        };
                    }
                    _ => {
                        return Err(MyError {
                            span: fun.span(),
                            text: "Unexpected syntax".to_string(),
                        });
                    }
                }
            }
            TokenTree::Punct(_) => r.append(token),
            TokenTree::Literal(_) => r.append(token),
        }
    }
    Ok(r)
}

impl FnMetadata {
    fn fix_args(&self) -> Punctuated<FnArg, Comma> {
        let mut r = self.params.clone();
        for x in r.iter_mut() {
            match x {
                FnArg::Receiver(_) => todo!(),
                FnArg::Typed(x) => {
                    fix_type(&mut x.ty);
                }
            }
        }
        r
    }

    fn generate_doc(&self) -> String {
        let mut r = " # Specification\n\n".to_string();
        if !self.attrs.requires.is_empty() {
            r += " ## Requires\n\n";
            for stream in &self.attrs.requires {
                r += &format!("* `{}`\n", stream);
            }
        }
        if !self.attrs.ensures.is_empty() {
            r += " ## Ensures\n\n";
            for stream in &self.attrs.ensures {
                r += &format!("* `{}`\n", stream);
            }
        }
        r
    }

    fn generate_intg(&self) -> Result<TokenStream> {
        let mut r = proc_macro2::TokenStream::default();
        for item in &self.attrs.requires {
            let params = self.fix_args().to_token_stream();
            let item = parse_our_special(item.clone(), params.clone())?;
            r.extend(quote! {
                #[allow(unused_variables)]
                #[allow(unused_parens)]
                const _: () = {
                    const fn __FORMAL_SPEC_fn( #params ) -> bool {
                        return #item ;
                    }
                };
            });
        }
        for item in &self.attrs.ensures {
            let mut params = self.fix_args().to_token_stream();
            if let syn::ReturnType::Type(_, ty) = &self.ty {
                params = quote! { #params, __FORMAL_SPEC_result: #ty };
            }
            let item = parse_our_special(item.clone(), params.clone())?;
            r.extend(quote! {
                #[allow(unused_variables)]
                #[allow(unused_parens)]
                const _: () = {
                    const fn __FORMAL_SPEC_fn( #params ) -> bool {
                        return #item ;
                    }
                };
            });
        }
        Ok(r)
    }

    fn generate_tokens(self) -> TokenStream {
        let docs = self.generate_doc();
        let intg = match self.generate_intg() {
            Ok(x) => x,
            Err(x) => return x.emit(),
        };
        let Self {
            name, params, body, ..
        } = self;
        let ty = self.ty.to_token_stream();
        quote! {
            #[doc=#docs]
            fn #name ( #params ) #ty #body

            #intg
        }
    }
}

fn fix_type(ty: &mut syn::Type) {
    if let syn::Type::Reference(x) = ty {
        x.mutability = None;
    }
}

#[allow(clippy::unnecessary_filter_map)]
fn parse_fn(item: TokenStream) -> FnMetadata {
    let parsed = parse2::<ItemFn>(item).unwrap();
    let mut attrs = FnAttrs::default();
    attrs.user_attrs = parsed
        .attrs
        .into_iter()
        .filter_map(|attr| {
            match attr.path.to_token_stream().to_string().as_str() {
                "requires" => attrs.requires.push(attr.tokens),
                "ensures" => attrs.ensures.push(attr.tokens),
                "proof" => attrs.proof = true,
                _ => return Some(attr),
            }
            None
        })
        .collect();
    FnMetadata {
        attrs,
        name: parsed.sig.ident,
        params: parsed.sig.inputs,
        ty: parsed.sig.output,
        body: *parsed.block,
    }
}

/*
fn parse_fn(item: TokenStream) -> FnMetadata {
    let mut attrs = FnAttrs::default();
    let mut iter = item.into_iter();
    while let Some(token) = iter.next() {
        match &token {
            TokenTree::Punct(p) => {
                if p.to_string() == "#" {
                    let body = iter.next().unwrap();
                    match body {
                        TokenTree::Group(body) => parse_attr(body.stream(), &mut attrs),
                        _ => panic!(),
                    }
                }
            }
            TokenTree::Ident(i) => {
                if i == "fn" {
                    let name = iter.next().unwrap();
                    let params = if let Some(TokenTree::Group(g)) = iter.next() {
                        g
                    } else {
                        panic!("Function needs some arguments");
                    };
                    let mut iter = iter.peekable();
                    let ty = match iter.peek() {
                        Some(TokenTree::Punct(x)) if x.to_string() == "-" => {
                            iter.next();
                            iter.next();
                            iter.next()
                        }
                        _ => None,
                    };
                    return FnMetadata {
                        attrs,
                        name,
                        params,
                        ty,
                        body: iter.next().unwrap(),
                    };
                }
            }
            _ => (),
        }
    }
    panic!("Attribute only should be on functions");
}
*/

fn restore_and_parse(name: &str, args: TokenStream, item: TokenStream) -> TokenStream {
    let name: TokenStream = name.parse().unwrap();
    let attr: TokenStream = if args.is_empty() {
        format!("#[{}]", name).parse().unwrap()
    } else {
        quote! { #[#name ( #args )] }
    };
    let body = quote! { #attr #item };
    parse_fn(body).generate_tokens()
}

#[proc_macro_attribute]
pub fn requires(
    args: proc_macro::TokenStream,
    item: proc_macro::TokenStream,
) -> proc_macro::TokenStream {
    restore_and_parse("requires", args.into(), item.into()).into()
}

#[proc_macro_attribute]
pub fn ensures(
    args: proc_macro::TokenStream,
    item: proc_macro::TokenStream,
) -> proc_macro::TokenStream {
    restore_and_parse("ensures", args.into(), item.into()).into()
}

#[proc_macro_attribute]
pub fn proof(
    args: proc_macro::TokenStream,
    item: proc_macro::TokenStream,
) -> proc_macro::TokenStream {
    restore_and_parse("proof", args.into(), item.into()).into()
}

impl MyError {
    pub(crate) fn emit(&self) -> TokenStream {
        let text = &self.text;
        quote_spanned! { self.span => compile_error!(#text); }
    }
}
