use std::fs;
use std::io::Write;
use std::process::Command;

use log::{debug, warn};

pub fn bindgen_rust_to_dart(
    rust_crate_dir: &str,
    c_output_path: &str,
    dart_output_path: &str,
    dart_class_name: &str,
    c_struct_names: Vec<String>,
) {
    cbindgen(rust_crate_dir, c_output_path, c_struct_names);
    ffigen(c_output_path, dart_output_path, dart_class_name);
}

fn execute_command(command: &mut Command) {
    debug!("execute command: {:?}", command);

    let result = command.output().unwrap();

    if result.status.success() {
        debug!("command={:?} output={:?}", command, result);
    } else if String::from_utf8_lossy(&result.stdout).contains("fatal error") {
        warn!(
            "See keywords such as `error` in command output. Maybe there is a problem? command={:?} output={:?}",
            command, result
        );
        // only warn, do not panic
    } else {
        warn!("command={:?} output={:?}", command, result);
        panic!("command execution failed. command={:?}", command);
    }
}

fn cbindgen(rust_crate_dir: &str, c_output_path: &str, c_struct_names: Vec<String>) {
    debug!(
        "execute cbindgen rust_crate_dir={} c_output_path={}",
        rust_crate_dir, c_output_path
    );

    let config = format!(
        r#"
language = "C"
[export]
include = [{}]
"#,
        c_struct_names
            .iter()
            .map(|name| format!("\"{}\"", name))
            .collect::<Vec<_>>()
            .join(", ")
    );
    debug!("cbindgen config: {}", config);

    let mut config_file = tempfile::NamedTempFile::new().unwrap();
    config_file.write_all(config.as_bytes()).unwrap();
    debug!("cbindgen config_file: {:?}", config_file);

    execute_command(
        Command::new("cbindgen")
            .arg("-v")
            .arg("--config")
            .arg(config_file.path())
            .arg("--output")
            .arg(c_output_path)
            .current_dir(fs::canonicalize(rust_crate_dir).unwrap()),
    );
}

fn ffigen(c_path: &str, dart_path: &str, dart_class_name: &str) {
    debug!("execute ffigen c_path={} dart_path={}", c_path, dart_path);
    let config = format!(
        "
        output: '{}'
        name: '{}'
        description: 'generated by flutter_rust_bridge'
        headers:
          entry-points:
            - '{}'
          include-directives:
            - '{}'
        comments: false
        preamble: |
          // ignore_for_file: camel_case_types, non_constant_identifier_names, avoid_positional_boolean_parameters, annotate_overrides
        ",
        dart_path, dart_class_name, c_path, c_path,
    );
    debug!("ffigen config: {}", config);

    let mut config_file = tempfile::NamedTempFile::new().unwrap();
    config_file.write_all(config.as_bytes()).unwrap();
    debug!("ffigen config_file: {:?}", config_file);

    // NOTE please install ffigen globally first: `dart pub global activate ffigen`
    execute_command(
        Command::new("dart")
            .arg("pub")
            .arg("global")
            .arg("run")
            .arg("ffigen")
            .arg("--config")
            .arg(config_file.path()),
    );
}

pub fn format_rust(path: &str) {
    debug!("execute format_rust path={}", path);
    execute_command(Command::new("rustfmt").arg(path));
}

pub fn format_dart(path: &str, line_length: i32) {
    debug!(
        "execute format_dart path={} line_length={}",
        path, line_length
    );
    execute_command(
        Command::new("dart")
            .arg("format")
            .arg(path)
            .arg("--line-length")
            .arg(line_length.to_string()),
    );
}
