/// Gives the minimum of two values.
///
/// * In the case of floats, if one value is NaN then the non-NaN value is
///   returned.
pub trait Min {
  /// Gets the minimum value.
  fn min(self, other: Self) -> Self;
}

/// Calls the [min](Min::min) trait method.
#[must_use]
#[inline(always)]
pub fn min<T: Min>(left: T, right: T) -> T {
  left.min(right)
}

impl Min for f32 {
  #[must_use]
  #[inline(always)]
  fn min(self, other: Self) -> Self {
    f32::min(self, other)
  }
}

impl Min for f64 {
  #[must_use]
  #[inline(always)]
  fn min(self, other: Self) -> Self {
    f64::min(self, other)
  }
}

#[cfg(feature = "portable_simd")]
impl<const N: usize> Min for core::simd::Simd<f32, N>
where
  core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
{
  #[must_use]
  #[inline(always)]
  fn min(self, other: Self) -> Self {
    self.min(other)
  }
}

#[cfg(feature = "portable_simd")]
impl<const N: usize> Min for core::simd::Simd<f64, N>
where
  core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
{
  #[must_use]
  #[inline(always)]
  fn min(self, other: Self) -> Self {
    self.min(other)
  }
}

macro_rules! max_scalar_int {
  ( $($t:ty),+ ) => {
    $(impl Min for $t {
      #[inline]
      #[must_use]
      fn min(self, other: Self) -> Self {
        other ^ ((self ^ other) & ((self < other) as $t).wrapping_neg())
      }
    })+
  }
}
max_scalar_int!(i8, u8, i16, u16, i32, u32, i64, u64, i128, u128, isize, usize);

macro_rules! max_simd_int {
  ( $($t:ty),+ ) => {
    $(
      #[cfg(feature = "portable_simd")]
      impl<const N: usize> Min for core::simd::Simd<$t, N>
      where
        core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
      {
        /// Lanewise minimum
        #[inline]
        #[must_use]
        fn min(self, other: Self) -> Self {
          use bytemuck::cast;
          other ^ ((self ^ other) & cast::<_, Self>(self.lanes_lt(other).to_int()))
        }
      }
    )+
  }
}
max_simd_int!(i8, u8, i16, u16, i32, u32, i64, u64, isize, usize);

#[test]
fn test_min() {
  assert_eq!(min(3.0, 5.0_f32), 3.0_f32);
  assert_eq!(min(-2.0, -3.0_f64), -3.0_f64);
  //
  assert_eq!(min(200, u8::MAX), 200);
  assert_eq!(min(i8::MIN, 0), i8::MIN);
  assert_eq!(min(i16::MAX - 1, i16::MAX), i16::MAX - 1);
  assert_eq!(min(0_u16, 1231), 0);
  //
  #[cfg(feature = "portable_simd")]
  {
    let left = core::simd::Simd::<i32, 4>::from([-1, 0, 1, i32::MAX]);
    let right = core::simd::Simd::<i32, 4>::from([8, 0, -12, i32::MIN]);
    let expected = core::simd::Simd::<i32, 4>::from([-1, 0, -12, i32::MIN]);
    assert_eq!(min(left, right), expected);
    //
    let left = core::simd::Simd::<u32, 4>::from([1, 0, 1, u32::MAX]);
    let right = core::simd::Simd::<u32, 4>::from([8, 0, 12, 9]);
    let expected = core::simd::Simd::<u32, 4>::from([1, 0, 1, 9]);
    assert_eq!(min(left, right), expected);
  }
}
