/// Gives the maximum of two values.
///
/// * In the case of floats, if one value is NaN then the non-NaN value is
///   returned.
pub trait Max {
  /// Gets the maximum value.
  fn max(self, other: Self) -> Self;
}

/// Calls the [max](Max::max) trait method.
#[must_use]
#[inline(always)]
pub fn max<T: Max>(left: T, right: T) -> T {
  left.max(right)
}

impl Max for f32 {
  #[must_use]
  #[inline(always)]
  fn max(self, other: Self) -> Self {
    f32::max(self, other)
  }
}

impl Max for f64 {
  #[must_use]
  #[inline(always)]
  fn max(self, other: Self) -> Self {
    f64::max(self, other)
  }
}

#[cfg(feature = "portable_simd")]
impl<const N: usize> Max for core::simd::Simd<f32, N>
where
  core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
{
  #[must_use]
  #[inline(always)]
  fn max(self, other: Self) -> Self {
    self.max(other)
  }
}

#[cfg(feature = "portable_simd")]
impl<const N: usize> Max for core::simd::Simd<f64, N>
where
  core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
{
  #[must_use]
  #[inline(always)]
  fn max(self, other: Self) -> Self {
    self.max(other)
  }
}

macro_rules! max_scalar_int {
  ( $($t:ty),+ ) => {
    $(impl Max for $t {
      #[inline]
      #[must_use]
      fn max(self, other: Self) -> Self {
        self ^ ((self ^ other) & ((self < other) as $t).wrapping_neg())
      }
    })+
  }
}
max_scalar_int!(i8, u8, i16, u16, i32, u32, i64, u64, i128, u128, isize, usize);

macro_rules! max_simd_int {
  ( $($t:ty),+ ) => {
    $(
      #[cfg(feature = "portable_simd")]
      impl<const N: usize> Max for core::simd::Simd<$t, N>
      where
        core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
      {
        /// Lanewise maximum
        #[inline]
        #[must_use]
        fn max(self, other: Self) -> Self {
          use bytemuck::cast;
          self ^ ((self ^ other) & cast::<_, Self>(self.lanes_lt(other).to_int()))
        }
      }
    )+
  }
}
max_simd_int!(i8, u8, i16, u16, i32, u32, i64, u64, isize, usize);

#[test]
fn test_max() {
  assert_eq!(max(3.0, 5.0_f32), 5.0_f32);
  assert_eq!(max(-2.0, -3.0_f64), -2.0_f64);
  //
  assert_eq!(max(200, u8::MAX), u8::MAX);
  assert_eq!(max(i8::MIN, 0), 0);
  assert_eq!(max(i16::MAX - 1, i16::MAX), i16::MAX);
  assert_eq!(max(0_u16, 1231), 1231);
  //
  #[cfg(feature = "portable_simd")]
  {
    let left = core::simd::Simd::<i32, 4>::from([-1, 0, 1, i32::MAX]);
    let right = core::simd::Simd::<i32, 4>::from([8, 0, -12, i32::MIN]);
    let expected = core::simd::Simd::<i32, 4>::from([8, 0, 1, i32::MAX]);
    assert_eq!(max(left, right), expected);
    //
    let left = core::simd::Simd::<u32, 4>::from([1, 0, 1, u32::MAX]);
    let right = core::simd::Simd::<u32, 4>::from([8, 0, 12, 9]);
    let expected = core::simd::Simd::<u32, 4>::from([8, 0, 12, u32::MAX]);
    assert_eq!(max(left, right), expected);
  }
}
