use super::get_exponent_bits::*;

pub trait GetExponentValue {
  type Output;
  fn get_exponent_value(self) -> Self::Output;
}

#[must_use]
#[inline(always)]
pub fn get_exponent_value<G: GetExponentValue>(g: G) -> G::Output {
  g.get_exponent_value()
}

impl GetExponentValue for f32 {
  type Output = i32;
  #[must_use]
  #[inline(always)]
  fn get_exponent_value(self) -> i32 {
    (get_exponent_bits(self) as i32 - (f32::MAX_EXP - 1)).max(f32::MIN_EXP - 1)
  }
}

impl GetExponentValue for f64 {
  type Output = i64;
  #[must_use]
  #[inline(always)]
  fn get_exponent_value(self) -> i64 {
    (get_exponent_bits(self) as i32 - (f64::MAX_EXP - 1)).max(f64::MIN_EXP - 1) as i64
  }
}

#[cfg(feature = "portable_simd")]
impl<const N: usize> GetExponentValue for core::simd::Simd<f32, N>
where
  core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
{
  type Output = core::simd::Simd<i32, N>;
  #[must_use]
  #[inline(always)]
  fn get_exponent_value(self) -> Self::Output {
    (crate::u32xN_to_i32xN(get_exponent_bits(self)) - Self::Output::splat(f32::MAX_EXP - 1))
      .max(Self::Output::splat(f32::MIN_EXP - 1))
  }
}

#[cfg(feature = "portable_simd")]
impl<const N: usize> GetExponentValue for core::simd::Simd<f64, N>
where
  core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
{
  type Output = core::simd::Simd<i64, N>;
  #[must_use]
  #[inline(always)]
  fn get_exponent_value(self) -> Self::Output {
    (crate::u64xN_to_i64xN(get_exponent_bits(self))
      - Self::Output::splat((f64::MAX_EXP - 1) as i64))
    .max(Self::Output::splat((f64::MIN_EXP - 1) as i64))
  }
}
