use crate::FloatConstants;

pub trait GetExponentBits {
  type Output;
  fn get_exponent_bits(self) -> Self::Output;
}

#[must_use]
#[inline(always)]
pub fn get_exponent_bits<G: GetExponentBits>(g: G) -> G::Output {
  g.get_exponent_bits()
}

impl GetExponentBits for f32 {
  type Output = u32;
  #[must_use]
  #[inline(always)]
  fn get_exponent_bits(self) -> u32 {
    ((self.to_bits() >> Self::MANTISSA_BITS) & Self::EXPONENT_BIT_MASK_LOW)
  }
}

impl GetExponentBits for f64 {
  type Output = u64;
  #[must_use]
  #[inline(always)]
  fn get_exponent_bits(self) -> u64 {
    ((self.to_bits() >> Self::MANTISSA_BITS) & Self::EXPONENT_BIT_MASK_LOW)
  }
}

#[cfg(feature = "portable_simd")]
impl<const N: usize> GetExponentBits for core::simd::Simd<f32, N>
where
  core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
{
  type Output = core::simd::Simd<u32, N>;
  #[must_use]
  #[inline(always)]
  fn get_exponent_bits(self) -> Self::Output {
    ((self.to_bits() >> Self::MANTISSA_BITS) & Self::EXPONENT_BIT_MASK_LOW)
  }
}

#[cfg(feature = "portable_simd")]
impl<const N: usize> GetExponentBits for core::simd::Simd<f64, N>
where
  core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
{
  type Output = core::simd::Simd<u64, N>;
  #[must_use]
  #[inline(always)]
  fn get_exponent_bits(self) -> Self::Output {
    ((self.to_bits() >> Self::MANTISSA_BITS) & Self::EXPONENT_BIT_MASK_LOW)
  }
}
