/// Gives the absolute value.
///
/// * NaN returns NaN
/// * Negative Infinity returns Infinity
/// * For all other values, return a non-negative value with the same magnitude.
///
/// ## Algorithm
///
/// This operation is very simple, just clear the sign bit.
pub trait Abs {
  /// Gives the absolute value.
  fn abs(self) -> Self;
}

/// Calls the [abs](Abs::abs) trait method.
#[must_use]
#[inline(always)]
pub fn abs<A: Abs>(x: A) -> A {
  x.abs()
}

impl Abs for f32 {
  #[must_use]
  #[inline(always)]
  fn abs(self) -> Self {
    pick! {
      if #[cfg(feature = "core_intrinsics")] {
        unsafe { core::intrinsics::fabsf32(self) }
      } else {
        f32::from_bits(self.to_bits() & !f32::to_bits(-0.0))
      }
    }
  }
}

impl Abs for f64 {
  #[must_use]
  #[inline(always)]
  fn abs(self) -> Self {
    pick! {
      if #[cfg(feature = "core_intrinsics")] {
        unsafe { core::intrinsics::fabsf64(self) }
      } else {
        f64::from_bits(self.to_bits() & !f64::to_bits(-0.0))
      }
    }
  }
}

#[cfg(feature = "portable_simd")]
impl<const N: usize> Abs for core::simd::Simd<f32, N>
where
  core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
{
  #[must_use]
  #[inline(always)]
  fn abs(self) -> Self {
    self.abs()
  }
}

#[cfg(feature = "portable_simd")]
impl<const N: usize> Abs for core::simd::Simd<f64, N>
where
  core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
{
  #[must_use]
  #[inline(always)]
  fn abs(self) -> Self {
    self.abs()
  }
}
