pub trait FloatConstants {
  type Unsigned;

  type Signed;

  const SIGN_BIT_MASK: Self::Unsigned;
  const EXPONENT_BIT_MASK_LOW: Self::Unsigned;

  const BITS: Self::Unsigned;
  const MANTISSA_BITS: Self::Unsigned;
  const NON_MANTISSA_BITS: Self::Unsigned;
  const MAX_REAL_EXP: Self::Signed;
  const MIN_REAL_EXP: Self::Signed;
}

impl FloatConstants for f32 {
  type Unsigned = u32;
  type Signed = i32;
  const SIGN_BIT_MASK: Self::Unsigned = i32::MIN as u32;
  const EXPONENT_BIT_MASK_LOW: Self::Unsigned = u32::MAX >> (1 + Self::MANTISSA_BITS);
  const BITS: Self::Unsigned = 32;
  const MANTISSA_BITS: Self::Unsigned = f32::MANTISSA_DIGITS - 1;
  const NON_MANTISSA_BITS: Self::Unsigned = Self::BITS - Self::MANTISSA_BITS;
  const MAX_REAL_EXP: i32 = f32::MAX_EXP - 1;
  const MIN_REAL_EXP: i32 = f32::MIN_EXP - 1;
}

impl FloatConstants for f64 {
  type Unsigned = u64;
  type Signed = i64;
  const SIGN_BIT_MASK: Self::Unsigned = i64::MIN as u64;
  const EXPONENT_BIT_MASK_LOW: Self::Unsigned = u64::MAX >> (1 + Self::MANTISSA_BITS);
  const BITS: Self::Unsigned = 64;
  const MANTISSA_BITS: Self::Unsigned = (f64::MANTISSA_DIGITS - 1) as u64;
  const NON_MANTISSA_BITS: Self::Unsigned = Self::BITS - Self::MANTISSA_BITS;
  const MAX_REAL_EXP: i64 = (f64::MAX_EXP - 1) as i64;
  const MIN_REAL_EXP: i64 = (f64::MIN_EXP - 1) as i64;
}

#[cfg(feature = "portable_simd")]
impl<const N: usize> FloatConstants for core::simd::Simd<f32, N>
where
  core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
{
  type Unsigned = core::simd::Simd<u32, N>;
  type Signed = core::simd::Simd<i32, N>;
  const SIGN_BIT_MASK: Self::Unsigned = Self::Unsigned::splat(f32::SIGN_BIT_MASK);
  const EXPONENT_BIT_MASK_LOW: Self::Unsigned = Self::Unsigned::splat(f32::EXPONENT_BIT_MASK_LOW);
  const BITS: Self::Unsigned = Self::Unsigned::splat(f32::BITS);
  const MANTISSA_BITS: Self::Unsigned = Self::Unsigned::splat(f32::MANTISSA_BITS);
  const NON_MANTISSA_BITS: Self::Unsigned = Self::Unsigned::splat(f32::NON_MANTISSA_BITS);
  const MAX_REAL_EXP: Self::Signed = Self::Signed::splat(f32::MAX_EXP - 1);
  const MIN_REAL_EXP: Self::Signed = Self::Signed::splat(f32::MIN_EXP - 1);
}

#[cfg(feature = "portable_simd")]
impl<const N: usize> FloatConstants for core::simd::Simd<f64, N>
where
  core::simd::LaneCount<N>: core::simd::SupportedLaneCount,
{
  type Unsigned = core::simd::Simd<u64, N>;
  type Signed = core::simd::Simd<i64, N>;
  const SIGN_BIT_MASK: Self::Unsigned = Self::Unsigned::splat(f64::SIGN_BIT_MASK);
  const EXPONENT_BIT_MASK_LOW: Self::Unsigned = Self::Unsigned::splat(f64::EXPONENT_BIT_MASK_LOW);
  const BITS: Self::Unsigned = Self::Unsigned::splat(f64::BITS);
  const MANTISSA_BITS: Self::Unsigned = Self::Unsigned::splat(f64::MANTISSA_BITS);
  const NON_MANTISSA_BITS: Self::Unsigned = Self::Unsigned::splat(f64::NON_MANTISSA_BITS);
  const MAX_REAL_EXP: Self::Signed = Self::Signed::splat((f64::MAX_EXP - 1) as i64);
  const MIN_REAL_EXP: Self::Signed = Self::Signed::splat((f64::MIN_EXP - 1) as i64);
}
