//!
//! # `flo_binding`, a data-driven binding library
//!
//! `flo_binding` is a library of types intended to help store state in interactive
//! applications. A binding is a mutable value that can notify observers when it
//! changes, or can supply its values as a stream. This is intended to make it easy
//! to build 'data-driven' applications, where updates to model state automatically
//! drive updates to the rest of the application.
//! 
//! This method of propagating state is sometimes called 'reactive programming',
//! and has also been called 'parametric programming' in the past: it's most notable
//! for being the methodology behind spreadsheets and CAD applications before its
//! recent rediscovery in web applications. `flo_binding` prefers the term
//! 'data-driven' as it's a more plain description of how changes are pushed through
//! an application using the library.
//! 
//! The advantage of using a binding library like this over a standard event-driven
//! architecture is that it's not necessary to send events or manually update state
//! when updating the application data model. Bugs are reduced because it's very
//! much harder for a dependent state update to go missing as this library can
//! automatically associate.
//! 
//! There are three main types of binding provided by the `flo_binding` library:
//! 
//! A normal binding, created by `bind(value)` can be updated to a new value by
//! calling `set()`. These are useful for representing state that is directly
//! manipulated by the user.
//! 
//! A computed binding, created by `compute(move || { some_func() })`. When a
//! computed binding accesses the value of another binding by calling `get()`, it
//! will automatically monitor it for changes and indicate that it has changed
//! whenever that value changes. This makes it easy to create chains of dependent 
//! bindings.
//! 
//! A stream binding, created by `bind_stream(stream, initial_value, |old_value, new_value| { /* ... */ });`.
//! This makes it possible for a binding to update its values in response to a
//! stream of events, such as might be available from a UI framework.
//! 
//! To make using computed bindings easier to generate, cloning a binding creates
//! a reference to the same piece of state. ie: `let a = bind(1); let b = a.clone();`
//! will give a and b a reference to the same value, so `a.set(2); b.get()` will
//! return 2.
//! 
//! ## Event-driven architecture
//! 
//! The `when_changed()` function can be called to register a function that is called
//! whenever a binding's value becomes invalidated. Once invalidated, a binding
//! remains invalidated until it's read with `get()`, which avoids computing values
//! for computed bindings that are never used. This can be used to integrate with
//! traditional UI frameworks.
//! 
//! Calling `when_changed()` will return a lifetime object: the event will stop firing
//! when this object is dropped (or when `done()` is called on it). `keep_alive()` can
//! be used to keep the event firing for as long as the binding exists if necessary.
//! 
//! The event-driven approach to application design has many disadvantages: the need to
//! manage event lifetimes is one, and another important one is the need to pass the
//! actual bindings around in order to attach events to them, so `flo_binding` provides
//! an alternative approach.
//! 
//! ## Stream-driven architecture
//! 
//! A superior alternative to the traditional OO-style event-driven architecture is
//! to take a stream-driven approach. Unlike events, streams manage their own lifetimes
//! and can be combined or split, and it's possible to pass around a stream without also 
//! providing access to its source, which is a property that can be exploited to produce 
//! a less interdependent application. FlowBetween, the application that flo_binding was
//! developed for is an example of a stream-driven application.
//! 
//! Calling `follow(binding)` will generate a stream of updates from a binding. This
//! can be used to update part of a user interface directly. It can also be used as
//! an input for `bind_stream()` to create more complicated update rules than are
//! allowed by a simple computed binding.
//! 
//! The companion library `desync` provides the `pipe()` and `pipe_in()` functions
//! which can be used to schedule events generated by a stream.
//! 
//! ## Example
//! 
//! ```
//! # use flo_binding::*;
//! let mut number      = bind(1);
//! let number_clone    = number.clone();
//! let plusone         = computed(move || number_clone.get() + 1);
//! 
//! let mut lifetime    = plusone.when_changed(notify(|| println!("Changed!")));
//! 
//! println!("{}", plusone.get());  // 2
//! # assert!(plusone.get() == 2);
//! number.set(2);                  // 'Changed!'
//! println!("{}", plusone.get());  // 3
//! # assert!(plusone.get() == 3);
//! lifetime.done();
//! 
//! number.set(3);                  // Lifetime is done, so no notification
//! println!("{}", plusone.get());  // 4
//! # assert!(plusone.get() == 4);
//! ```
//!
#![warn(bare_trait_objects)]

mod traits;
pub mod binding_context;
mod binding;
mod computed;
mod bindref;
mod notify_fn;
mod releasable;
#[cfg(feature = "stream")]
mod follow;
#[cfg(feature = "stream")]
mod bind_stream;
#[cfg(feature = "stream")]
mod rope_binding;

pub use self::traits::*;
pub use self::binding::*;
pub use self::computed::*;
pub use self::bindref::*;
pub use self::notify_fn::*;
#[cfg(feature = "stream")]
pub use self::follow::*;
#[cfg(feature = "stream")]
pub use self::bind_stream::*;
#[cfg(feature = "stream")]
pub use self::rope_binding::*;

///
/// Creates a simple bound value with the specified initial value
///
pub fn bind<Value: Clone+PartialEq>(val: Value) -> Binding<Value> {
    Binding::new(val)
}

///
/// Creates a computed value that tracks bindings accessed during the function call and marks itself as changed when any of these dependencies also change
///
pub fn computed<Value, TFn>(calculate_value: TFn) -> ComputedBinding<Value, TFn>
where Value: Clone+Send, TFn: 'static+Send+Sync+Fn() -> Value {
    ComputedBinding::new(calculate_value)
}

#[cfg(test)]
mod test {
    use super::*;
    use super::binding_context::*;

    use std::thread;
    use std::sync::*;
    use std::time::Duration;

    #[test]
    fn can_create_binding() {
        let bound = bind(1);
        assert!(bound.get() == 1);
    }

    #[test]
    fn can_update_binding() {
        let bound = bind(1);

        bound.set(2);
        assert!(bound.get() == 2);
    }

    #[test]
    fn notified_on_change() {
        let bound       = bind(1);
        let changed     = bind(false);

        let notify_changed = changed.clone();
        bound.when_changed(notify(move || notify_changed.set(true))).keep_alive();

        assert!(changed.get() == false);
        bound.set(2);
        assert!(changed.get() == true);
    }

    #[test]
    fn not_notified_on_no_change() {
        let bound       = bind(1);
        let changed     = bind(false);

        let notify_changed = changed.clone();
        bound.when_changed(notify(move || notify_changed.set(true))).keep_alive();

        assert!(changed.get() == false);
        bound.set(1);
        assert!(changed.get() == false);
    }

    #[test]
    fn notifies_after_each_change() {
        let bound           = bind(1);
        let change_count    = bind(0);

        let notify_count    = change_count.clone();
        bound.when_changed(notify(move || { let count = notify_count.get(); notify_count.set(count+1) })).keep_alive();

        assert!(change_count.get() == 0);
        bound.set(2);
        assert!(change_count.get() == 1);

        bound.set(3);
        assert!(change_count.get() == 2);

        bound.set(4);
        assert!(change_count.get() == 3);
    }

    #[test]
    fn dispatches_multiple_notifications() {
        let bound           = bind(1);
        let change_count    = bind(0);

        let notify_count    = change_count.clone();
        let notify_count2   = change_count.clone();
        bound.when_changed(notify(move || { let count = notify_count.get(); notify_count.set(count+1) })).keep_alive();
        bound.when_changed(notify(move || { let count = notify_count2.get(); notify_count2.set(count+1) })).keep_alive();

        assert!(change_count.get() == 0);
        bound.set(2);
        assert!(change_count.get() == 2);

        bound.set(3);
        assert!(change_count.get() == 4);

        bound.set(4);
        assert!(change_count.get() == 6);
    }

    #[test]
    fn stops_notifying_after_release() {
        let bound           = bind(1);
        let change_count    = bind(0);

        let notify_count = change_count.clone();
        let mut lifetime = bound.when_changed(notify(move || { let count = notify_count.get(); notify_count.set(count+1) }));

        assert!(change_count.get() == 0);
        bound.set(2);
        assert!(change_count.get() == 1);

        lifetime.done();
        assert!(change_count.get() == 1);
        bound.set(3);
        assert!(change_count.get() == 1);
    }

    #[test]
    fn release_only_affects_one_notification() {
        let bound           = bind(1);
        let change_count    = bind(0);

        let notify_count    = change_count.clone();
        let notify_count2   = change_count.clone();
        let mut lifetime    = bound.when_changed(notify(move || { let count = notify_count.get(); notify_count.set(count+1) }));
        bound.when_changed(notify(move || { let count = notify_count2.get(); notify_count2.set(count+1) })).keep_alive();

        assert!(change_count.get() == 0);
        bound.set(2);
        assert!(change_count.get() == 2);

        bound.set(3);
        assert!(change_count.get() == 4);

        bound.set(4);
        assert!(change_count.get() == 6);

        lifetime.done();

        bound.set(5);
        assert!(change_count.get() == 7);

        bound.set(6);
        assert!(change_count.get() == 8);

        bound.set(7);
        assert!(change_count.get() == 9);
    }

    #[test]
    fn binding_context_is_notified() {
        let bound = bind(1);

        bound.set(2);

        let (value, context) = BindingContext::bind(|| bound.get());
        assert!(value == 2);

        let changed = bind(false);
        let notify_changed = changed.clone();
        context.when_changed(notify(move || notify_changed.set(true))).keep_alive();

        assert!(changed.get() == false);
        bound.set(3);
        assert!(changed.get() == true);
    }

    #[test]
    fn can_compute_value() {
        let bound           = bind(1);

        let computed_from   = bound.clone();
        let computed        = computed(move || computed_from.get() + 1);

        assert!(computed.get() == 2);
    }

    #[test]
    fn can_recompute_value() {
        let bound           = bind(1);

        let computed_from   = bound.clone();
        let computed        = computed(move || computed_from.get() + 1);

        assert!(computed.get() == 2);

        bound.set(2);
        assert!(computed.get() == 3);

        bound.set(3);
        assert!(computed.get() == 4);
    }

    #[test]
    fn can_recursively_compute_values() {
        let bound               = bind(1);

        let computed_from       = bound.clone();
        let computed_val        = computed(move || computed_from.get() + 1);

        let more_computed_from  = computed_val.clone();
        let more_computed       = computed(move || more_computed_from.get() + 1);

        assert!(computed_val.get() == 2);
        assert!(more_computed.get() == 3);

        bound.set(2);
        assert!(computed_val.get() == 3);
        assert!(more_computed.get() == 4);

        bound.set(3);
        assert!(computed_val.get() == 4);
        assert!(more_computed.get() == 5);
    }

    #[test]
    fn can_recursively_compute_values_2() {
        let bound               = bind(1);

        let computed_from       = bound.clone();
        let computed_val        = computed(move || computed_from.get() + 1);
        let more_computed       = computed(move || computed_val.get() + 1);

        assert!(more_computed.get() == 3);

        bound.set(2);
        assert!(more_computed.get() == 4);

        bound.set(3);
        assert!(more_computed.get() == 5);
    }

    #[test]
    fn can_recursively_compute_values_3() {
        let bound               = bind(1);

        let computed_from       = bound.clone();
        let computed_val        = computed(move || computed_from.get() + 1);
        let more_computed       = computed(move || computed_val.get() + 1);
        let even_more_computed  = computed(move || more_computed.get() + 1);

        assert!(even_more_computed.get() == 4);

        bound.set(2);
        assert!(even_more_computed.get() == 5);

        bound.set(3);
        assert!(even_more_computed.get() == 6);
    }

    #[test]
    #[should_panic]
    fn panics_if_computed_generated_during_binding() {
        let bound               = bind(1);

        let computed_from       = bound.clone();
        let computed_val        = computed(move || computed_from.get() + 1);
        let even_more_computed  = computed(move || {
            let computed_val = computed_val.clone();

            // This computed binding would be dropped after the first evaluation, which would result in the binding never updating.
            // We should panic here.
            let more_computed = computed(move || computed_val.get() + 1);
            more_computed.get() + 1
        });

        assert!(even_more_computed.get() == 4);

        bound.set(2);
        assert!(even_more_computed.get() == 5);

        bound.set(3);
        assert!(even_more_computed.get() == 6);
    }

    #[test]
    fn computed_only_recomputes_as_needed() {
        let bound               = bind(1);

        let counter             = Arc::new(Mutex::new(0));
        let compute_counter     = counter.clone();
        let computed_from       = bound.clone();
        let computed            = computed(move || {
            let mut counter = compute_counter.lock().unwrap();
            *counter = *counter + 1;

            computed_from.get() + 1
        });

        assert!(computed.get() == 2);
        {
            let counter = counter.lock().unwrap();
            assert!(counter.clone() == 1);
        }

        assert!(computed.get() == 2);
        {
            let counter = counter.lock().unwrap();
            assert!(counter.clone() == 1);
        }

        bound.set(2);
        assert!(computed.get() == 3);
        {
            let counter = counter.lock().unwrap();
            assert!(counter.clone() == 2);
        }
    }

    #[test]
    fn computed_caches_values() {
        let update_count            = Arc::new(Mutex::new(0));
        let bound                   = bind(1);

        let computed_update_count   = Arc::clone(&update_count);
        let computed_from           = bound.clone();
        let computed                = computed(move || {
            let mut computed_update_count = computed_update_count.lock().unwrap();
            *computed_update_count += 1;

            computed_from.get() + 1
        });

        assert!(computed.get() == 2);
        assert!(*update_count.lock().unwrap() == 1);

        assert!(computed.get() == 2);
        assert!(*update_count.lock().unwrap() == 1);

        bound.set(2);
        assert!(computed.get() == 3);
        assert!(*update_count.lock().unwrap() == 2);

        bound.set(3);
        assert!(*update_count.lock().unwrap() == 2);
        assert!(computed.get() == 4);
        assert!(*update_count.lock().unwrap() == 3);
    }

    #[test]
    fn computed_notifies_of_changes() {
        let bound           = bind(1);

        let computed_from   = bound.clone();
        let computed        = computed(move || computed_from.get() + 1);

        let changed         = bind(false);
        let notify_changed  = changed.clone();
        computed.when_changed(notify(move || notify_changed.set(true))).keep_alive();

        assert!(computed.get() == 2);
        assert!(changed.get() == false);

        bound.set(2);
        assert!(changed.get() == true);
        assert!(computed.get() == 3);

        changed.set(false);
        bound.set(3);
        assert!(changed.get() == true);
        assert!(computed.get() == 4);
    }

    #[test]
    fn computed_switches_dependencies() {
        let switch          = bind(false);
        let val1            = bind(1);
        let val2            = bind(2);

        let computed_switch = switch.clone();
        let computed_val1   = val1.clone();
        let computed_val2   = val2.clone();
        let computed        = computed(move || {
            // Use val1 when switch is false, and val2 when switch is true
            if computed_switch.get() {
                computed_val2.get() + 1
            } else {
                computed_val1.get() + 1
            }
        });

        let changed         = bind(false);
        let notify_changed  = changed.clone();
        computed.when_changed(notify(move || notify_changed.set(true))).keep_alive();

        // Initial value of computed (first get 'arms' when_changed too)
        assert!(computed.get() == 2);
        assert!(changed.get() == false);

        // Setting val2 shouldn't cause computed to become 'changed' initially
        val2.set(3);
        assert!(changed.get() == false);
        assert!(computed.get() == 2);

        // ... but setting val1 should
        val1.set(2);
        assert!(changed.get() == true);
        assert!(computed.get() == 3);

        // Flicking the switch will use the val2 value we set earlier
        changed.set(false);
        switch.set(true);
        assert!(changed.get() == true);
        assert!(computed.get() == 4);

        // Updating val2 should now mark us as changed
        changed.set(false);
        val2.set(4);
        assert!(changed.get() == true);
        assert!(computed.get() == 5);
        
        // Updating val1 should not mark us as changed
        changed.set(false);
        val1.set(5);
        assert!(changed.get() == false);
        assert!(computed.get() == 5);
    }

    #[test]
    fn change_during_computation_recomputes() {
        // Create a computed binding that delays for a bit while reading
        let some_binding = bind(1);
        let some_computed = {
            let some_binding = some_binding.clone();
            computed(move || {
                let result = some_binding.get() + 1;
                thread::sleep(Duration::from_millis(250));
                result
            })
        };

        // Start a thread that reads a value
        {
            let some_computed = some_computed.clone();
            thread::spawn(move || {
                assert!(some_computed.get() == 2);
            });
        }

        // Let the thread start running (give it enough time to start computing and reach the sleep statement)
        // TODO: thread::sleep might fail on systems that are slow enough or due to glitches (will fail spuriously if we update the binding before the calculation starts)
        thread::sleep(Duration::from_millis(10));

        // Update the value in the binding while the computed is running
        some_binding.set(2);

        // Computed value should update
        assert!(some_computed.get() == 3);
    }

    #[test]
    fn computed_propagates_changes() {
        let bound               = bind(1);

        let computed_from       = bound.clone();
        let propagates_from     = computed(move || computed_from.get() + 1);
        let computed_propagated = propagates_from.clone();
        let computed            = computed(move || computed_propagated.get() + 1);

        let changed             = bind(false);
        let notify_changed      = changed.clone();
        computed.when_changed(notify(move || notify_changed.set(true))).keep_alive();

        assert!(propagates_from.get() == 2);
        assert!(computed.get() == 3);
        assert!(changed.get() == false);

        bound.set(2);
        assert!(propagates_from.get() == 3);
        assert!(computed.get() == 4);
        assert!(changed.get() == true);

        changed.set(false);
        bound.set(3);
        assert!(changed.get() == true);
        assert!(propagates_from.get() == 4);
        assert!(computed.get() == 5);
    }

    #[test]
    fn computed_stops_notifying_when_released() {
        let bound           = bind(1);

        let computed_from   = bound.clone();
        let computed        = computed(move || computed_from.get() + 1);

        let changed         = bind(false);
        let notify_changed  = changed.clone();
        let mut lifetime    = computed.when_changed(notify(move || notify_changed.set(true)));

        assert!(computed.get() == 2);
        assert!(changed.get() == false);

        bound.set(2);
        assert!(changed.get() == true);
        assert!(computed.get() == 3);

        changed.set(false);
        lifetime.done();

        bound.set(3);
        assert!(changed.get() == false);
        assert!(computed.get() == 4);

        bound.set(4);
        assert!(changed.get() == false);
        assert!(computed.get() == 5);
    }

    #[test]
    fn computed_doesnt_notify_more_than_once() {
        let bound           = bind(1);

        let computed_from   = bound.clone();
        let computed        = computed(move || computed_from.get() + 1);

        let changed         = bind(false);
        let notify_changed  = changed.clone();
        computed.when_changed(notify(move || notify_changed.set(true))).keep_alive();

        assert!(computed.get() == 2);
        assert!(changed.get() == false);

        // Setting the value marks the computed as changed
        bound.set(2);
        assert!(changed.get() == true);
        changed.set(false);

        // ... but when it's already changed we don't notify again
        bound.set(3);
        assert!(changed.get() == false);

        assert!(computed.get() == 4);

        // Once we've retrieved the value, we'll get notified of changes again
        bound.set(4);
        assert!(changed.get() == true);
    }

    #[test]
    fn computed_stops_notifying_once_out_of_scope() {
        let bound           = bind(1);
        let changed         = bind(false);

        {
            let computed_from   = bound.clone();
            let computed        = computed(move || computed_from.get() + 1);

            let notify_changed  = changed.clone();
            computed.when_changed(notify(move || notify_changed.set(true))).keep_alive();

            assert!(computed.get() == 2);
            assert!(changed.get() == false);

            bound.set(2);
            assert!(changed.get() == true);
            assert!(computed.get() == 3);
        };

        // The computed value should have been disposed of so we should get no more notifications once we reach here
        changed.set(false);
        bound.set(3);
        assert!(changed.get() == false);
    }
}
