mod csv_altimeter;

use csv_altimeter::*;
use flight_computer::flight_phase::FlightPhase;
use fugit::{MillisDuration, MillisDurationU64};
use std::{ops::Range, path::Path};

/// Zero duration
const NULL_DURATION: MillisDuration<u64> = MillisDurationU64::from_ticks(0);
/// Tolerance after the "real" takeoff time (in ms)
const TAKEOFF_TOLERANCE: MillisDuration<u64> = MillisDurationU64::from_ticks(40_000);
/// Tolerance after the "real" exit time (in ms)
const FREEFALL_TOLERANCE: MillisDuration<u64> = MillisDurationU64::from_ticks(5000);
/// Tolerance after the "real" landing time (in ms)
const LANDING_TOLERANCE: MillisDuration<u64> = MillisDurationU64::from_ticks(30_000);

fn detect_takeoff(
    path: &Path,
    real_takeoff: MillisDuration<u64>,
    time_range: Range<MillisDuration<u64>>,
) {
    let (switch_timestamp, agl, computer) =
        detect_phase_from_csv(path, time_range, FlightPhase::Ground, FlightPhase::Climb);

    println!("Switch timestamp: {}", switch_timestamp);

    let diff = switch_timestamp - real_takeoff;
    println!(
        "Detected takeoff: {} ms, Difference: {} ms, AGL: {} m, Ground reference: {:?}",
        switch_timestamp,
        diff,
        agl,
        computer.ground_ref()
    );
    assert!(diff <= TAKEOFF_TOLERANCE && diff >= NULL_DURATION);

    let reference_diff = computer.ground_ref().agl;
    println!("Reference diff: {} m", reference_diff);
    assert!(reference_diff.abs() <= 3.0);
}

fn detect_freefall(
    path: &Path,
    real_exit: MillisDuration<u64>,
    time_range: Range<MillisDuration<u64>>,
) {
    let (switch_timestamp, agl, _) = detect_phase_from_csv(
        path,
        time_range,
        FlightPhase::Climb,
        FlightPhase::FlightLogging,
    );

    let diff = switch_timestamp.ticks() as i64 - real_exit.ticks() as i64;
    println!(
        "Detected exit: {} ms, Difference: {} ms, AGL: {} m",
        switch_timestamp, diff, agl
    );
    assert!(diff.abs() <= (FREEFALL_TOLERANCE.ticks() as i64));
}

fn detect_landing(
    path: &Path,
    real_landing: MillisDuration<u64>,
    time_range: Range<MillisDuration<u64>>,
) {
    let (switch_timestamp, agl, _) = detect_phase_from_csv(
        path,
        time_range,
        FlightPhase::FlightLogging,
        FlightPhase::Ground,
    );

    let diff = switch_timestamp - real_landing;
    println!(
        "Detected exit: {} ms, Difference: {} ms, AGL: {} m",
        switch_timestamp, diff, agl
    );
    assert!(diff <= LANDING_TOLERANCE && diff >= NULL_DURATION);
}

#[test]
fn phase_switching_1() {
    const REAL_TAKEOFF: u64 = 116_000;
    const REAL_EXIT: u64 = 969_121;
    const REAL_LANDING: u64 = 1_613_337;

    let path = Path::new("./test-data/takeoff-freefall-landing.csv");
    println!("Analyzing '{}'...", path.display());

    // Detect Ground -> Climb
    detect_takeoff(
        path,
        MillisDurationU64::from_ticks(REAL_TAKEOFF),
        Range {
            start: MillisDurationU64::from_ticks(0),
            end: MillisDurationU64::from_ticks(217_000),
        },
    );

    // Detect Climb -> Freefall
    detect_freefall(
        path,
        MillisDurationU64::from_ticks(REAL_EXIT),
        Range {
            start: MillisDurationU64::from_ticks(116_000),
            end: MillisDurationU64::from_ticks(1_613_337),
        },
    );

    // Detect freefall -> Ground
    detect_landing(
        path,
        MillisDurationU64::from_ticks(REAL_LANDING),
        Range {
            start: MillisDurationU64::from_ticks(969_121),
            end: MillisDurationU64::from_ticks(1_675_000),
        },
    );
}

#[test]
fn phase_switching_2() {
    const REAL_TAKEOFF: u64 = 668_039;
    const REAL_EXIT: u64 = 1_470_000;

    let path = Path::new("./test-data/takeoff-freefall.csv");
    println!("Analyzing '{}'...", path.display());

    // Detect Ground -> Climb
    detect_takeoff(
        path,
        MillisDurationU64::from_ticks(REAL_TAKEOFF),
        Range {
            start: MillisDurationU64::from_ticks(0),
            end: MillisDurationU64::from_ticks(863_738),
        },
    );
    // Detect Climb -> Freefall
    detect_freefall(
        path,
        MillisDurationU64::from_ticks(REAL_EXIT),
        Range {
            start: MillisDurationU64::from_ticks(668_039),
            end: MillisDurationU64::from_ticks(1_488_000),
        },
    );
}

#[test]
fn phase_switching_3() {
    const REAL_LANDING: u64 = 1_068_000;

    let path = Path::new("./test-data/landing.csv");
    println!("Analyzing '{}'...", path.display());

    // Detect Freefall -> Ground
    detect_landing(
        path,
        MillisDurationU64::from_ticks(REAL_LANDING),
        Range {
            start: MillisDurationU64::from_ticks(0),
            end: MillisDurationU64::from_ticks(1_136_400),
        },
    );
}
