//! Test harness for the expedite-gps project

use embedded_hal::blocking::delay::DelayMs;

use csv::{self, DeserializeRecordsIntoIter};
use fugit::{MillisDuration, MillisDurationU64};
use serde::Deserialize;
use std::{fs::File, io::Read, ops::Range, path::Path};

use flight_computer::{
    altimeter::Altimeter,
    flight_phase::{BaroComputer, BaroData, FlightPhase},
};

#[derive(Debug, Deserialize)]
pub struct LogPoint {
    #[serde(rename = "Datetime")]
    _datetime: String,
    #[serde(rename = "Millis")]
    millis: u64,
    #[serde(rename = "Lat")]
    _lat: isize,
    #[serde(rename = "Long")]
    _long: isize,
    #[serde(rename = "AGL_m")]
    agl: f32,
    #[serde(rename = "GPSAlt_cm")]
    _gps_alt: isize,
    #[serde(rename = "Speed_mkn")]
    _speed: usize,
    #[serde(rename = "Heading")]
    _heading: Option<f32>,
    #[serde(rename = "Temp_degC")]
    _temp: f32,
    #[serde(rename = "Charge_Bat_V")]
    _charge: usize,
}

/// ALtimeter mock based on reading values from CSV flight logs.
pub struct CsvAltimeter<R> {
    records: DeserializeRecordsIntoIter<R, LogPoint>,
    current_phase: FlightPhase,
    time_range: Range<MillisDuration<u64>>,
    old_millis: MillisDuration<u64>,
}

impl<R> CsvAltimeter<R>
where
    R: Read,
{
    pub fn new(
        records: DeserializeRecordsIntoIter<R, LogPoint>,
        current_phase: FlightPhase,
        time_range: Range<MillisDuration<u64>>,
    ) -> Self {
        Self {
            records,
            current_phase,
            time_range,
            old_millis: MillisDurationU64::from_ticks(0),
        }
    }
}

struct MockDelay;

impl DelayMs<u8> for MockDelay {
    fn delay_ms(&mut self, ms: u8) {
        std::thread::sleep(std::time::Duration::from_millis(ms as u64))
    }
}

impl<R> Altimeter for CsvAltimeter<R>
where
    R: Read,
{
    fn measure<D: DelayMs<u8>>(
        &mut self,
        _delay: &mut D,
        ground_ref: BaroData,
        _: MillisDurationU64,
    ) -> BaroData {
        // Loop over all CSV records
        for result in &mut self.records {
            let record: LogPoint = result.unwrap();

            // Get timestamp
            let millis = MillisDurationU64::from_ticks(record.millis);

            // Get wait time according to current phase
            let wait_time = self.current_phase.phase_check_wait_time();

            // Wait until wait_time has "elapsed" and we are in the correct time range
            if (millis - self.old_millis) >= wait_time
                && millis >= self.time_range.start
                && millis <= self.time_range.end
            {
                // We simulate a measurement from a ground pressure reference by using the AGL
                // at ground reference.
                let agl = record.agl - ground_ref.agl;

                let measurement = BaroData {
                    agl,
                    pressure: 0.0,
                    timestamp: millis,
                };

                self.old_millis = millis;

                return measurement;
            }
        }

        panic!("Reached end of file!");
    }
}

pub fn detect_phase_from_csv(
    path: &Path,
    time_range: Range<MillisDuration<u64>>,
    phase_from: FlightPhase,
    phase_to: FlightPhase,
) -> (MillisDuration<u64>, f32, BaroComputer<CsvAltimeter<File>>) {
    // Build the CSV reader and iterate over each record.
    let rdr = csv::ReaderBuilder::new()
        .comment(Some(b'#'))
        .from_path(path)
        .unwrap();

    let altimeter = CsvAltimeter::new(rdr.into_deserialize(), phase_from, time_range);

    let mut baro_computer = BaroComputer::new(
        altimeter,
        phase_from,
        BaroData {
            agl: 0.0,
            pressure: 0.0,
            timestamp: MillisDurationU64::from_ticks(0),
        },
    );

    loop {
        let mut switch_point = BaroData {
            agl: 0.0,
            pressure: 0.0,
            timestamp: MillisDurationU64::from_ticks(0),
        };
        let old_phase = baro_computer.current_phase();

        baro_computer.measure(&mut MockDelay {}, MillisDurationU64::from_ticks(0));
        baro_computer.check_phase_switch(|_new_phase, sp, _| {
            switch_point = sp;
        });
        let new_phase = baro_computer.current_phase();

        if old_phase == phase_from && new_phase == phase_to {
            return (switch_point.timestamp, switch_point.agl, baro_computer);
        }
    }
}
