//! # Utilities

/// Prevent a variable from being optimized out
#[inline(never)]
#[cfg(debug_assertions)]
pub fn deoptimize<T>(var: &T) {
    let var = unsafe { core::ptr::read_volatile(var) };
    core::mem::forget(var);
}

#[derive(Debug)]
/// A circular buffer that only provides push and look-behind capabilities.
pub struct WheelBuffer<T, const N: usize> {
    write_idx: usize,
    buf: [T; N],
}

impl<T, const N: usize> WheelBuffer<T, N>
where
    T: Copy,
{
    #[inline]
    pub fn new(fill_with: T) -> Self {
        Self {
            write_idx: 0,
            buf: [fill_with; N],
        }
    }

    #[inline]
    pub fn push(&mut self, new: T) {
        self.write_idx = if self.write_idx < N - 1 {
            self.write_idx + 1
        } else {
            0
        };

        self.buf[self.write_idx] = new;
    }

    /// Get the element at the specified index
    ///
    /// # Errors
    ///
    /// May return `Err` if the index offset is larger than the buffer size.
    #[inline]
    pub fn peek(&self, offset: usize) -> Result<T, BufferError> {
        if offset >= N {
            return Err(BufferError::OffsetTooLarge);
        }

        let offset = self.write_idx as isize - offset as isize;

        Ok(if offset >= 0 {
            self.buf[offset as usize]
        } else {
            self.buf[(offset + N as isize) as usize]
        })
    }

    #[inline]
    pub fn latest(&self) -> T {
        // Unwrap here because peek(0) should never fail
        self.peek(0).unwrap()
    }
}

#[derive(Clone, Copy, Debug)]
pub enum BufferError {
    OffsetTooLarge,
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn wheelbuffer_validate_offset() {
        let fill_with = 0;
        let mut wheelbuf: WheelBuffer<usize, 10> = WheelBuffer::new(fill_with);
        wheelbuf.push(1);
        wheelbuf.push(2);
        wheelbuf.push(3);

        assert_eq!(wheelbuf.peek(0).unwrap(), 3);
        assert_eq!(wheelbuf.peek(1).unwrap(), 2);
        assert_eq!(wheelbuf.peek(2).unwrap(), 1);

        // Test a buffer index that hasn't been written to yet
        assert_eq!(wheelbuf.peek(3).unwrap(), fill_with);
        // Test the edge case of buffer length
        assert_eq!(wheelbuf.peek(9).unwrap(), fill_with);
    }

    #[test]
    #[should_panic]
    fn wheelbuffer_offset_too_large() {
        let wheelbuf: WheelBuffer<usize, 10> = WheelBuffer::new(0);
        // Test the edge case of a too large index
        let _ = wheelbuf.peek(10).unwrap();
    }

    #[test]
    fn wheelbuffer_offset_larger_than_isize_max() {
        let wheelbuf: WheelBuffer<isize, 10> = WheelBuffer::new(0);
        assert!(wheelbuf.peek(usize::MAX - 1).is_err());
    }
}
