#![cfg_attr(target_family = "wasm", feature(wasi_ext))]

mod file_ext;
mod page;
mod page_no;
mod size_info;

pub use page::*;
pub use page_no::*;

pub use page_lock as lock_api;

use file_ext::FileExt;
use size_info::SizeInfo;
use std::cell::UnsafeCell;
use std::convert::TryInto;
use std::fs::File;
use std::io::{Error, ErrorKind, Result};
use std::sync::atomic::{AtomicU32, Ordering};

pub use page_lock::RwLock;
use tokio::task::spawn_blocking;

/// **NOTE**: Caller must ensure that closure must not panic.
/// If it does, it will be undefined behavior.
macro_rules! asyncify { [$body: expr] => { unsafe { spawn_blocking(move || $body).await.unwrap_unchecked() } }; }

pub struct Pages<N, const SIZE: usize> {
    /// Total Page number
    len: AtomicU32,
    file: &'static File,
    pub lock_api: RwLock<N>,
    metadata: UnsafeCell<[u8; SIZE]>,
}

unsafe impl<N: Send, const S: usize> Send for Pages<N, S> {}
unsafe impl<N: Sync, const S: usize> Sync for Pages<N, S> {}

impl<N: PageNo, const SIZE: usize> Pages<N, SIZE> {
    /// Create a new `Pages` instance.
    pub fn open(file: File) -> Result<Self> {
        let size_info = SizeInfo {
            block_size: SIZE as u32,
            pages_len_nbytes: N::SIZE as u8,
        };
        let file_len = file.metadata()?.len();
        // So that, There is no residue bytes.
        if file_len % SIZE as u64 != 0 {
            return Err(ErrorKind::InvalidData.into());
        }
        let mut metadata = [0; SIZE];
        let len = if file_len == 0 {
            // Is new file? set the length to 1.
            file.set_len(SIZE as u64)?;
            1
        } else {
            file.read_exact_at(&mut metadata, 0)?;
            let info = SizeInfo::from(metadata[0..4].try_into().unwrap());
            if info != size_info {
                return Err(Error::new(
                    ErrorKind::InvalidInput,
                    format!("Expected {:?}, but got: {:?}", info, size_info),
                ));
            }
            metadata[0..4].copy_from_slice(&size_info.to_bytes());
            file_len as u32 / SIZE as u32
        };
        Ok(Self {
            len: len.into(),
            lock_api: RwLock::new(),
            metadata: metadata.into(),
            file: Box::leak(Box::new(file)),
        })
    }

    pub async fn sync_data(&self) -> Result<()> {
        let file = self.file;
        asyncify!(file.sync_data())
    }

    pub async fn sync_all(&self) -> Result<()> {
        let file = self.file;
        asyncify!(file.sync_all())
    }

    /// Read a page, without locking.
    pub async fn get(&self, num: N) -> Result<[u8; SIZE]> {
        debug_assert!(num.as_u32() < self.len());
        let num = num.as_u32() as u64;
        let file = self.file;
        asyncify!({
            let mut buf = [0; SIZE];
            file.read_exact_at(&mut buf, SIZE as u64 * num)?;
            Ok(buf)
        })
    }

    pub async fn read(&self, num: N) -> Result<Reader<'_, N, [u8; SIZE]>> {
        let _guard = self.lock_api.read(num).await;
        let data = self.get(num).await?;
        Ok(Reader { data, _guard })
    }

    pub async fn goto(&self, num: N) -> Result<Page<'_, N, SIZE>> {
        let _lock = self.lock_api.write(num).await;
        let buf = self.get(num).await?;
        Ok(Page {
            _lock,
            num,
            buf,
            pages: self,
        })
    }

    pub async fn write(&self, num: N, buf: [u8; SIZE]) -> Result<()> {
        debug_assert!(num.as_u32() < self.len());
        let num = num.as_u32() as u64;
        let file = self.file;
        asyncify!(file.write_all_at(&buf, SIZE as u64 * num))
    }

    /// Increase page number by `count`
    pub async fn alloc(&self, count: u32) -> Result<N> {
        let old_len = self.len.fetch_add(count, Ordering::SeqCst);
        let file = self.file;
        asyncify!(file.set_len(SIZE as u64 * (old_len + count) as u64))?;
        Ok(N::new(old_len))
    }

    pub async fn create(&self, buf: [u8; SIZE]) -> Result<N> {
        let num = self.len.fetch_add(1, Ordering::SeqCst);
        self.write(PageNo::new(num), buf).await?;
        Ok(N::new(num))
    }

    pub fn alloc_sync(&self, count: u32) -> Result<N> {
        let old_len = self.len.fetch_add(count, Ordering::SeqCst);
        self.file.set_len(SIZE as u64 * (old_len + count) as u64)?;
        Ok(N::new(old_len))
    }

    #[inline]
    #[allow(clippy::len_without_is_empty)]
    pub fn len(&self) -> u32 {
        self.len.load(Ordering::Relaxed)
    }

    #[inline]
    pub fn inner(&self) -> &File {
        self.file
    }

    #[allow(clippy::mut_from_ref)]
    /// Get a mutable reference to the pages's metadata.
    pub unsafe fn metadata(&self) -> &mut [u8] {
        &mut (*self.metadata.get())[4..]
    }
}

impl<N, const SIZE: usize> Drop for Pages<N, SIZE> {
    fn drop(&mut self) {
        self.file.write_all_at(self.metadata.get_mut(), 0).unwrap();
    }
}
