use super::*;
use std::future::Future;
use std::ops::{Deref, DerefMut};

pub struct Page<'a, N: PageNo, const SIZE: usize> {
    pub buf: [u8; SIZE],
    pub(super) num: N,
    pub(super) pages: &'a Pages<N, SIZE>,
    pub(super) _lock: page_lock::WriteGuard<'a, N>,
}

impl<'a, N: PageNo, const SIZE: usize> Page<'a, N, SIZE> {
    pub fn write(self) -> impl Future<Output = Result<()>> + 'a {
        self.pages.write(self.num, self.buf)
    }
    
    pub fn write_buf(self, buf: [u8; SIZE]) -> impl Future<Output = Result<()>> + 'a {
        self.pages.write(self.num, buf)
    }
}

impl<N: PageNo, const SIZE: usize> Deref for Page<'_, N, SIZE> {
    type Target = [u8; SIZE];
    fn deref(&self) -> &Self::Target {
        &self.buf
    }
}

impl<N: PageNo, const SIZE: usize> DerefMut for Page<'_, N, SIZE> {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.buf
    }
}

pub struct Reader<'a, N: PageNo, T> {
    pub(super) data: T,
    pub(super) _guard: page_lock::ReadGuard<'a, N>,
}

impl<N: PageNo, T> Reader<'_, N, T> {
    pub fn into_inner(self) -> T {
        self.data
    }
}

impl<N: PageNo, T> Deref for Reader<'_, N, T> {
    type Target = T;
    fn deref(&self) -> &Self::Target {
        &self.data
    }
}
