use std::convert::TryInto;
use std::fmt::Debug;
use std::hash::Hash;

pub trait PageNo: Ord + Hash + Copy + Unpin + Sync + Send + Default + Debug {
    /// Total number of bytes.
    const SIZE: usize;
    fn new(_: u32) -> Self;
    fn as_u32(self) -> u32;
    fn to_bytes(&self) -> Vec<u8>;
    fn from_bytes(_: &[u8]) -> Self;
}

macro_rules! impl_trait {
    ($name:ident for $($t:ty:$S:expr)*) => ($(
        impl $name for $t {
            const SIZE: usize = $S;
            #[inline]
            fn new(num: u32) -> Self { num.try_into().unwrap() }
            #[inline]
            fn as_u32(self) -> u32 { self.try_into().unwrap() }
            #[inline]
            fn to_bytes(&self) -> Vec<u8> { self.to_le_bytes().to_vec() }
            #[inline]
            fn from_bytes(buf: &[u8]) -> Self { Self::from_le_bytes(buf.try_into().unwrap()) }
        }
    )*)
}
impl_trait!(PageNo for u8:1 u16:2 u32:4);

#[repr(transparent)]
#[derive(Hash, PartialEq, Eq, Clone, Copy, PartialOrd, Ord, Default, Debug)]
pub struct U24(u32);

impl PageNo for U24 {
    const SIZE: usize = 3;

    fn new(num: u32) -> Self {
        assert!(num < 16777215);
        Self(num)
    }

    fn to_bytes(&self) -> Vec<u8> {
        let [a, b, c, _] = self.0.to_le_bytes();
        [a, b, c].to_vec()
    }

    fn from_bytes(buf: &[u8]) -> Self {
        Self(u32::from_le_bytes([buf[0], buf[1], buf[2], 0]))
    }

    fn as_u32(self) -> u32 {
        self.0
    }
}
