#![warn(clippy::pedantic)]

use std::fs::File;
use std::{error::Error, io};

#[cfg(feature = "clap")]
use clap::Parser;

#[cfg(feature = "ui")]
use crossterm::{
    event::{self, DisableMouseCapture, EnableMouseCapture, Event, KeyCode},
    execute,
    terminal::{
        disable_raw_mode, enable_raw_mode, EnterAlternateScreen,
        LeaveAlternateScreen,
    },
};
#[cfg(feature = "ui")]
use tui::{
    backend::{Backend, CrosstermBackend},
    layout::{Alignment, Constraint, Direction, Layout},
    text::{Span, Spans, Text},
    widgets::{Paragraph, Widget, Wrap},
    Frame, Terminal,
};

use flashed::App;
use flashed::Opts;

#[cfg(feature = "ui")]
mod theme;

#[cfg(not(feature = "ui"))]
fn main() {
    compile_error!("Really sorry but you don't have a ui enabled.");
}

#[cfg(feature = "ui")]
fn main() -> Result<(), Box<dyn Error>> {
    let opts = Opts::parse();

    // Open the cards
    let mut f = File::open(&opts.input)?;
    let cards = serde_json::from_reader(&mut f).unwrap();

    // Create app and run it
    let mut app = App::new(cards, opts);
    if !app.opts.reset {
        app.read_scores()?;
        app.retain_undue();
        app.order_playables();
    }

    // setup terminal
    enable_raw_mode()?;
    let mut stdout = io::stdout();
    execute!(stdout, EnterAlternateScreen, EnableMouseCapture)?;
    let backend = CrosstermBackend::new(stdout);
    let mut terminal = Terminal::new(backend)?;
    let res = run_app(&mut terminal, app);

    // restore terminal
    disable_raw_mode()?;
    execute!(
        terminal.backend_mut(),
        LeaveAlternateScreen,
        DisableMouseCapture
    )?;
    terminal.show_cursor()?;

    match res {
        Ok(o) => {
            if o {
                println!("All cards learned!")
            } else {
                println!("Exiting!")
            }
        }
        Err(err) => eprintln!("{:?}", err),
    }

    Ok(())
}

#[cfg(feature = "ui")]
fn run_app<B: Backend>(
    terminal: &mut Terminal<B>,
    mut app: App,
) -> io::Result<bool> {
    while app.unfinished_count() > 0 {
        terminal.draw(|f| ui(f, &app))?;

        if let Event::Key(key) = event::read()? {
            match key.code {
                KeyCode::Char(' ') => {
                    app.flip();
                }
                KeyCode::Char('q') => {
                    app.write_scores()?;
                    return Ok(false);
                }
                KeyCode::Char('j') => {
                    app.change_current_card_score(-1);
                    app.next_card();
                }
                KeyCode::Char('k') => {
                    app.change_current_card_score(1);
                    app.next_card();
                }
                _ => {}
            }
        }
    }

    app.write_scores()?;
    Ok(true)
}

#[cfg(feature = "ui")]
fn ui<B: Backend>(f: &mut Frame<B>, app: &App) {
    let chunks = Layout::default()
        .direction(Direction::Vertical)
        .constraints(
            [
                Constraint::Percentage(50),
                Constraint::Length(f.size().height / 2 - 1),
                Constraint::Length(1),
            ]
            .as_ref(),
        )
        .split(f.size());

    f.render_widget(create_instructions(app), chunks[0]);

    f.render_widget(create_card(app), chunks[1]);

    f.render_widget(create_stats(app), chunks[2]);
}

/// Creates the instructions to be shown at the top
/// of the app.
#[cfg(feature = "ui")]
fn create_instructions(_app: &App) -> impl Widget {
    // Create the help message
    //
    // Press q to exit, space to flip, j to demote, k to promote.
    let msg = vec![
        Span::raw("Press "),
        Span::styled("q", *theme::KEY),
        Span::raw(" to exit, "),
        Span::styled("space", *theme::KEY),
        Span::raw(" to flip"),
        Span::raw(", "),
        Span::styled("j", *theme::KEY),
        Span::raw(" to demote, "),
        Span::styled("k", *theme::KEY),
        Span::raw(" to promote"),
        Span::raw("."),
    ];

    // Convert the spans into a text and apply
    // the style.
    let text = Text::from(Spans::from(msg));

    Paragraph::new(text)
}

/// Creates the card for the user to flip
#[cfg(feature = "ui")]
fn create_card(app: &App) -> impl Widget + '_ {
    let mut card;

    // Display the text
    if app.flipped {
        card = Paragraph::new(app.card().answer.as_str())
            .style(*theme::CARD_FLIPPED);
    } else {
        card = Paragraph::new(app.card().question.as_str()).style(*theme::CARD);
    }

    // Format it
    card = card.alignment(Alignment::Center).wrap(Wrap { trim: false });

    card
}

/// Creates the stats about the card and deck
#[cfg(feature = "ui")]
fn create_stats(app: &App) -> impl Widget + '_ {
    // Write the stats
    let msg = vec![
        Span::raw("Card Score: "),
        Span::styled(app.card_score().to_string(), *theme::STAT),
        Span::raw(", Correct: "),
        Span::styled(
            (app.cards().len() - app.unfinished_count()).to_string(),
            *theme::STAT,
        ),
        Span::raw(", Still to go: "),
        Span::styled(app.unfinished_count().to_string(), *theme::STAT),
        Span::raw(", Total: "),
        Span::styled(app.cards().len().to_string(), *theme::STAT),
    ];

    // Convert the spans into a text.
    let text = Text::from(Spans::from(msg));

    Paragraph::new(text)
}
