use std::collections::HashMap;
use std::fs::File;
use std::io;
use std::mem;
use std::time::Duration;

use chrono::{DateTime, Utc};

use crate::card::Card;
use crate::scores::Scores;
use crate::Opts;

/// A list of durations which cards will be waiting by.
pub static DURATIONS: [Duration; 5] = [
    Duration::from_secs(0),                     // Instant
    Duration::from_secs(60 * 60 * 24),          // Daily
    Duration::from_secs(60 * 60 * 24 * 7),      // Weekly
    Duration::from_secs(60 * 60 * 24 * 28),     // Monthly
    Duration::from_secs(60 * 60 * 24 * 28 * 2), // Bimonthly
];

/// A convenience type representing an index of [`App::cards`]
pub type CardIdx = usize;

/// App holds the state of the application
#[derive(Debug, Clone)]
pub struct App {
    card: CardIdx,
    playable_card_pos: CardIdx,
    cards: Vec<Card>,
    playable_cards: Vec<CardIdx>,
    scores: HashMap<CardIdx, usize>,
    dues: HashMap<CardIdx, DateTime<Utc>>,
    circulating: Vec<CardIdx>,
    /// This determines whether the card should show the
    /// question or the answer.
    pub flipped: bool,
    /// The (command line) options of the [`App`] to determine
    /// how it willl run.
    pub opts: Opts,
}

impl App {
    /// Creates a new app.
    pub fn new(cards: Vec<Card>, opts: Opts) -> Self {
        let mut app = Self {
            card: 0,
            playable_card_pos: 0,
            playable_cards: (0..cards.len()).collect(),
            cards,
            scores: HashMap::new(),
            circulating: Vec::new(),
            flipped: false,
            dues: HashMap::new(),
            opts,
        };

        app.shuffle_all();
        app.retain_undue();
        app.next_card();

        app
    }

    /// Put the playable cards in order based of their score
    pub fn order_playables(&mut self) {
        self.playable_cards.sort_unstable_by_key(|card| {
            (
                self.circulating.contains(card),
                *self.scores.get(card).unwrap_or(&usize::MAX),
                *self.dues.get(card).unwrap_or(&chrono::MAX_DATETIME),
            )
        });
    }

    /// Returns an immutable copy of all the cards
    pub fn cards(&self) -> &[Card] {
        &self.cards
    }

    /// Returns in immutable copy of when different cards
    /// are due.
    ///
    /// The key is garunteed to probably be an index from `[Self::cards]`
    pub fn dues(&self) -> &HashMap<CardIdx, DateTime<Utc>> {
        &self.dues
    }

    /// Returns in immutable copy of each cards scores.
    ///
    /// The key is garunteed to probably be an index from `[Self::cards]`
    pub fn scores(&self) -> &HashMap<usize, usize> {
        &self.scores
    }

    /// Returns a copy of cards which are mandatorily in
    /// circulation until answered correctly
    pub fn circulating(&self) -> &[CardIdx] {
        &self.circulating
    }

    /// Flips the current card to show the other side.
    pub fn flip(&mut self) {
        self.flipped ^= true;
    }

    /// Gets the range of cards currently being played with.
    pub fn playable_range(&self) -> usize {
        self.opts
            .testing
            .unwrap_or(self.playable_cards.len())
            .min(self.playable_cards.len())
    }

    /// Shuffles the deck so you can play with different cards
    pub fn shuffle(&mut self) {
        let range = 0..self.playable_range();
        fastrand::shuffle(&mut self.playable_cards[range]);
    }

    /// Shuffles the whole deck so you can play with different cards
    pub fn shuffle_all(&mut self) {
        fastrand::shuffle(&mut self.playable_cards);
    }

    /// Removes any cards which aren't due from the deck
    pub fn retain_undue(&mut self) {
        // Take the memory because otherwise we will have borrowed
        // self twice.
        let mut playable_cards = mem::take(&mut self.playable_cards);

        // Remove all the cards which we don't need
        playable_cards.retain(|x| !self.current_card_is_finished(*x));

        // Add it back into self
        self.playable_cards = playable_cards;

        // Make sure our cursor is in a valid place
        if !self.playable_cards.contains(&self.card) {
            self.card = *self.playable_cards.get(0).unwrap_or(&0);
        }
    }

    /// Moves on to the next card
    pub fn next_card(&mut self) {
        self.retain_undue();
        if !self.playable_cards.is_empty() {
            self.playable_card_pos += 1;
            self.playable_card_pos %= self.playable_range();
            self.card = self.playable_cards[self.playable_card_pos];
            self.flipped = false;
            if self.playable_card_pos == 0 {
                self.shuffle();
            }
        }
    }

    /// Gets the current card from the app.
    pub fn card(&self) -> &Card {
        &self.cards[self.card]
    }

    /// Gets the score of the current card from the app.
    pub fn card_score(&self) -> usize {
        *self.scores.get(&self.card).unwrap_or(&0)
    }

    /// Resets the duration of the current card.
    ///
    /// This will index [`DURATIONS`] and work out from that
    /// how far forth it should be due.
    pub fn reset_current_card_duration(&mut self) {
        let score = self.card_score().min(DURATIONS.len());
        let duration = DURATIONS[score];
        let duration = chrono::Duration::from_std(duration);
        let duration =
            duration.unwrap_or_else(|_| chrono::Duration::max_value());
        let val = Utc::now() + duration;
        self.dues.insert(self.card, val);
    }

    /// Changes the score of the current card.
    ///
    /// A card can not have a negative score. If the
    /// `zeroize` option is true then a negative `val`
    /// will set the score to zero.
    pub fn change_current_card_score(&mut self, val: isize) {
        let entry = self.scores.entry(self.card).or_default();
        if val > 0 {
            *entry = entry.saturating_add(val.try_into().unwrap_or(0));
            if let Some(x) =
                self.circulating.iter().position(|x| *x == self.card)
            {
                self.circulating.swap_remove(x);
            }
        } else {
            if self.opts.zeroize {
                *entry = 0;
            } else {
                *entry =
                    entry.saturating_sub((0 - val).try_into().unwrap_or(0));
                if !self.circulating.contains(&self.card) {
                    self.circulating.push(self.card)
                }
            }
        }
        self.reset_current_card_duration();
    }

    /// Determines if the current card still should be played
    /// or if its due date has expired.
    pub fn current_card_is_finished(&self, card: usize) -> bool {
        let now = Utc::now();

        if self.circulating.contains(&card) {
            false
        } else {
            if let Some(val) = self.dues.get(&card) {
                now < *val
            } else {
                false
            }
        }
    }

    /// Gets how many cards still need to be played.
    pub fn unfinished_count(&self) -> usize {
        let mut count = 0;

        for card in 0..self.cards.len() {
            if !self.current_card_is_finished(card) {
                count += 1;
            }
        }

        count
    }

    /// Writes the scores to the file specified in [`App::opts`].
    ///
    /// Will not write if `[Opts.nowrite`] prevents it.
    #[cfg(feature = "serde_json")]
    pub fn write_scores(&self) -> io::Result<()> {
        if !self.opts.nowrite {
            let new_path = self.opts.input.with_extension("score.json");
            let file = File::create(new_path)?;

            let scores: Scores<'_> = self.into();
            serde_json::to_writer(&file, &scores)?;
        }

        Ok(())
    }

    /// Adds a [`flashed::Scores`] into an [`App`]
    ///
    /// Will not write if `[Opts.reset`] prevents it.
    pub fn add_scores(&mut self, scores: Scores) {
        if !self.opts.reset {
            for (card, score) in scores.scores {
                let pos = self.cards().iter().position(|x| x == card.as_ref());
                if let Some(pos) = pos {
                    self.scores.insert(pos, score);
                }
            }

            for (card, due) in scores.dues {
                let pos = self.cards().iter().position(|x| x == card.as_ref());
                if let Some(pos) = pos {
                    self.dues.insert(pos, due);
                }
            }

            for card in scores.circulating {
                let pos = self.cards().iter().position(|x| x == card.as_ref());
                if let Some(pos) = pos {
                    self.circulating.push(pos);
                }
            }

            self.retain_undue();
        }
    }

    /// Reads the scores from the file in [`App::opts`]
    #[cfg(feature = "serde_json")]
    pub fn read_scores(&mut self) -> io::Result<()> {
        let new_path = self.opts.input.with_extension("score.json");
        match File::open(new_path) {
            Ok(file) => {
                let scores: Scores<'_> = serde_json::from_reader(&file)?;
                self.add_scores(scores);
                Ok(())
            }
            Err(e) if e.kind() == io::ErrorKind::NotFound => Ok(()),
            Err(e) => Err(e),
        }
    }
}
