use std::collections::HashMap;
use std::fs::File;
use std::io;
use std::time::Duration;
use std::mem;

use chrono::{DateTime, Utc};

use crate::card::Card;
use crate::scores::Scores;
use crate::Opts;

pub static DURATIONS: [Duration; 5] = [
    Duration::from_secs(0),                     // Instant
    Duration::from_secs(60 * 60 * 24),          // Daily
    Duration::from_secs(60 * 60 * 24 * 7),      // Weekly
    Duration::from_secs(60 * 60 * 24 * 28),     // Monthly
    Duration::from_secs(60 * 60 * 24 * 28 * 2), // Bimonthly
];

pub type CardIdx = usize;

/// App holds the state of the application
#[derive(Debug, Clone)]
pub struct App {
    card: CardIdx,
    playable_card_pos: CardIdx,
    cards: Vec<Card>,
    playable_cards: Vec<CardIdx>,
    scores: HashMap<CardIdx, usize>,
    dues: HashMap<CardIdx, DateTime<Utc>>,
    pub flipped: bool,
    pub opts: Opts,
}

impl App {
    /// Creates a new app.
    pub fn new(cards: Vec<Card>, opts: Opts) -> Self {
        let mut app = Self {
            card: 0,
            playable_card_pos: 0,
            playable_cards: (0..cards.len()).collect(),
            cards,
            scores: HashMap::new(),
            flipped: false,
            dues: HashMap::new(),
            opts,
        };

        app.shuffle();
        app.next_card();

        app
    }

    /// Returns an immutable copy of all the cards
    pub fn cards(&self) -> &[Card] {
        &self.cards
    }

    /// Returns in immutable copy of when different cards
    /// are due.
    ///
    /// The key is garunteed to probably be an index from `[Self::cards]`
    pub fn dues(&self) -> &HashMap<CardIdx, DateTime<Utc>> {
        &self.dues
    }

    /// Returns in immutable copy of each cards scores.
    ///
    /// The key is garunteed to probably be an index from `[Self::cards]`
    pub fn scores(&self) -> &HashMap<usize, usize> {
        &self.scores
    }

    /// Flips the current card to show the other side.
    pub fn flip(&mut self) {
        self.flipped ^= true;
    }

    /// Shuffles the deck so you can play with different cards
    pub fn shuffle(&mut self) {
        fastrand::shuffle(&mut self.playable_cards);
    }

    /// Removes any cards which aren't due from the deck
    pub fn retain_undue(&mut self) {
        // Take the memory because otherwise we will have borrowed
        // self twice.
        let mut playable_cards = mem::take(&mut self.playable_cards);

        // Remove all the cards which we don't need
        playable_cards.retain(|x| !self.card_is_finished(*x));

        // Add it back into self
        self.playable_cards = playable_cards;

        // Make sure our cursor is in a valid place
        if !self.playable_cards.contains(&self.card) {
            self.card = *self.playable_cards.get(0).unwrap_or(&0);
        }
    }

    /// Moves on to the next card
    pub fn next_card(&mut self) {
        self.retain_undue();
        if !self.playable_cards.is_empty() {
            dbg!(self.playable_cards.len());
            self.playable_card_pos += 1;
            self.playable_card_pos %= self.playable_cards.len();
            self.card = self.playable_cards[self.playable_card_pos];
            self.flipped = false;
        }
    }

    pub fn card(&self) -> &Card {
        &self.cards[self.card]
    }

    pub fn card_score(&self) -> usize {
        *self.scores.get(&self.card).unwrap_or(&0)
    }

    pub fn reset_card_duration(&mut self) {
        let score = self.card_score().min(DURATIONS.len());
        let duration = DURATIONS[score];
        let duration = chrono::Duration::from_std(duration);
        let duration =
            duration.unwrap_or_else(|_| chrono::Duration::max_value());
        let val = Utc::now() + duration;
        self.dues.insert(self.card, val);
    }

    pub fn change_card_score(&mut self, val: isize) {
        let entry = self.scores.entry(self.card).or_default();
        if val > 0 {
            *entry = entry.saturating_add(val.try_into().unwrap_or(0));
        } else {
            *entry = entry.saturating_sub(val.try_into().unwrap_or(0));
        }
        self.reset_card_duration();
    }

    pub fn card_is_finished(&self, card: usize) -> bool {
        let now = Utc::now();

        if let Some(val) = self.dues.get(&card) {
            now < *val
        } else {
            false
        }
    }

    pub fn unfinished_count(&self) -> usize {
        let mut count = 0;

        for card in 0..self.cards.len() {
            if !self.card_is_finished(card) {
                count += 1;
            }
        }

        count
    }

    pub fn write_scores(&self) -> io::Result<()> {
        let new_path = self.opts.input.with_extension("score.json");
        let file = File::create(new_path)?;

        let scores: Scores<'_> = self.into();
        serde_json::to_writer(&file, &scores)?;

        Ok(())
    }

    pub fn read_scores(&mut self) -> io::Result<()> {
        let new_path = self.opts.input.with_extension("score.json");
        match File::open(new_path) {
            Ok(file) => {
                let scores: Scores<'_> = serde_json::from_reader(&file)?;

                for (card, score) in scores.scores {
                    let pos =
                        self.cards().iter().position(|x| x == card.as_ref());
                    if let Some(pos) = pos {
                        self.scores.insert(pos, score);
                    }
                }

                for (card, due) in scores.dues {
                    let pos =
                        self.cards().iter().position(|x| x == card.as_ref());
                    if let Some(pos) = pos {
                        self.dues.insert(pos, due);
                    }
                }

                Ok(())
            }
            Err(e) if e.kind() == io::ErrorKind::NotFound => Ok(()),
            Err(e) => Err(e),
        }
    }
}
