use ctx::GraphicsContext;
use vulkano::{
    command_buffer::{AutoCommandBufferBuilder, CommandBufferUsage, SubpassContents},
    swapchain::{self, AcquireError, SwapchainCreationError},
    sync::{self, FlushError, GpuFuture},
};

pub mod ctx;

#[derive(Default, Debug, Clone)]
pub struct Vertex {
    position: [f32; 2],
}
vulkano::impl_vertex!(Vertex, position);

pub fn render(ctx: &mut GraphicsContext) {
    if let Some(fut) = ctx.future.as_mut() {
        fut.cleanup_finished();
    }

    // Whenever the window resizes we need to recreate everything dependent on the window size.
    // In this example that includes the swapchain, the framebuffers and the dynamic state viewport.
    if ctx.recreate_swapchain {
        // Get the new dimensions of the window.
        let dimensions: [u32; 2] = ctx.window().inner_size().into();
        let (new_swapchain, new_images) =
            match ctx.swapchain.recreate().dimensions(dimensions).build() {
                Ok(r) => r,
                // This error tends to happen when the user is manually resizing the window.
                // Simply restarting the loop is the easiest way to fix this issue.
                Err(SwapchainCreationError::UnsupportedDimensions) => return,
                Err(e) => panic!("Failed to recreate swapchain: {:?}", e),
            };

        ctx.swapchain = new_swapchain;
        // Because framebuffers contains an Arc on the old swapchain, we need to
        // recreate framebuffers as well.
        ctx.framebuffers =
            ctx::create_framebuffers(&new_images, ctx.render_pass.clone(), &mut ctx.dynamic_state);
        ctx.recreate_swapchain = false;
    }

    // Before we can draw on the output, we have to *acquire* an image from the swapchain. If
    // no image is available (which happens if you submit draw commands too quickly), then the
    // function will block.
    // This operation returns the index of the image that we are allowed to draw upon.
    //
    // This function can block if no image is available. The parameter is an optional timeout
    // after which the function call will return an error.
    let (image_num, suboptimal, acquire_future) =
        match swapchain::acquire_next_image(ctx.swapchain.clone(), None) {
            Ok(r) => r,
            Err(AcquireError::OutOfDate) => {
                ctx.recreate_swapchain = true;
                return;
            }
            Err(e) => panic!("Failed to acquire next image: {:?}", e),
        };

    // acquire_next_image can be successful, but suboptimal. This means that the swapchain image
    // will still work, but it may not display correctly. With some drivers this can be when
    // the window resizes, but it may not cause the swapchain to become out of date.
    if suboptimal {
        ctx.recreate_swapchain = true;
    }

    // Specify the color to clear the framebuffer with i.e. blue
    let clear_values = vec![[0.0, 0.0, 1.0, 1.0].into()];

    // In order to draw, we have to build a *command buffer*. The command buffer object holds
    // the list of commands that are going to be executed.
    //
    // Building a command buffer is an expensive operation (usually a few hundred
    // microseconds), but it is known to be a hot path in the driver and is expected to be
    // optimized.
    //
    // Note that we have to pass a queue family when we create the command buffer. The command
    // buffer will only be executable on that given queue family.
    let mut builder = AutoCommandBufferBuilder::primary(
        ctx.device.clone(),
        ctx.queue.family(),
        CommandBufferUsage::OneTimeSubmit,
    )
    .unwrap();

    builder
        .begin_render_pass(
            ctx.framebuffers[image_num].clone(),
            SubpassContents::Inline,
            clear_values,
        )
        .unwrap()
        .draw(
            ctx.pipeline.clone(),
            &ctx.dynamic_state,
            ctx.vertex_buffer.clone(),
            (),
            (),
            std::iter::empty(),
        )
        .unwrap()
        .end_render_pass()
        .unwrap();

    // Finish building the command buffer by calling `build`.
    let command_buffer = builder.build().unwrap();

    let future = ctx
        .future
        .take()
        .unwrap()
        .join(acquire_future)
        .then_execute(ctx.queue.clone(), command_buffer)
        .unwrap()
        // The color output is now expected to contain our triangle. But in order to show it on
        // the screen, we have to *present* the image by calling `present`.
        //
        // This function does not actually present the image immediately. Instead it submits a
        // present command at the end of the queue. This means that it will only be presented once
        // the GPU has finished executing the command buffer that draws the triangle.
        .then_swapchain_present(ctx.queue.clone(), ctx.swapchain.clone(), image_num)
        .then_signal_fence_and_flush();

    match future {
        Ok(future) => {
            ctx.future = Some(future.boxed());
        }
        Err(FlushError::OutOfDate) => {
            ctx.recreate_swapchain = true;
            ctx.future = Some(sync::now(ctx.device.clone()).boxed());
        }
        Err(e) => {
            println!("Failed to flush future: {:?}", e);
            ctx.future = Some(sync::now(ctx.device.clone()).boxed());
        }
    }
}
