use std::time::{Duration, Instant};

use flak_gfx::ctx::GraphicsContext;
use winit::{
    event::{Event, WindowEvent},
    event_loop::{ControlFlow, EventLoop},
    window::WindowBuilder,
};

/// A container for frame information like timesteps and graphics context
struct Frame {
    /// Time since last frame
    delta: Duration,
    /// Timestamp of when this frame began
    timestamp: Instant,
    /// How long it took for the frame to execute
    exec_time: Duration,
    /// Grapics context
    gfx_ctx: GraphicsContext,
}

impl Frame {
    pub fn new(gfx_ctx: GraphicsContext) -> Self {
        Self {
            delta: Duration::default(),
            timestamp: Instant::now(),
            exec_time: Duration::default(),
            gfx_ctx,
        }
    }

    pub fn start(&mut self) {
        let now = Instant::now();
        self.delta = now - self.timestamp;
        self.timestamp = now;
    }

    pub fn end(&mut self) {
        self.exec_time = Instant::now() - self.timestamp;
    }

    pub fn delta_time(&self) -> Duration {
        self.delta
    }

    pub fn execution_time(&self) -> Duration {
        self.exec_time
    }
}

pub fn run() {
    let event_loop = EventLoop::new();
    let window_builder = WindowBuilder::new();
    let gfx_ctx = GraphicsContext::new(&event_loop, window_builder);

    let mut frame_count = 0;
    let mut frame_total_time = Duration::default();
    let mut fps = 0;
    let mut frame = Frame::new(gfx_ctx);
    event_loop.run(move |event, _, control_flow| match event {
        Event::WindowEvent {
            event: WindowEvent::CloseRequested,
            ..
        } => *control_flow = ControlFlow::Exit,
        Event::WindowEvent {
            event: WindowEvent::Resized(_),
            ..
        } => frame.gfx_ctx.recreate_swapchain = true,
        Event::MainEventsCleared => {
            frame.start();

            flak_gfx::render(&mut frame.gfx_ctx);

            frame.end();
            if frame_total_time > Duration::from_millis(1000) {
                fps = frame_count;
                frame_total_time = Duration::default();
                frame_count = 0;
            } else {
                frame_total_time += frame.delta_time();
                frame_count += 1;
            }
            print!("\rFPS: {}", fps);
            std::io::Write::flush(&mut std::io::stdout()).unwrap();
        }
        _ => (),
    });
}
