use anyhow::{Context, Result};
use std::fs::{self};
use std::io;
use std::path::PathBuf;
use structopt::StructOpt;

pub fn run(args: Cli) -> Result<()> {
    let stdout = io::stdout();
    let writer = io::BufWriter::new(stdout);

    let content = read_document(&args)?;

    get_occurences_in_file(&content, &args.pattern, writer)?;

    Ok(())
}

pub fn get_occurences_in_file(
    content: &str,
    pattern: &str,
    mut writer: impl io::Write,
) -> Result<()> {
    for line in content.lines() {
        if line.contains(pattern) {
            writeln!(writer, "{}", line)?;
        }
    }
    Ok(())
}

pub fn read_document(args: &Cli) -> Result<String> {
    let content = fs::read_to_string(&args.path)
        .with_context(|| format!("Could not read file '{}'", &args.path.display()))?;

    Ok(content)
}

#[derive(StructOpt)]
pub struct Cli {
    pattern: String,
    #[structopt(parse(from_os_str))]
    path: PathBuf,
}
impl Cli {
    pub fn pattern(&self) -> &str {
        &self.pattern
    }
}

#[cfg(test)]
mod test {
    use crate::get_occurences_in_file;

    #[test]
    fn get_occurences_in_file_match() {
        let mut result = Vec::new();
        let expected = b"toto\n".to_vec();
        get_occurences_in_file("titi\ntoto\ntata", "toto", &mut result).unwrap();
        assert_eq!(expected, result);
    }
    #[test]
    fn get_occurences_in_file_nomatch() {
        let mut result = Vec::new();
        let expected = b"".to_vec();
        get_occurences_in_file("titi\ntonton\ntutu", "toto", &mut result).unwrap();
        assert_eq!(expected, result);
    }
}
