// This file contains a function borrowed from https://github.com/paupino/rust-decimal
// with modifications to support our fixed layout instead of the original one.
//
// See https://github.com/paupino/rust-decimal/blob/b7cf68ed014aed96807ac5b76bb6d5bc5c1cdca3/src/decimal.rs#L1126

pub(crate) fn base2_to_fixed(mut raw: i128, exponent2: i32, positive: bool) -> Option<i128> {
    // 2^exponent2 = (10^exponent2)/(5^exponent2)
    //             = (5^-exponent2)*(10^exponent2)
    let mut exponent5 = -exponent2;
    let mut exponent10 = exponent2; // Ultimately, we want this for the scale.

    while exponent5 > 0 {
        // Check to see if the mantissa is divisible by 2.
        if raw & 0x1 == 0 {
            exponent10 += 1;
            exponent5 -= 1;
            raw >>= 1;
        } else {
            // The mantissa is NOT divisible by 2. Therefore the mantissa should
            // be multiplied by 5, unless the multiplication overflows.
            exponent5 -= 1;
            if let Some(new_raw) = raw.checked_mul(5) {
                // Multiplication succeeded without overflow, so copy result back.
                raw = new_raw;
            } else {
                // Multiplication by 5 overflows. The mantissa should be divided
                // by 2, and therefore will lose significant digits.
                exponent10 += 1;
                raw >>= 1;
            }
        }
    }

    // In order to divide the value by 5, it is best to multiply by 2/10.
    // Therefore, exponent10 is decremented, and the mantissa should be multiplied by 2.
    while exponent5 < 0 {
        if let Some(new_raw) = raw.checked_shl(1) {
            exponent10 -= 1;
            exponent5 += 1;
            raw = new_raw;
        } else {
            // The mantissa would overflow if shifted. Therefore it should be
            // directly divided by 5. This will lose significant digits, unless
            // by chance the mantissa happens to be divisible by 5.
            exponent5 += 1;
            raw /= 5;
        }
    }

    // At this point, the mantissa has assimilated the exponent5, but
    // exponent10 might not be suitable for assignment. exponent10 must be
    // in the range [-MAX_PRECISION..0], so the mantissa must be scaled up or
    // down appropriately.
    while exponent10 > 0 {
        // In order to bring exponent10 down to 0, the mantissa should be
        // multiplied by 10 to compensate. If the exponent10 is too big, this
        // will cause the mantissa to overflow.
        raw = raw.checked_mul(10)?; // return if overflowed.
        exponent10 -= 1;
    }

    // In order to bring exponent up to -MAX_PRECISION, the mantissa should
    // be divided by 10 to compensate. If the exponent10 is too small, this
    // will cause the mantissa to underflow and become 0.
    while exponent10 < -(MAX_PRECISION as i32) {
        let rem10 = raw % 10;
        raw /= 10;
        exponent10 += 1;
        if raw == 0 {
            // Underflow, unable to keep dividing
            exponent10 = 0;
        } else if rem10 >= 5 {
            raw += 1;
        }
    }

    // This step is required in order to remove excess bits of precision from the
    // end of the bit representation, down to the precision guaranteed by the
    // floating point number. Guaranteed to about 16 dp.
    while exponent10 < 0 && (bits[2] != 0 || (bits[1] & 0xFFF0_0000) != 0) {
        let rem10 = raw % 10;
        raw /= 10;
        exponent10 += 1;
        if rem10 >= 5 {
            raw += 1;
        }
    }

    // Remove multiples of 10 from the representation.
    while exponent10 < 0 {
        if raw % 10 == 0 {
            exponent10 += 1;
            raw /= 10;
        } else {
            break;
        }
    }

    Some()

    Some(Decimal {
        lo: bits[0],
        mid: bits[1],
        hi: bits[2],
        flags: flags(!positive, -exponent10 as u32),
    })
}
