//! A finger client library
//!
//! See [RFC 1288](https://datatracker.ietf.org/doc/html/rfc1288), which this library loosely
//! implements. The main difference is that it parses requests and responses as UTF-8 instead of
//! (extended) ASCII.
//!
//! Be mature.
//!
//! Inspired by Elif Batuman's "The Idiot"

use std::io::prelude::*;
use std::net::{TcpStream, ToSocketAddrs};
use std::str;
use std::time::{Duration, Instant};
#[derive(Debug)]
pub enum Error {
    InvalidFingerURL,
    HostError,
    Timeout,
    IoError(std::io::Error),
    Utf8Error(str::Utf8Error),
}

impl From<str::Utf8Error> for Error {
    fn from(err: str::Utf8Error) -> Error {
        Error::Utf8Error(err)
    }
}

impl From<std::io::Error> for Error {
    fn from(err: std::io::Error) -> Error {
        Error::IoError(err)
    }
}

/// A request object, with configuration parameters
pub struct Request {
    query: String,
    host: String,
    timeout: Option<u64>, // seconds
    max_response_len: Option<u64>,
}

/// Build a finger request
///
/// rfc1288 2.2 is ignored, queries are UTF-8 strings.
/// Host can include port. Defaults to 79 if not included
pub fn finger(query: &str, host: &str) -> Request {
    let mut q = String::new();
    q.push_str(query);

    // Fix query if CRLF was omitted
    if !q.ends_with("\r\n") {
        q.push_str("\r\n")
    }

    Request {
        query: q.to_owned(),
        host: host.to_owned(),
        timeout: None,
        max_response_len: None,
    }
}

impl Request {
    /// Set timeout. Default is no timeout.
    pub fn timeout(mut self, timeout: u64) -> Request {
        self.timeout = Some(timeout);
        self
    }

    /// Set max bytes to read before closing connection. Default is no limit.
    pub fn max_response_len(mut self, max: u64) -> Request {
        self.max_response_len = Some(max);
        self
    }

    /// Send the finger request
    ///
    /// rfc1288 2.2 is ignored, we will allow UTF-8. Returns a UTF-8 string
    pub fn send(self) -> Result<String, Error> {
        let dest: String;
        if !self.host.contains(":") {
            dest = format!("{}:79", self.host);
        } else {
            dest = self.host;
        }
        let server = dest.to_socket_addrs()?.next().ok_or(Error::HostError)?;
        let timeout = Duration::from_secs(self.timeout.unwrap());
        let start = Instant::now();
        let mut stream = TcpStream::connect_timeout(&server, timeout)?;
        // TODO improve timeout code
        if start + timeout < Instant::now() {
            return Err(Error::Timeout); // TODO make timeout errors consistent
        }
        stream.set_write_timeout(Some((start + timeout) - Instant::now()))?;
        stream.write(self.query.as_bytes())?;
        if start + timeout < Instant::now() {
            return Err(Error::Timeout);
        }
        stream.set_read_timeout(Some((start + timeout) - Instant::now()))?;

        let mut line: Vec<u8> = vec![];
        match self.max_response_len {
            Some(mrl) => stream.take(mrl).read_to_end(&mut line)?,
            None => stream.read_to_end(&mut line)?,
        };
        // Finger responses are small, and always text. Parse as string.
        return Ok(str::from_utf8(&line)?.to_owned());
    }
}
